/*
  GRANITE DATA SERVICES
  Copyright (C) 2007-2008 ADEQUATE SYSTEMS SARL

  This file is part of Granite Data Services.

  Granite Data Services is free software; you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as published by
  the Free Software Foundation; either version 3 of the License, or (at your
  option) any later version.

  Granite Data Services is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
  for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with this library; if not, see <http://www.gnu.org/licenses/>.
*/

package org.granite.tide;

import java.lang.reflect.Method;
import java.util.Set;

import javax.servlet.http.HttpSession;

import org.granite.config.flex.Destination;
import org.granite.context.GraniteContext;
import org.granite.logging.Logger;
import org.granite.messaging.amf.io.convert.Converters;
import org.granite.messaging.service.ServiceException;
import org.granite.messaging.service.ServiceFactory;
import org.granite.messaging.service.ServiceInvocationContext;
import org.granite.messaging.service.ServiceInvoker;
import org.granite.messaging.service.security.SecurityServiceException;
import org.granite.messaging.webapp.HttpGraniteContext;
import org.granite.tide.data.DataContext;
import org.granite.tide.data.DataEnabled;
import org.granite.tide.data.DataMergeContext;
import org.granite.tide.data.DataEnabled.PublishMode;
import org.granite.tide.validators.EntityValidator;
import org.granite.tide.validators.InvalidValue;
import org.granite.util.ClassUtil;

import flex.messaging.messages.RemotingMessage;

 
/**
 * Base class for Tide service invokers
 * Adapts the Tide invocation model with Granite
 * 
 * @author William DRAI
 */
public class TideServiceInvoker<T extends ServiceFactory> extends ServiceInvoker<T> {

    private static final long serialVersionUID = 1L;
     
    private static final Logger log = Logger.getLogger(TideServiceInvoker.class);
    
    public static final String VALIDATOR_CLASS_NAME = "validatorClassName";
    
    /**
     * Current tide context
     */
    private TideServiceContext tideContext = null;
    
    private transient EntityValidator validator = null;
    private boolean validatorNotAvailable = false;
    
    
    public TideServiceInvoker(Destination destination, T factory) throws ServiceException {
        super(destination, factory);
        this.invokee = this;
        this.tideContext = lookupContext();
        this.tideContext.initCall();
        initValidator();
    }
    
    public TideServiceInvoker(Destination destination, T factory, TideServiceContext tideContext) throws ServiceException {
        super(destination, factory);
        this.invokee = this;
        this.tideContext = tideContext;
        this.tideContext.initCall();
        initValidator();
    }
    
    
    public Object initializeObject(Object parent, String[] propertyNames) {
        return tideContext.lazyInitialize(parent, propertyNames);
    }
    
    
    private static final InvalidValue[] EMPTY_INVALID_VALUES = new InvalidValue[0];
    
    private void initValidator() {
    	if (validator != null || validatorNotAvailable)
    		return;
    	
        String className = this.destination.getProperties().get(VALIDATOR_CLASS_NAME);
        if (className == null)
            className = "org.granite.tide.hibernate.HibernateValidator";
        
        try {
            Class<?> validatorClass = Thread.currentThread().getContextClassLoader().loadClass(className);
            validator = (EntityValidator)validatorClass.newInstance();
        }
        catch (ClassNotFoundException e) {
            // Ignore: Hibernate Validator not present
        }
        catch (IllegalAccessException e) {
            // Ignore: Hibernate Validator not present
        }
        catch (InstantiationException e) {
            // Ignore: Hibernate Validator not present
        }
        
        if (validator == null) {
            log.info("Validator class " + className + " not found: validation not enabled");
            validatorNotAvailable = true;
        }
    }
    
    public InvalidValue[] validateObject(Object entity, String propertyName, Object value) {
    	initValidator();
        if (entity != null && validator != null)
            return validator.getPotentialInvalidValues(entity.getClass(), propertyName, value);
        
        return EMPTY_INVALID_VALUES;
    }

    
    public void login() {
    }
    
    public void logout() {
        HttpGraniteContext context = (HttpGraniteContext)GraniteContext.getCurrentInstance();
        HttpSession session = context.getSession(false);
        if (session != null)
            session.invalidate();
    }
    
    public void resyncContext() {
    }
    
    
    protected TideServiceContext lookupContext() {
        return null;
    }
    
    protected TideServiceContext getTideContext() {
        return tideContext;
    }
    
    
    
    @Override
    protected Object adjustInvokee(RemotingMessage request, String methodName, Object[] args) throws ServiceException {
        if ("invokeComponent".equals(methodName)) {
            String componentName = (String)args[0];
            String componentClassName = (String)args[1];
            Class<?> componentClass = null;
            try {
	            if (componentClassName != null)
	            	componentClass = ClassUtil.forName(componentClassName);
            }
            catch (ClassNotFoundException e) {
            	throw new ServiceException("Component class not found " + componentClassName, e);
            }
            log.debug("Setting invokee to {0}", componentName);
            
            Object instance = tideContext.findComponent(componentName, componentClass);
            Set<Class<?>> componentClasses = instance != null ? tideContext.findComponentClasses(componentName, componentClass) : null;
            
            GraniteContext context = GraniteContext.getCurrentInstance();
            if (instance != null && componentClasses != null && context.getGraniteConfig().isComponentTideEnabled(componentName, componentClasses, instance))
                return tideContext.adjustInvokee(instance, componentName, componentClasses);
            
            if (instance != null)
                log.error("SECURITY CHECK: Remote call refused to a non Tide-enabled component: " + componentName + "." + args[1] + ", class: " + componentClasses + ", instance: " + instance);
            throw SecurityServiceException.newAccessDeniedException("Component [" + componentName + (componentClassName != null ? " of class " + componentClassName : "") + "] not found");
        }
        
        return super.adjustInvokee(request, methodName, args);
    }
    

    @Override
    protected Object[] beforeMethodSearch(Object invokee, String methodName, Object[] args) {
        if ("invokeComponent".equals(methodName)) { 
        	return tideContext.beforeMethodSearch(invokee, methodName, args);
        } 
        else if ("initializeObject".equals(methodName)) {
        	return new Object[] { methodName, new Object[] { args[0], args[1] } };
        } 
        else if ("validateObject".equals(methodName)) {
            return new Object[] { methodName, new Object[] { args[0], args[1], args[2] } };
        }
        	
        return new Object[] { methodName, new Object[] {} };
    }


    @Override
    protected void beforeInvocation(ServiceInvocationContext context) {
        RemotingMessage message = (RemotingMessage)context.getMessage();
        GraniteContext graniteContext = GraniteContext.getCurrentInstance();
        
        Object[] originArgs = (Object[])message.getBody();
        IInvocationCall call = (IInvocationCall)originArgs[originArgs.length-1];
        
        String operation = message.getOperation();
        String componentName = "invokeComponent".equals(operation) ? (String)originArgs[0] : null;
        String componentClassName = "invokeComponent".equals(operation) ? (String)originArgs[1] : null;
        Class<?> componentClass = null;
        try {
	        if (componentClassName != null)
	        	componentClass = ClassUtil.forName(componentClassName);
        }
        catch (ClassNotFoundException e) {
        	throw new ServiceException("Component class not found " + componentClassName, e);
        }
        
        graniteContext.getRequestMap().put(TideServiceInvoker.class.getName(), this);
        
        if (componentName != null) {
        	Converters converters = graniteContext.getGraniteConfig().getConverters();
        	
        	Set<Class<?>> componentClasses = tideContext.findComponentClasses(componentName, componentClass);
        	for (Class<?> cClass : componentClasses) {
        		try {
        			Method m = cClass.getMethod(context.getMethod().getName(), context.getMethod().getParameterTypes());
        			for (int i = 0; i < m.getGenericParameterTypes().length; i++)
        				context.getParameters()[i] = converters.convert(context.getParameters()[i], m.getGenericParameterTypes()[i]);
        			
        			break;
        		}
        		catch (NoSuchMethodException e) {
        		}
        	}
        	
        	for (Class<?> cClass : componentClasses) {
	        	DataEnabled dataEnabled = cClass.getAnnotation(DataEnabled.class);
	        	if (dataEnabled != null) {
	        		DataContext.init(dataEnabled.topic(), dataEnabled.params(), dataEnabled.publish());
	        		prepareDataObserver(dataEnabled);
	        		break;
	        	}
        	}
        }
        
        tideContext.prepareCall(context, call, componentName, componentClass);
    }
        
    protected void prepareDataObserver(DataEnabled dataEnabled) {
		DataContext.observe();
    }


    @Override
    protected Object afterInvocation(ServiceInvocationContext context, Object result) {
    	Object res = null;
    	
    	String componentName = null;
        Class<?> componentClass = null;
    	try {
	        Object[] originArgs = (Object[])context.getMessage().getBody();
	        String operation = ((RemotingMessage)context.getMessage()).getOperation();
	        componentName = "invokeComponent".equals(operation) ? (String)originArgs[0] : null;
	        String componentClassName = "invokeComponent".equals(operation) ? (String)originArgs[1] : null;
	        try {
		        if (componentClassName != null)
		        	componentClass = ClassUtil.forName(componentClassName);
	        }
	        catch (ClassNotFoundException e) {
	        	throw new ServiceException("Component class not found " + componentClassName, e);
	        }
    	}
    	finally {
    		res = tideContext.postCall(context, result, componentName, componentClass);
    	}
    	
    	DataMergeContext.remove();
    	publishDataUpdates();
		DataContext.remove();
    	
    	return res;
    }
    
    protected void publishDataUpdates() {
		DataContext.publish(PublishMode.ON_SUCCESS);
    }

    
    @Override
    protected void afterInvocationError(ServiceInvocationContext context, Throwable error) {
    	String componentName = null;
        Class<?> componentClass = null;
    	try {
	        Object[] originArgs = (Object[])context.getMessage().getBody();
	        String operation = ((RemotingMessage)context.getMessage()).getOperation();
	        componentName = "invokeComponent".equals(operation) ? (String)originArgs[0] : null;
	        String componentClassName = "invokeComponent".equals(operation) ? (String)originArgs[1] : null;
	        try {
		        if (componentClassName != null)
		        	componentClass = ClassUtil.forName(componentClassName);
	        }
	        catch (ClassNotFoundException e) {
	        	throw new ServiceException("Component class not found " + componentClassName, e);
	        }
    	}
    	finally {
    		tideContext.postCallFault(context, error, componentName, componentClass);
    	}
    	
    	DataMergeContext.remove();
		DataContext.remove();
    }
}
