/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.stack;

import com.oracle.svm.core.Uninterruptible;
import com.oracle.svm.core.code.CodeInfo;
import com.oracle.svm.core.code.CodeInfoAccess;
import com.oracle.svm.core.code.CodeInfoDecoder;
import com.oracle.svm.core.code.CodeInfoTable;
import com.oracle.svm.core.code.FrameInfoQueryResult;
import com.oracle.svm.core.code.UntetheredCodeInfo;
import com.oracle.svm.core.deopt.DeoptimizationSupport;
import com.oracle.svm.core.deopt.DeoptimizedFrame;
import com.oracle.svm.core.heap.RestrictHeapAccess;
import com.oracle.svm.core.log.Log;
import com.oracle.svm.core.stack.JavaFrameAnchor;
import com.oracle.svm.core.stack.JavaFrameAnchors;
import com.oracle.svm.core.stack.JavaStackWalk;
import com.oracle.svm.core.stack.JavaStackWalker;
import com.oracle.svm.core.stack.ParameterizedStackFrameVisitor;
import jdk.graal.compiler.word.Word;
import org.graalvm.nativeimage.IsolateThread;
import org.graalvm.nativeimage.StackValue;
import org.graalvm.nativeimage.c.function.CodePointer;
import org.graalvm.word.Pointer;
import org.graalvm.word.UnsignedWord;
import org.graalvm.word.WordBase;

public class ThreadStackPrinter {
    @Uninterruptible(reason="Prevent deoptimization of stack frames while in this method.")
    public static boolean printStacktrace(IsolateThread thread, Pointer initialSP, CodePointer initialIP, StackFramePrintVisitor printVisitor, Log log) {
        Pointer sp = initialSP;
        CodePointer ip = initialIP;
        UntetheredCodeInfo info = CodeInfoTable.lookupCodeInfo(ip);
        if (info.isNull()) {
            ThreadStackPrinter.logFrame(log, sp, ip);
            JavaFrameAnchor anchor = JavaFrameAnchors.getFrameAnchor(thread);
            if (anchor.isNonNull()) {
                sp = anchor.getLastJavaSP();
                ip = anchor.getLastJavaIP();
            } else {
                return false;
            }
        }
        JavaStackWalk walk = (JavaStackWalk)StackValue.get((int)JavaStackWalker.sizeOfJavaStackWalk());
        JavaStackWalker.initialize(walk, thread, sp, ip);
        return JavaStackWalker.doWalk(walk, thread, printVisitor, log);
    }

    @Uninterruptible(reason="IP is not within Java code, so there is no risk that it gets invalidated.", calleeMustBe=false)
    private static void logFrame(Log log, Pointer sp, CodePointer ip) {
        StackFramePrintVisitor.logSPAndIP(log, sp, ip);
        log.string("  IP is not within Java code. Trying frame anchor of last Java frame instead.").newline();
    }

    public static class StackFramePrintVisitor
    extends ParameterizedStackFrameVisitor {
        private static final int MAX_STACK_FRAMES_PER_THREAD_TO_PRINT = 100000;
        private final CodeInfoDecoder.FrameInfoCursor frameInfoCursor = new CodeInfoDecoder.FrameInfoCursor();
        private int invocationCount;
        private int printedFrames;
        private Pointer expectedSP;

        public StackFramePrintVisitor reset(int invocationCount) {
            assert (invocationCount >= 1 && invocationCount <= 3);
            this.invocationCount = invocationCount;
            this.printedFrames = 0;
            this.expectedSP = (Pointer)Word.nullPointer();
            return this;
        }

        @Override
        @RestrictHeapAccess(access=RestrictHeapAccess.Access.NO_ALLOCATION, reason="Used for crash log")
        protected final boolean visitRegularFrame(Pointer sp, CodePointer ip, CodeInfo codeInfo, Object data) {
            return this.visitFrame(sp, ip, codeInfo, null, (Log)data);
        }

        @Override
        @RestrictHeapAccess(access=RestrictHeapAccess.Access.NO_ALLOCATION, reason="Used for crash log")
        protected boolean visitDeoptimizedFrame(Pointer originalSP, CodePointer deoptStubIP, DeoptimizedFrame deoptFrame, Object data) {
            return this.visitFrame(originalSP, deoptStubIP, (CodeInfo)Word.nullPointer(), deoptFrame, (Log)data);
        }

        @Override
        protected final boolean unknownFrame(Pointer sp, CodePointer ip, Object data) {
            Log log = (Log)data;
            this.logFrameRaw(log, sp, ip, (CodeInfo)Word.nullPointer());
            log.string("  IP is not within Java code. Aborting stack trace printing.").newline();
            return false;
        }

        private boolean visitFrame(Pointer sp, CodePointer ip, CodeInfo codeInfo, DeoptimizedFrame deoptFrame, Log log) {
            if (this.printedFrames >= 100000) {
                log.string("... (truncated)").newline();
                return false;
            }
            if (this.invocationCount >= 3) {
                this.logJavaFrameMinimalInfo(log, sp, ip, codeInfo, deoptFrame, true);
                log.string("CodeInfo ").zhex((WordBase)codeInfo).string(", ");
                log.string("DeoptimizedFrame ").zhex((WordBase)Word.objectToUntrackedPointer((Object)deoptFrame));
            } else {
                if (this.expectedSP.isNonNull() && sp != this.expectedSP) {
                    this.logNativeFrame(log, this.expectedSP, ip, sp.subtract((UnsignedWord)this.expectedSP).rawValue());
                    log.newline();
                }
                if (deoptFrame != null) {
                    this.logDeoptimizedJavaFrame(log, sp, ip, deoptFrame);
                } else {
                    this.logRegularJavaFrame(log, sp, ip, codeInfo);
                }
            }
            log.newline();
            return true;
        }

        private void logDeoptimizedJavaFrame(Log log, Pointer sp, CodePointer ip, DeoptimizedFrame deoptFrame) {
            for (DeoptimizedFrame.VirtualFrame frame = deoptFrame.getTopFrame(); frame != null; frame = frame.getCaller()) {
                if (this.printedFrames >= 100000) {
                    log.string("... (truncated)").newline();
                    break;
                }
                boolean isCompilationRoot = frame.getCaller() == null;
                StackFramePrintVisitor.printFrameIdentifier(log, (CodeInfo)Word.nullPointer(), deoptFrame, isCompilationRoot, false);
                this.logFrameRaw(log, sp, ip, deoptFrame.getSourceTotalFrameSize());
                StackFramePrintVisitor.logFrameInfo(log, frame.getFrameInfo(), "image code, deopt");
                if (isCompilationRoot) continue;
                log.newline();
            }
        }

        private void logRegularJavaFrame(Log log, Pointer sp, CodePointer ip, CodeInfo codeInfo) {
            if (this.invocationCount == 1 || CodeInfoAccess.isAOTImageCodeSlow(codeInfo)) {
                boolean isFirst = true;
                this.frameInfoCursor.initialize(codeInfo, ip, false);
                while (this.frameInfoCursor.advance()) {
                    if (this.printedFrames >= 100000) {
                        log.string("... (truncated)").newline();
                        break;
                    }
                    if (!isFirst) {
                        log.newline();
                    }
                    boolean isCompilationRoot = !this.frameInfoCursor.hasCaller();
                    this.logVirtualFrame(log, sp, ip, codeInfo, this.frameInfoCursor.get(), isCompilationRoot);
                    isFirst = false;
                }
                if (isFirst) {
                    this.logJavaFrameMinimalInfo(log, sp, ip, codeInfo, null, true);
                    log.string("missing metadata");
                }
            } else {
                this.logJavaFrameMinimalInfo(log, sp, ip, codeInfo, null, true);
                log.string("CodeInfo ").zhex((WordBase)codeInfo);
            }
        }

        private void logVirtualFrame(Log log, Pointer sp, CodePointer ip, CodeInfo codeInfo, FrameInfoQueryResult frameInfo, boolean isCompilationRoot) {
            this.logJavaFrameMinimalInfo(log, sp, ip, codeInfo, null, isCompilationRoot);
            String codeInfoName = DeoptimizationSupport.enabled() ? CodeInfoAccess.getName(codeInfo) : null;
            StackFramePrintVisitor.logFrameInfo(log, frameInfo, codeInfoName);
        }

        private void logJavaFrameMinimalInfo(Log log, Pointer sp, CodePointer ip, CodeInfo codeInfo, DeoptimizedFrame deoptFrame, boolean isCompilationRoot) {
            StackFramePrintVisitor.printFrameIdentifier(log, codeInfo, deoptFrame, isCompilationRoot, false);
            this.logFrameRaw(log, sp, ip, codeInfo);
        }

        private void logNativeFrame(Log log, Pointer sp, CodePointer ip, long frameSize) {
            StackFramePrintVisitor.printFrameIdentifier(log, (CodeInfo)Word.nullPointer(), null, true, true);
            this.logFrameRaw(log, sp, ip, frameSize);
            log.string("unknown");
        }

        private void logFrameRaw(Log log, Pointer sp, CodePointer ip, CodeInfo codeInfo) {
            long frameSize = -1L;
            if (codeInfo.isNonNull()) {
                frameSize = CodeInfoAccess.lookupTotalFrameSize(codeInfo, ip);
            }
            this.logFrameRaw(log, sp, ip, frameSize);
        }

        private void logFrameRaw(Log log, Pointer sp, CodePointer ip, long frameSize) {
            StackFramePrintVisitor.logSPAndIP(log, sp, ip);
            log.string(" size=");
            if (frameSize >= 0L) {
                log.signed(frameSize, 4, 1);
                this.expectedSP = sp.add(Word.unsigned((long)frameSize));
            } else {
                log.string("?").spaces(3);
                this.expectedSP = (Pointer)Word.nullPointer();
            }
            log.spaces(2);
            ++this.printedFrames;
        }

        private static void logSPAndIP(Log log, Pointer sp, CodePointer ip) {
            log.string("SP ").zhex((WordBase)sp).spaces(1);
            log.string("IP ").zhex((WordBase)ip);
        }

        private static void logFrameInfo(Log log, FrameInfoQueryResult frameInfo, String runtimeMethodInfoName) {
            if (runtimeMethodInfoName != null) {
                log.string("[").string(runtimeMethodInfoName).string("] ");
            }
            frameInfo.log(log);
        }

        private static void printFrameIdentifier(Log log, CodeInfo codeInfo, DeoptimizedFrame deoptFrame, boolean isCompilationRoot, boolean isNative) {
            char ch = StackFramePrintVisitor.getFrameIdentifier(codeInfo, deoptFrame, isCompilationRoot, isNative);
            log.character(ch).spaces(2);
        }

        private static char getFrameIdentifier(CodeInfo codeInfo, DeoptimizedFrame deoptFrame, boolean isCompilationRoot, boolean isNative) {
            if (isNative) {
                return 'C';
            }
            if (deoptFrame != null) {
                return 'D';
            }
            if (!isCompilationRoot) {
                return 'i';
            }
            if (codeInfo.isNonNull() && CodeInfoAccess.isAOTImageCodeSlow(codeInfo)) {
                return 'A';
            }
            return 'J';
        }
    }
}

