/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.jfr;

import com.oracle.svm.core.SubstrateOptions;
import com.oracle.svm.core.jdk.RuntimeSupport;
import com.oracle.svm.core.jfr.SubstrateJVM;
import com.oracle.svm.core.jfr.events.EndChunkNativePeriodicEvents;
import com.oracle.svm.core.jfr.events.EveryChunkNativePeriodicEvents;
import com.oracle.svm.core.util.UserError;
import com.oracle.svm.core.util.VMError;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.ParseException;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import jdk.jfr.FlightRecorder;
import jdk.jfr.Recording;
import jdk.jfr.internal.LogLevel;
import jdk.jfr.internal.LogTag;
import jdk.jfr.internal.Logger;
import jdk.jfr.internal.OldObjectSample;
import jdk.jfr.internal.PrivateAccess;
import jdk.jfr.internal.SecuritySupport;
import jdk.jfr.internal.Utils;
import jdk.jfr.internal.jfc.JFC;
import org.graalvm.compiler.api.replacements.Fold;
import org.graalvm.compiler.core.common.SuppressFBWarnings;
import org.graalvm.nativeimage.ImageSingletons;
import org.graalvm.nativeimage.Platform;
import org.graalvm.nativeimage.Platforms;

public class JfrManager {
    private static final String DEFAULT_JFC_NAME = "default";
    @Platforms(value={Platform.HOSTED_ONLY.class})
    final boolean hostedEnabled;

    @Platforms(value={Platform.HOSTED_ONLY.class})
    public JfrManager(boolean hostedEnabled) {
        this.hostedEnabled = hostedEnabled;
    }

    public static boolean isJFREnabled() {
        return SubstrateOptions.FlightRecorder.getValue() != false || !SubstrateOptions.StartFlightRecording.getValue().isEmpty();
    }

    @Fold
    public static JfrManager get() {
        return (JfrManager)ImageSingletons.lookup(JfrManager.class);
    }

    public RuntimeSupport.Hook startupHook() {
        return isFirstIsolate -> {
            JfrManager.parseFlightRecorderLogging(SubstrateOptions.FlightRecorderLogging.getValue());
            JfrManager.periodicEventSetup();
            if (JfrManager.isJFREnabled()) {
                JfrManager.initRecording();
            }
        };
    }

    public RuntimeSupport.Hook shutdownHook() {
        return isFirstIsolate -> {
            if (JfrManager.isJFREnabled()) {
                FlightRecorder.removePeriodicEvent(EveryChunkNativePeriodicEvents::emit);
                FlightRecorder.removePeriodicEvent(EndChunkNativePeriodicEvents::emit);
            }
        };
    }

    private static void parseFlightRecorderLogging(String option) {
        SubstrateJVM.getJfrLogging().parseConfiguration(option);
    }

    private static void periodicEventSetup() throws SecurityException {
        FlightRecorder.addPeriodicEvent(EveryChunkNativePeriodicEvents.class, EveryChunkNativePeriodicEvents::emit);
        FlightRecorder.addPeriodicEvent(EndChunkNativePeriodicEvents.class, EndChunkNativePeriodicEvents::emit);
    }

    private static void initRecording() {
        Map<JfrStartArgument, String> args = JfrManager.parseStartFlightRecording();
        String name = args.get((Object)JfrStartArgument.Name);
        String[] settings = JfrManager.parseSettings(args);
        Long delay = JfrManager.parseDuration(args, JfrStartArgument.Delay);
        Long duration = JfrManager.parseDuration(args, JfrStartArgument.Duration);
        Boolean disk = JfrManager.parseBoolean(args, JfrStartArgument.Disk);
        String path = args.get((Object)JfrStartArgument.Filename);
        Long maxAge = JfrManager.parseDuration(args, JfrStartArgument.MaxAge);
        Long maxSize = JfrManager.parseMaxSize(args, JfrStartArgument.MaxSize);
        Boolean dumpOnExit = JfrManager.parseBoolean(args, JfrStartArgument.DumpOnExit);
        Boolean pathToGcRoots = JfrManager.parseBoolean(args, JfrStartArgument.PathToGCRoots);
        try {
            if (Logger.shouldLog(LogTag.JFR_DCMD, LogLevel.DEBUG)) {
                Logger.log(LogTag.JFR_DCMD, LogLevel.DEBUG, "Executing DCmdStart: name=" + name + ", settings=" + Arrays.asList(settings) + ", delay=" + delay + ", duration=" + duration + ", disk=" + disk + ", filename=" + path + ", maxage=" + maxAge + ", maxsize=" + maxSize + ", dumponexit =" + dumpOnExit + ", path-to-gc-roots=" + pathToGcRoots);
            }
            if (name != null) {
                try {
                    Integer.parseInt(name);
                    throw new Exception("Name of recording can't be numeric");
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
            if (duration == null && Boolean.FALSE.equals(dumpOnExit) && path != null) {
                throw new Exception("Filename can only be set for a time bound recording or if dumponexit=true. Set duration/dumponexit or omit filename.");
            }
            if (settings.length == 1 && settings[0].length() == 0) {
                throw new Exception("No settings specified. Use settings=none to start without any settings");
            }
            HashMap<String, String> s = new HashMap<String, String>();
            for (String configName : settings) {
                try {
                    s.putAll(JFC.createKnown(configName).getSettings());
                }
                catch (FileNotFoundException e) {
                    throw new Exception("Could not find settings file'" + configName + "'", e);
                }
                catch (IOException | ParseException e) {
                    throw new Exception("Could not parse settings file '" + settings[0] + "'", e);
                }
            }
            OldObjectSample.updateSettingPathToGcRoots(s, pathToGcRoots);
            if (duration != null && duration < 1000000000L) {
                throw new Exception("Could not start recording, duration must be at least 1 second.");
            }
            if (delay != null && delay < 1000000000L) {
                throw new Exception("Could not start recording, delay must be at least 1 second.");
            }
            Recording recording = new Recording();
            if (name != null) {
                recording.setName(name);
            }
            if (disk != null) {
                recording.setToDisk(disk);
            }
            recording.setSettings(s);
            SecuritySupport.SafePath safePath = null;
            if (path != null) {
                try {
                    Path p;
                    if (dumpOnExit == null) {
                        dumpOnExit = Boolean.TRUE;
                    }
                    if (Files.isDirectory(p = Paths.get(path, new String[0]), new LinkOption[0]) && Boolean.TRUE.equals(dumpOnExit)) {
                        PrivateAccess.getInstance().getPlatformRecording(recording).setDumpOnExitDirectory(new SecuritySupport.SafePath(p));
                    } else {
                        safePath = JfrManager.resolvePath(recording, path);
                        recording.setDestination(safePath.toPath());
                    }
                }
                catch (IOException | InvalidPathException e) {
                    recording.close();
                    throw new Exception("Could not start recording, not able to write to file: " + path, e);
                }
            }
            if (maxAge != null) {
                recording.setMaxAge(Duration.ofNanos(maxAge));
            }
            if (maxSize != null) {
                recording.setMaxSize(maxSize);
            }
            if (duration != null) {
                recording.setDuration(Duration.ofNanos(duration));
            }
            if (dumpOnExit != null) {
                recording.setDumpOnExit(dumpOnExit);
            }
            StringBuilder msg = new StringBuilder();
            if (delay != null) {
                Duration dDelay = Duration.ofNanos(delay);
                recording.scheduleStart(dDelay);
                msg.append("Recording " + recording.getId() + " scheduled to start in ");
                msg.append(Utils.formatTimespan(dDelay, " "));
                msg.append(".");
            } else {
                recording.start();
                msg.append("Started recording " + recording.getId() + ".");
            }
            if (recording.isToDisk() && duration == null && maxAge == null && maxSize == null) {
                msg.append(" No limit specified, using maxsize=250MB as default.");
                recording.setMaxSize(0xFA00000L);
            }
            if (safePath != null && duration != null) {
                msg.append(" The result will be written to:");
                msg.append(System.getProperty("line.separator"));
                msg.append(JfrManager.getPath(safePath));
                msg.append(System.getProperty("line.separator"));
            }
            Logger.log(LogTag.JFR_SYSTEM, LogLevel.INFO, msg.toString());
        }
        catch (Throwable e) {
            VMError.shouldNotReachHere(e);
        }
    }

    private static SecuritySupport.SafePath resolvePath(Recording recording, String filename) throws InvalidPathException {
        if (filename == null) {
            return JfrManager.makeGenerated(recording, Paths.get(".", new String[0]));
        }
        Path path = Paths.get(filename, new String[0]);
        if (Files.isDirectory(path, new LinkOption[0])) {
            return JfrManager.makeGenerated(recording, path);
        }
        return new SecuritySupport.SafePath(path.toAbsolutePath().normalize());
    }

    private static SecuritySupport.SafePath makeGenerated(Recording recording, Path directory) {
        return new SecuritySupport.SafePath(directory.toAbsolutePath().resolve(Utils.makeFilename(recording)).normalize());
    }

    private static String getPath(SecuritySupport.SafePath path) {
        if (path == null) {
            return "N/A";
        }
        try {
            return JfrManager.getPath(SecuritySupport.getAbsolutePath(path).toPath());
        }
        catch (IOException ioe) {
            return JfrManager.getPath(path.toPath());
        }
    }

    private static String getPath(Path path) {
        try {
            return path.toAbsolutePath().toString();
        }
        catch (SecurityException e) {
            return path.toString();
        }
    }

    private static Map<JfrStartArgument, String> parseStartFlightRecording() {
        HashMap<JfrStartArgument, String> optionsMap = new HashMap<JfrStartArgument, String>();
        String text = SubstrateOptions.StartFlightRecording.getValue();
        if (!text.isEmpty()) {
            String[] options;
            JfrStartArgument[] possibleArguments = JfrStartArgument.values();
            for (String option : options = text.split(",")) {
                String[] keyVal = option.split("=");
                JfrStartArgument arg = JfrManager.findArgument(possibleArguments, keyVal[0]);
                if (arg == null) {
                    throw VMError.shouldNotReachHere("Unknown argument '" + keyVal[0] + "' in " + SubstrateOptions.StartFlightRecording.getName());
                }
                optionsMap.put(arg, keyVal[1]);
            }
        }
        return optionsMap;
    }

    private static String[] parseSettings(Map<JfrStartArgument, String> args) throws UserError.UserException {
        String settings = args.get((Object)JfrStartArgument.Settings);
        if (settings == null) {
            return new String[]{DEFAULT_JFC_NAME};
        }
        if (settings.equals("none")) {
            return new String[0];
        }
        return settings.split(",");
    }

    @SuppressFBWarnings(value={"NP_BOOLEAN_RETURN_NULL"}, justification="null allowed as return value")
    private static Boolean parseBoolean(Map<JfrStartArgument, String> args, JfrStartArgument key) throws IllegalArgumentException {
        String value = args.get((Object)key);
        if (value == null) {
            return null;
        }
        if ("true".equalsIgnoreCase(value)) {
            return true;
        }
        if ("false".equalsIgnoreCase(value)) {
            return false;
        }
        throw VMError.shouldNotReachHere("Could not parse JFR argument '" + key.cmdLineKey + "=" + value + "'. Expected a boolean value.");
    }

    private static Long parseDuration(Map<JfrStartArgument, String> args, JfrStartArgument key) {
        String value = args.get((Object)key);
        if (value != null) {
            try {
                long time;
                int idx = JfrManager.indexOfFirstNonDigitCharacter(value);
                try {
                    time = Long.parseLong(value.substring(0, idx));
                }
                catch (NumberFormatException e) {
                    throw new IllegalArgumentException("Expected a number.");
                }
                if (idx == value.length()) {
                    if (time != 0L) {
                        throw new IllegalArgumentException("Unit is required.");
                    }
                    return 0L;
                }
                String unit = value.substring(idx);
                if ("ns".equals(unit)) {
                    return Duration.ofNanos(time).toNanos();
                }
                if ("us".equals(unit)) {
                    return Duration.ofNanos(time * 1000L).toNanos();
                }
                if ("ms".equals(unit)) {
                    return Duration.ofMillis(time).toNanos();
                }
                if ("s".equals(unit)) {
                    return Duration.ofSeconds(time).toNanos();
                }
                if ("m".equals(unit)) {
                    return Duration.ofMinutes(time).toNanos();
                }
                if ("h".equals(unit)) {
                    return Duration.ofHours(time).toNanos();
                }
                if ("d".equals(unit)) {
                    return Duration.ofDays(time).toNanos();
                }
                throw new IllegalArgumentException("Unit is invalid.");
            }
            catch (IllegalArgumentException e) {
                throw VMError.shouldNotReachHere("Could not parse JFR argument '" + key.cmdLineKey + "=" + value + "'. " + e.getMessage());
            }
        }
        return null;
    }

    private static Long parseMaxSize(Map<JfrStartArgument, String> args, JfrStartArgument key) {
        String value = args.get((Object)key);
        if (value != null) {
            try {
                long number;
                int idx = JfrManager.indexOfFirstNonDigitCharacter(value);
                try {
                    number = Long.parseLong(value.substring(0, idx));
                }
                catch (NumberFormatException e) {
                    throw new IllegalArgumentException("Expected a number.");
                }
                if (idx == value.length()) {
                    return number;
                }
                char unit = value.substring(idx).charAt(0);
                switch (unit) {
                    case 'K': 
                    case 'k': {
                        return number * 1024L;
                    }
                    case 'M': 
                    case 'm': {
                        return number * 1024L * 1024L;
                    }
                    case 'G': 
                    case 'g': {
                        return number * 1024L * 1024L * 1024L;
                    }
                }
                return number;
            }
            catch (IllegalArgumentException e) {
                throw VMError.shouldNotReachHere("Could not parse JFR argument '" + key.cmdLineKey + "=" + value + "'. " + e.getMessage());
            }
        }
        return null;
    }

    private static int indexOfFirstNonDigitCharacter(String durationText) {
        int idx;
        for (idx = 0; idx < durationText.length() && Character.isDigit(durationText.charAt(idx)); ++idx) {
        }
        return idx;
    }

    private static JfrStartArgument findArgument(JfrStartArgument[] possibleArguments, String value) {
        for (JfrStartArgument arg : possibleArguments) {
            if (!arg.cmdLineKey.equals(value)) continue;
            return arg;
        }
        return null;
    }

    private static enum JfrStartArgument {
        Name("name"),
        Settings("settings"),
        Delay("delay"),
        Duration("duration"),
        Filename("filename"),
        Disk("disk"),
        MaxAge("maxage"),
        MaxSize("maxsize"),
        DumpOnExit("dumponexit"),
        PathToGCRoots("path-to-gc-roots");

        private final String cmdLineKey;

        private JfrStartArgument(String key) {
            this.cmdLineKey = key;
        }
    }
}

