/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.hosted;

import com.oracle.svm.core.util.InterruptImageBuilding;
import com.oracle.svm.core.util.VMError;
import com.oracle.svm.hosted.NativeImageClassLoader;
import com.oracle.svm.hosted.NativeImageGenerator;
import com.oracle.svm.util.ModuleSupport;
import java.io.IOException;
import java.io.InputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.channels.ClosedByInterruptException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.graalvm.collections.EconomicSet;
import org.graalvm.compiler.serviceprovider.JavaVersionUtil;
import org.graalvm.compiler.word.Word;
import org.graalvm.nativeimage.Platform;
import org.graalvm.nativeimage.Platforms;

public final class ImageClassLoader {
    public static final String PROPERTY_IMAGEINCLUDEBUILTINMODULES = "substratevm.ImageIncludeBuiltinModules";
    private static final String CLASS_EXTENSION = ".class";
    private static final int CLASS_EXTENSION_LENGTH = ".class".length();
    private static final int CLASS_LOADING_MAX_SCALING = 8;
    private static final int CLASS_LOADING_TIMEOUT_IN_MINUTES = 10;
    final Platform platform;
    private final NativeImageClassLoader classLoader;
    private final EconomicSet<Class<?>> applicationClasses = EconomicSet.create();
    private final EconomicSet<Class<?>> hostedOnlyClasses = EconomicSet.create();
    private final EconomicSet<Method> systemMethods = EconomicSet.create();
    private final EconomicSet<Field> systemFields = EconomicSet.create();
    private static Set<Path> excludeDirectories;

    private ImageClassLoader(Platform platform, NativeImageClassLoader classLoader) {
        this.platform = platform;
        this.classLoader = classLoader;
    }

    public static ImageClassLoader create(Platform platform, NativeImageClassLoader classLoader) {
        NativeImageGenerator.setSystemPropertiesForImageEarly();
        ImageClassLoader result = new ImageClassLoader(platform, classLoader);
        result.initAllClasses();
        return result;
    }

    private static Path toRealPath(Path p) {
        try {
            return p.toRealPath(new LinkOption[0]);
        }
        catch (IOException e) {
            throw VMError.shouldNotReachHere("Path.toRealPath failed for " + p, e);
        }
    }

    private static void addOptionalModule(Set<String> modules, String name) {
        if (ModuleSupport.hasSystemModule((String)name)) {
            modules.add(name);
        }
    }

    private void initAllClasses() {
        ForkJoinPool executor = new ForkJoinPool(Math.min(Runtime.getRuntime().availableProcessors(), 8));
        if (JavaVersionUtil.JAVA_SPEC > 8) {
            HashSet<String> modules = new HashSet<String>();
            modules.add("jdk.internal.vm.ci");
            ImageClassLoader.addOptionalModule(modules, "org.graalvm.sdk");
            ImageClassLoader.addOptionalModule(modules, "jdk.internal.vm.compiler");
            ImageClassLoader.addOptionalModule(modules, "com.oracle.graal.graal_enterprise");
            String includeModulesStr = System.getProperty(PROPERTY_IMAGEINCLUDEBUILTINMODULES);
            if (includeModulesStr != null) {
                modules.addAll(Arrays.asList(includeModulesStr.split(",")));
            }
            for (String moduleResource : ModuleSupport.getModuleResources(modules)) {
                if (!moduleResource.endsWith(CLASS_EXTENSION)) continue;
                executor.execute(() -> this.handleClassFileName(moduleResource, '/'));
            }
        }
        TreeSet<Path> uniquePaths = new TreeSet<Path>(Comparator.comparing(ImageClassLoader::toRealPath));
        uniquePaths.addAll(this.classpath());
        uniquePaths.parallelStream().forEach(path -> this.loadClassesFromPath(executor, (Path)path));
        boolean completed = executor.awaitQuiescence(10L, TimeUnit.MINUTES);
        if (!completed) {
            throw VMError.shouldNotReachHere("timed out while initializing classes");
        }
        executor.shutdownNow();
    }

    private static Set<Path> getExcludeDirectories() {
        Path root = Paths.get("/", new String[0]);
        return Arrays.asList("dev", "sys", "proc", "etc", "var", "tmp", "boot", "lost+found").stream().map(root::resolve).collect(Collectors.toSet());
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void loadClassesFromPath(ForkJoinPool executor, Path path) {
        if (!Files.exists(path, new LinkOption[0])) return;
        if (Files.isRegularFile(path, new LinkOption[0])) {
            try {
                FileSystem probeJarFileSystem;
                URI jarURI = new URI("jar:" + path.toAbsolutePath().toUri());
                try {
                    probeJarFileSystem = FileSystems.newFileSystem(jarURI, Collections.emptyMap());
                }
                catch (UnsupportedOperationException e) {
                    return;
                }
                if (probeJarFileSystem == null) return;
                try (FileSystem jarFileSystem = probeJarFileSystem;){
                    this.initAllClasses(jarFileSystem.getPath("/", new String[0]), Collections.emptySet(), executor);
                    return;
                }
            }
            catch (ClosedByInterruptException ignored) {
                throw new InterruptImageBuilding();
            }
            catch (IOException e) {
                throw VMError.shouldNotReachHere(e);
            }
            catch (URISyntaxException e) {
                throw VMError.shouldNotReachHere(e);
            }
        }
        this.initAllClasses(path, excludeDirectories, executor);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void findSystemElements(Class<?> systemClass) {
        Method[] declaredMethods = null;
        try {
            declaredMethods = systemClass.getDeclaredMethods();
        }
        catch (Throwable t) {
            ImageClassLoader.handleClassLoadingError(t);
        }
        if (declaredMethods != null) {
            for (Method systemMethod : declaredMethods) {
                if (!ImageClassLoader.annotationsAvailable(systemMethod) || !NativeImageGenerator.includedIn(this.platform, systemMethod.getAnnotation(Platforms.class))) continue;
                EconomicSet<Method> economicSet = this.systemMethods;
                synchronized (economicSet) {
                    this.systemMethods.add((Object)systemMethod);
                }
            }
        }
        Field[] declaredFields = null;
        try {
            declaredFields = systemClass.getDeclaredFields();
        }
        catch (Throwable t) {
            ImageClassLoader.handleClassLoadingError(t);
        }
        if (declaredFields != null) {
            for (Field systemField : declaredFields) {
                if (!ImageClassLoader.annotationsAvailable(systemField) || !NativeImageGenerator.includedIn(this.platform, systemField.getAnnotation(Platforms.class))) continue;
                EconomicSet<Field> economicSet = this.systemFields;
                synchronized (economicSet) {
                    this.systemFields.add((Object)systemField);
                }
            }
        }
    }

    private static boolean canLoadAnnotations(AnnotatedElement element) {
        try {
            element.getAnnotations();
            return true;
        }
        catch (Throwable t) {
            ImageClassLoader.handleClassLoadingError(t);
            return false;
        }
    }

    private static boolean annotationsAvailable(AnnotatedElement element) {
        try {
            Annotation[] annotations = element.getAnnotations();
            return annotations.length != 0;
        }
        catch (Throwable t) {
            ImageClassLoader.handleClassLoadingError(t);
            return false;
        }
    }

    private static void handleClassLoadingError(Throwable t) {
    }

    private void handleClassFileName(String unversionedClassFileName, char fileSystemSeparatorChar) {
        String unversionedClassFileNameWithoutSuffix = unversionedClassFileName.substring(0, unversionedClassFileName.length() - CLASS_EXTENSION_LENGTH);
        if (unversionedClassFileNameWithoutSuffix.equals("module-info")) {
            return;
        }
        String className = unversionedClassFileNameWithoutSuffix.replace(fileSystemSeparatorChar, '.');
        Class<?> clazz = null;
        try {
            clazz = this.forName(className);
        }
        catch (Throwable t) {
            ImageClassLoader.handleClassLoadingError(t);
        }
        if (clazz != null) {
            this.handleClass(clazz);
        }
    }

    private void initAllClasses(final Path root, final Set<Path> excludes, final ForkJoinPool executor) {
        SimpleFileVisitor<Path> visitor = new SimpleFileVisitor<Path>(){
            private final char fileSystemSeparatorChar;
            {
                this.fileSystemSeparatorChar = root.getFileSystem().getSeparator().charAt(0);
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (excludes.contains(dir)) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                return super.preVisitDirectory(dir, attrs);
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (excludes.contains(file.getParent())) {
                    return FileVisitResult.SKIP_SIBLINGS;
                }
                String fileName = root.relativize(file).toString();
                if (fileName.endsWith(ImageClassLoader.CLASS_EXTENSION)) {
                    executor.execute(() -> ImageClassLoader.this.handleClassFileName(this.unversionedFileName(fileName), this.fileSystemSeparatorChar));
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                return FileVisitResult.CONTINUE;
            }

            private String unversionedFileName(String fileName) {
                int versionedSuffixIndex;
                String versionedPrefix = "META-INF/versions/";
                String versionedSuffix = "/";
                String result = fileName;
                if (fileName.startsWith("META-INF/versions/") && (versionedSuffixIndex = fileName.indexOf("/", "META-INF/versions/".length())) >= 0) {
                    result = fileName.substring(versionedSuffixIndex + "/".length());
                }
                return result;
            }
        };
        try {
            Files.walkFileTree(root, (FileVisitor<? super Path>)visitor);
        }
        catch (IOException ex) {
            throw VMError.shouldNotReachHere(ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleClass(Class<?> clazz) {
        boolean inPlatform = true;
        boolean isHostedOnly = false;
        AnnotatedElement cur = clazz.getPackage();
        if (cur == null) {
            cur = clazz;
        }
        do {
            if (!ImageClassLoader.canLoadAnnotations(cur)) {
                return;
            }
            Platforms platformsAnnotation = cur.getAnnotation(Platforms.class);
            if (ImageClassLoader.containsHostedOnly(platformsAnnotation)) {
                isHostedOnly = true;
            } else if (!NativeImageGenerator.includedIn(this.platform, platformsAnnotation)) {
                inPlatform = false;
            }
            if (cur instanceof Package) {
                cur = clazz;
                continue;
            }
            try {
                cur = ((Class)cur).getEnclosingClass();
            }
            catch (Throwable t) {
                ImageClassLoader.handleClassLoadingError(t);
                cur = null;
            }
        } while (cur != null);
        if (inPlatform) {
            EconomicSet<Class<?>> economicSet;
            if (isHostedOnly) {
                economicSet = this.hostedOnlyClasses;
                synchronized (economicSet) {
                    this.hostedOnlyClasses.add(clazz);
                }
            }
            economicSet = this.applicationClasses;
            synchronized (economicSet) {
                this.applicationClasses.add(clazz);
            }
            this.findSystemElements(clazz);
        }
    }

    private static boolean containsHostedOnly(Platforms platformsAnnotation) {
        if (platformsAnnotation != null) {
            for (Class platformClass : platformsAnnotation.value()) {
                if (platformClass != Platform.HOSTED_ONLY.class) continue;
                return true;
            }
        }
        return false;
    }

    public Enumeration<URL> findResourcesByName(String resource) throws IOException {
        return this.classLoader.getResources(resource);
    }

    public InputStream findResourceAsStreamByName(String resource) {
        return this.classLoader.getResourceAsStream(resource);
    }

    public Class<?> findClassByName(String name) {
        return this.findClassByName(name, true);
    }

    public Class<?> findClassByName(String name, boolean failIfClassMissing) {
        try {
            if (name.indexOf(46) == -1) {
                switch (name) {
                    case "boolean": {
                        return Boolean.TYPE;
                    }
                    case "char": {
                        return Character.TYPE;
                    }
                    case "float": {
                        return Float.TYPE;
                    }
                    case "double": {
                        return Double.TYPE;
                    }
                    case "byte": {
                        return Byte.TYPE;
                    }
                    case "short": {
                        return Short.TYPE;
                    }
                    case "int": {
                        return Integer.TYPE;
                    }
                    case "long": {
                        return Long.TYPE;
                    }
                    case "void": {
                        return Void.TYPE;
                    }
                }
            }
            return this.forName(name);
        }
        catch (ClassNotFoundException | NoClassDefFoundError ex) {
            if (failIfClassMissing) {
                throw VMError.shouldNotReachHere("class " + name + " not found");
            }
            return null;
        }
    }

    private Class<?> forName(String name) throws ClassNotFoundException {
        return Class.forName(name, false, this.classLoader);
    }

    @Deprecated
    public List<String> getClasspath() {
        return this.classpath().stream().map(Path::toString).collect(Collectors.toList());
    }

    public List<Path> classpath() {
        return Stream.concat(this.classLoader.buildcp.stream(), this.classLoader.imagecp.stream()).collect(Collectors.toList());
    }

    public <T> List<Class<? extends T>> findSubclasses(Class<T> baseClass, boolean includeHostedOnly) {
        ArrayList<Class<? extends T>> result = new ArrayList<Class<? extends T>>();
        ImageClassLoader.addSubclasses(this.applicationClasses, baseClass, result);
        if (includeHostedOnly) {
            ImageClassLoader.addSubclasses(this.hostedOnlyClasses, baseClass, result);
        }
        return result;
    }

    private static <T> void addSubclasses(EconomicSet<Class<?>> classes, Class<T> baseClass, ArrayList<Class<? extends T>> result) {
        for (Class systemClass : classes) {
            if (!baseClass.isAssignableFrom(systemClass)) continue;
            result.add(systemClass.asSubclass(baseClass));
        }
    }

    public List<Class<?>> findAnnotatedClasses(Class<? extends Annotation> annotationClass, boolean includeHostedOnly) {
        ArrayList result = new ArrayList();
        ImageClassLoader.addAnnotatedClasses(this.applicationClasses, annotationClass, result);
        if (includeHostedOnly) {
            ImageClassLoader.addAnnotatedClasses(this.hostedOnlyClasses, annotationClass, result);
        }
        return result;
    }

    private static void addAnnotatedClasses(EconomicSet<Class<?>> classes, Class<? extends Annotation> annotationClass, ArrayList<Class<?>> result) {
        for (Class systemClass : classes) {
            if (systemClass.getAnnotation(annotationClass) == null) continue;
            result.add(systemClass);
        }
    }

    public List<Method> findAnnotatedMethods(Class<? extends Annotation> annotationClass) {
        ArrayList<Method> result = new ArrayList<Method>();
        for (Method method : this.systemMethods) {
            if (method.getAnnotation(annotationClass) == null) continue;
            result.add(method);
        }
        return result;
    }

    public List<Method> findAnnotatedMethods(Class<? extends Annotation>[] annotationClasses) {
        ArrayList<Method> result = new ArrayList<Method>();
        for (Method method : this.systemMethods) {
            boolean match = true;
            for (Class<? extends Annotation> annotationClass : annotationClasses) {
                if (method.getAnnotation(annotationClass) != null) continue;
                match = false;
                break;
            }
            if (!match) continue;
            result.add(method);
        }
        return result;
    }

    public List<Field> findAnnotatedFields(Class<? extends Annotation> annotationClass) {
        ArrayList<Field> result = new ArrayList<Field>();
        for (Field field : this.systemFields) {
            if (field.getAnnotation(annotationClass) == null) continue;
            result.add(field);
        }
        return result;
    }

    public List<Class<? extends Annotation>> allAnnotations() {
        return StreamSupport.stream(this.applicationClasses.spliterator(), false).filter(Class::isAnnotation).map(clazz -> clazz).collect(Collectors.toList());
    }

    <T extends Annotation> List<T> findAnnotations(Class<T> annotationClass) {
        ArrayList<T> result = new ArrayList<T>();
        for (Class<?> clazz : this.findAnnotatedClasses(annotationClass, false)) {
            result.add(clazz.getAnnotation(annotationClass));
        }
        for (Method method : this.findAnnotatedMethods(annotationClass)) {
            result.add(method.getAnnotation(annotationClass));
        }
        for (Field field : this.findAnnotatedFields(annotationClass)) {
            result.add(field.getAnnotation(annotationClass));
        }
        return result;
    }

    public NativeImageClassLoader getClassLoader() {
        return this.classLoader;
    }

    static {
        Word.ensureInitialized();
        excludeDirectories = ImageClassLoader.getExcludeDirectories();
    }
}

