/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.graal.pointsto.reports;

import com.oracle.graal.pointsto.BigBang;
import com.oracle.graal.pointsto.meta.AnalysisMethod;
import com.oracle.graal.pointsto.meta.AnalysisType;
import com.oracle.graal.pointsto.meta.InvokeInfo;
import com.oracle.graal.pointsto.meta.PointsToAnalysisMethod;
import com.oracle.graal.pointsto.reports.AnalysisReportsOptions;
import com.oracle.graal.pointsto.reports.ReportUtils;
import com.oracle.graal.pointsto.util.AnalysisError;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Deque;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import jdk.graal.compiler.java.LambdaUtils;
import jdk.vm.ci.code.BytecodePosition;
import jdk.vm.ci.meta.JavaKind;
import jdk.vm.ci.meta.ResolvedJavaMethod;

public final class CallTreePrinter {
    public static final Pattern CAMEL_CASE_PATTERN = Pattern.compile("\\b[a-zA-Z]|[A-Z]|\\.");
    private final BigBang bb;
    private final Map<AnalysisMethod, MethodNode> methodToNode;
    private static final String METHOD_FORMAT = "%H.%n(%P):%R";

    public static void print(BigBang bb, String reportsPath, String reportName) {
        CallTreePrinter printer = new CallTreePrinter(bb);
        printer.buildCallTree();
        AnalysisReportsOptions.CallTreeType optionValue = (AnalysisReportsOptions.CallTreeType)((Object)AnalysisReportsOptions.PrintAnalysisCallTreeType.getValue(bb.getOptions()));
        switch (optionValue) {
            case TXT: {
                ReportUtils.report("call tree", reportsPath, "call_tree_" + reportName, "txt", printer::printMethods);
                break;
            }
            case CSV: {
                CallTreePrinter.printCsvFiles(printer.methodToNode, reportsPath, reportName);
                break;
            }
            default: {
                throw AnalysisError.shouldNotReachHere("Unsupported CallTreeType " + String.valueOf((Object)optionValue) + " used with PrintAnalysisCallTreeType option");
            }
        }
        ReportUtils.report("list of used methods", reportsPath, "used_methods_" + reportName, "txt", printer::printUsedMethods);
        ReportUtils.report("list of used classes", reportsPath, "used_classes_" + reportName, "txt", writer -> printer.printClasses((PrintWriter)writer, false));
        ReportUtils.report("list of used packages", reportsPath, "used_packages_" + reportName, "txt", writer -> printer.printClasses((PrintWriter)writer, true));
    }

    public CallTreePrinter(BigBang bb) {
        this.bb = bb;
        this.methodToNode = new LinkedHashMap<AnalysisMethod, MethodNode>();
    }

    public void buildCallTree() {
        ArrayList<ResolvedJavaMethod> roots = new ArrayList<ResolvedJavaMethod>();
        for (AnalysisMethod analysisMethod : this.bb.getUniverse().getMethods()) {
            if (analysisMethod.isDirectRootMethod() && analysisMethod.isSimplyImplementationInvoked()) {
                roots.add(analysisMethod);
            }
            if (!analysisMethod.isVirtualRootMethod()) continue;
            for (AnalysisMethod impl : analysisMethod.getImplementations()) {
                AnalysisError.guarantee(impl.isImplementationInvoked());
                roots.add(impl);
            }
        }
        roots.sort(ReportUtils.methodComparator);
        for (AnalysisMethod analysisMethod : roots) {
            this.methodToNode.put(analysisMethod, new MethodNode(analysisMethod, true));
        }
        ArrayDeque<MethodNode> workList = new ArrayDeque<MethodNode>(this.methodToNode.values());
        while (!workList.isEmpty()) {
            MethodNode methodNode = workList.removeFirst();
            ArrayList<InvokeInfo> invokeInfos = new ArrayList<InvokeInfo>();
            for (InvokeInfo invokeInfo : methodNode.method.getInvokes()) {
                invokeInfos.add(invokeInfo);
            }
            invokeInfos.sort(ReportUtils.invokeInfoComparator);
            for (InvokeInfo invokeInfo : invokeInfos) {
                this.processInvoke(invokeInfo, methodNode, workList);
            }
        }
    }

    private void processInvoke(InvokeInfo invokeInfo, MethodNode callerNode, Deque<MethodNode> workList) {
        InvokeNode invokeNode = new InvokeNode(invokeInfo.getTargetMethod(), invokeInfo.isDirectInvoke(), CallTreePrinter.sourceReference(invokeInfo.getPosition()));
        callerNode.addInvoke(invokeNode);
        invokeInfo.getAllCallees().stream().sorted(ReportUtils.methodComparator).forEach(callee -> {
            if (this.methodToNode.containsKey(callee)) {
                MethodNodeReference calleeNode = new MethodNodeReference(this.methodToNode.get(callee));
                invokeNode.addCallee(calleeNode);
            } else {
                MethodNode calleeNode = new MethodNode((AnalysisMethod)callee);
                invokeNode.addCallee(calleeNode);
                this.methodToNode.put((AnalysisMethod)callee, calleeNode);
                workList.add(calleeNode);
            }
        });
    }

    private static SourceReference[] sourceReference(BytecodePosition position) {
        ArrayList<SourceReference> sourceReference = new ArrayList<SourceReference>();
        for (BytecodePosition state = position; state != null; state = state.getCaller()) {
            sourceReference.add(new SourceReference(state.getBCI(), state.getMethod().asStackTraceElement(state.getBCI())));
        }
        return sourceReference.toArray(new SourceReference[sourceReference.size()]);
    }

    private void printMethods(PrintWriter out) {
        out.println("VM Entry Points");
        Iterator iterator = this.methodToNode.values().stream().filter(n -> n.isEntryPoint).iterator();
        while (iterator.hasNext()) {
            MethodNode node = (MethodNode)iterator.next();
            boolean lastEntryPoint = !iterator.hasNext();
            out.format("%s%s %s, parsing reason:  %s %n", lastEntryPoint ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", "entry", node.format(), PointsToAnalysisMethod.unwrapInvokeReason(node.method.getImplementationInvokedReason()));
            CallTreePrinter.printCallTreeNode(out, lastEntryPoint ? "    " : "\u2502   ", node);
        }
        out.println();
    }

    private static void printCallTreeNode(PrintWriter out, String prefix, MethodNode node) {
        for (int invokeIdx = 0; invokeIdx < node.invokes.size(); ++invokeIdx) {
            boolean lastInvoke;
            InvokeNode invoke = node.invokes.get(invokeIdx);
            boolean bl = lastInvoke = invokeIdx == node.invokes.size() - 1;
            if (invoke.isDirectInvoke) {
                if (invoke.callees.size() <= 0) continue;
                Node calleeNode = invoke.callees.get(0);
                out.format("%s%s%s %s @bci=%s %n", prefix, lastInvoke ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", "directly calls", calleeNode.format(), invoke.formatLocation());
                if (!(calleeNode instanceof MethodNode)) continue;
                CallTreePrinter.printCallTreeNode(out, prefix + (lastInvoke ? "    " : "\u2502   "), (MethodNode)calleeNode);
                continue;
            }
            out.format("%s%s%s %s @bci=%s%n", prefix, lastInvoke ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", "virtually calls", invoke.formatTarget(), invoke.formatLocation());
            for (int calleeIdx = 0; calleeIdx < invoke.callees.size(); ++calleeIdx) {
                boolean lastCallee = calleeIdx == invoke.callees.size() - 1;
                Node calleeNode = invoke.callees.get(calleeIdx);
                out.format("%s%s%s %s %n", prefix + (lastInvoke ? "    " : "\u2502   "), lastCallee ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", "is overridden by", calleeNode.format());
                if (!(calleeNode instanceof MethodNode)) continue;
                CallTreePrinter.printCallTreeNode(out, prefix + (lastInvoke ? "    " : "\u2502   ") + (lastCallee ? "    " : "\u2502   "), (MethodNode)calleeNode);
            }
        }
    }

    private void printUsedMethods(PrintWriter out) {
        ArrayList<String> methodsList = new ArrayList<String>();
        for (AnalysisMethod method : this.methodToNode.keySet()) {
            methodsList.add(method.format("%H.%n(%p):%r"));
        }
        methodsList.sort(null);
        for (String name : methodsList) {
            out.println(name);
        }
    }

    private void printClasses(PrintWriter out, boolean packageNameOnly) {
        ArrayList<String> classList = new ArrayList<String>(this.classesSet(packageNameOnly));
        classList.sort(null);
        for (String name : classList) {
            out.println(name);
        }
    }

    public Set<String> classesSet(boolean packageNameOnly) {
        HashSet<String> classSet = new HashSet<String>();
        for (AnalysisMethod method : this.methodToNode.keySet()) {
            String name = method.getDeclaringClass().toJavaName(true);
            if (packageNameOnly && LambdaUtils.isLambdaClassName((String)(name = CallTreePrinter.packagePrefix(name)))) {
                name = CallTreePrinter.packagePrefix(name);
            }
            classSet.add(name);
        }
        return classSet;
    }

    private static String packagePrefix(String name) {
        int lastDot = name.lastIndexOf(46);
        if (lastDot == -1) {
            return name;
        }
        return name.substring(0, lastDot);
    }

    private static void printCsvFiles(Map<AnalysisMethod, MethodNode> methodToNode, String reportsPath, String reportName) {
        HashSet<MethodNode> nodes = new HashSet<MethodNode>();
        List<MethodNode> entrypoints = methodToNode.values().stream().filter(n -> n.isEntryPoint).toList();
        for (MethodNode entrypoint : entrypoints) {
            CallTreePrinter.walkNodes(entrypoint, nodes, methodToNode);
        }
        String msgPrefix = "call tree csv file for ";
        String timeStamp = ReportUtils.getTimeStampString();
        CallTreePrinter.toCsvFile(msgPrefix + "methods", reportsPath, "call_tree_methods", reportName, timeStamp, writer -> CallTreePrinter.printMethodNodes(methodToNode.values(), writer));
        CallTreePrinter.toCsvFile(msgPrefix + "invokes", reportsPath, "call_tree_invokes", reportName, timeStamp, writer -> CallTreePrinter.printInvokeNodes(methodToNode, writer));
        CallTreePrinter.toCsvFile(msgPrefix + "targets", reportsPath, "call_tree_targets", reportName, timeStamp, writer -> CallTreePrinter.printCallTargets(methodToNode, writer));
    }

    private static void toCsvFile(String description, String reportsPath, String prefix, String reportName, String timeStamp, Consumer<PrintWriter> reporter) {
        String name = prefix + "_" + reportName;
        Path csvFile = ReportUtils.report(description, reportsPath, name, "csv", reporter, true, timeStamp);
        Path csvLink = Paths.get(reportsPath, new String[0]).resolve(prefix + ".csv");
        if (Files.exists(csvLink, LinkOption.NOFOLLOW_LINKS)) {
            try {
                Files.delete(csvLink);
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        try {
            Files.createSymbolicLink(csvLink, csvFile.getFileName(), new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static void printMethodNodes(Collection<MethodNode> methods, PrintWriter writer) {
        writer.println(CallTreePrinter.convertToCSV("Id", "Name", "Type", "Parameters", "Return", "Display", "Flags", "IsEntryPoint"));
        methods.stream().map(CallTreePrinter::methodNodeInfo).map(CallTreePrinter::convertToCSV).forEach(writer::println);
    }

    private static void printInvokeNodes(Map<AnalysisMethod, MethodNode> methodToNode, PrintWriter writer) {
        writer.println(CallTreePrinter.convertToCSV("Id", "MethodId", "BytecodeIndexes", "TargetId", "IsDirect"));
        methodToNode.values().stream().flatMap(node -> node.invokes.stream().filter(invoke -> !invoke.callees.isEmpty()).map(invoke -> CallTreePrinter.invokeNodeInfo(methodToNode, node, invoke))).map(CallTreePrinter::convertToCSV).forEach(writer::println);
    }

    private static void printCallTargets(Map<AnalysisMethod, MethodNode> methodToNode, PrintWriter writer) {
        writer.println(CallTreePrinter.convertToCSV("InvokeId", "TargetId"));
        methodToNode.values().stream().flatMap(node -> node.invokes.stream().filter(invoke -> !invoke.callees.isEmpty()).flatMap(invoke -> invoke.callees.stream().map(callee -> CallTreePrinter.callTargetInfo(invoke, callee)))).map(CallTreePrinter::convertToCSV).forEach(writer::println);
    }

    private static List<String> methodNodeInfo(MethodNode method) {
        return CallTreePrinter.resolvedJavaMethodInfo(method.id, method.method);
    }

    private static List<String> invokeNodeInfo(Map<AnalysisMethod, MethodNode> methodToNode, MethodNode method, InvokeNode invoke) {
        return Arrays.asList(String.valueOf(invoke.id), String.valueOf(method.id), CallTreePrinter.showBytecodeIndexes(CallTreePrinter.bytecodeIndexes(invoke)), String.valueOf(methodToNode.get((Object)invoke.targetMethod).id), String.valueOf(invoke.isDirectInvoke));
    }

    private static List<String> callTargetInfo(InvokeNode invoke, Node callee) {
        MethodNode methodNode;
        if (callee instanceof MethodNodeReference) {
            MethodNodeReference ref = (MethodNodeReference)callee;
            methodNode = ref.methodNode;
        } else {
            methodNode = (MethodNode)callee;
        }
        MethodNode node = methodNode;
        return Arrays.asList(String.valueOf(invoke.id), String.valueOf(node.id));
    }

    private static void walkNodes(MethodNode methodNode, Set<MethodNode> nodes, Map<AnalysisMethod, MethodNode> methodToNode) {
        for (InvokeNode invoke : methodNode.invokes) {
            methodToNode.computeIfAbsent(invoke.targetMethod, MethodNode::new);
            if (invoke.isDirectInvoke) {
                if (invoke.callees.size() <= 0) continue;
                Node calleeNode = invoke.callees.get(0);
                CallTreePrinter.addNode(calleeNode, nodes);
                if (!(calleeNode instanceof MethodNode)) continue;
                CallTreePrinter.walkNodes((MethodNode)calleeNode, nodes, methodToNode);
                continue;
            }
            for (Node calleeNode : invoke.callees) {
                if (!(calleeNode instanceof MethodNode)) continue;
                CallTreePrinter.walkNodes((MethodNode)calleeNode, nodes, methodToNode);
            }
        }
    }

    private static void addNode(Node calleeNode, Set<MethodNode> nodes) {
        MethodNode methodNode = calleeNode instanceof MethodNode ? (MethodNode)calleeNode : ((MethodNodeReference)calleeNode).methodNode;
        nodes.add(methodNode);
    }

    private static List<Integer> bytecodeIndexes(InvokeNode node) {
        return Stream.of(node.sourceReferences).map(source -> source.bci).collect(Collectors.toList());
    }

    private static String showBytecodeIndexes(List<Integer> bytecodeIndexes) {
        return bytecodeIndexes.stream().map(String::valueOf).collect(Collectors.joining("->"));
    }

    private static List<String> resolvedJavaMethodInfo(Integer id, AnalysisMethod method) {
        String parameters = method.getSignature().getParameterCount(false) > 0 ? method.format("%P").replace(",", "") : "empty";
        return Arrays.asList(id == null ? null : Integer.toString(id), method.getName(), method.getDeclaringClass().toJavaName(true), parameters, ((AnalysisType)method.getSignature().getReturnType()).toJavaName(true), CallTreePrinter.display(method), CallTreePrinter.flags(method), String.valueOf(method.isEntryPoint()));
    }

    private static String display(AnalysisMethod method) {
        AnalysisType type = method.getDeclaringClass();
        String typeName = type.toJavaName(true);
        if (type.getJavaKind() == JavaKind.Object) {
            ArrayList<String> matchResults = new ArrayList<String>();
            Matcher matcher = CAMEL_CASE_PATTERN.matcher(typeName);
            while (matcher.find()) {
                matchResults.add(matcher.toMatchResult().group());
            }
            return String.join((CharSequence)"", matchResults) + "." + method.getName();
        }
        return typeName + "." + method.getName();
    }

    private static String flags(AnalysisMethod method) {
        StringBuilder sb = new StringBuilder();
        if (method.isPublic()) {
            sb.append('p');
        } else if (method.isPrivate()) {
            sb.append('P');
        } else if (method.isProtected()) {
            sb.append('d');
        }
        if (method.isStatic()) {
            sb.append('s');
        }
        if (method.isFinal()) {
            sb.append('f');
        }
        if (method.isSynchronized()) {
            sb.append('S');
        }
        if (method.isBridge()) {
            sb.append('b');
        }
        if (method.isVarArgs()) {
            sb.append('v');
        }
        if (method.isNative()) {
            sb.append('n');
        }
        if (method.isAbstract()) {
            sb.append('a');
        }
        if (method.isSynthetic()) {
            sb.append('y');
        }
        return sb.toString();
    }

    private static String convertToCSV(String ... data) {
        return String.join((CharSequence)",", data);
    }

    private static String convertToCSV(List<String> data) {
        return String.join((CharSequence)",", data);
    }

    static class MethodNode
    implements Node {
        static int methodId = 0;
        private final int id = methodId++;
        private final AnalysisMethod method;
        private final List<InvokeNode> invokes;
        private final boolean isEntryPoint;

        MethodNode(AnalysisMethod method) {
            this(method, false);
        }

        MethodNode(AnalysisMethod method, boolean isEntryPoint) {
            this.method = method;
            this.invokes = new ArrayList<InvokeNode>();
            this.isEntryPoint = isEntryPoint;
        }

        void addInvoke(InvokeNode invoke) {
            this.invokes.add(invoke);
        }

        @Override
        public String format() {
            return this.method.format(CallTreePrinter.METHOD_FORMAT) + " id=" + this.id;
        }
    }

    static class InvokeNode {
        static int invokeId = 0;
        private final int id = invokeId++;
        private final AnalysisMethod targetMethod;
        private final List<Node> callees;
        private final boolean isDirectInvoke;
        private final SourceReference[] sourceReferences;

        InvokeNode(AnalysisMethod targetMethod, boolean isDirectInvoke, SourceReference[] sourceReferences) {
            this.targetMethod = targetMethod;
            this.isDirectInvoke = isDirectInvoke;
            this.sourceReferences = sourceReferences;
            this.callees = new ArrayList<Node>();
        }

        void addCallee(Node callee) {
            this.callees.add(callee);
        }

        String formatLocation() {
            return Arrays.stream(this.sourceReferences).map(s -> String.valueOf(s.bci)).collect(Collectors.joining("->"));
        }

        String formatTarget() {
            return this.targetMethod.format(CallTreePrinter.METHOD_FORMAT);
        }
    }

    static class SourceReference {
        final int bci;
        final StackTraceElement trace;

        SourceReference(int bci, StackTraceElement trace) {
            this.bci = bci;
            this.trace = trace;
        }
    }

    static interface Node {
        public String format();
    }

    static class MethodNodeReference
    implements Node {
        private final MethodNode methodNode;

        MethodNodeReference(MethodNode methodNode) {
            this.methodNode = methodNode;
        }

        @Override
        public String format() {
            return this.methodNode.method.format(CallTreePrinter.METHOD_FORMAT) + " id-ref=" + this.methodNode.id;
        }
    }
}

