/*
 * (c) 2001-2009, Progress Software Corporation and/or its subsidiaries or affiliates.  All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.fusesource.commons.management.basic;

import java.lang.management.ManagementFactory;
import java.util.EventObject;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.Logger;

import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.ObjectName;

import org.fusesource.commons.management.ManagementStrategy;
import org.fusesource.commons.management.Statistic;
import org.fusesource.commons.management.Statistic.UpdateMode;

/**
 * A base class from which JMX based ManagementStrategy implementations may use as a starting point.
 * <p>
 * This base class implements the following:
 * <ul>
 *   <li>Support for injection, discovery (using the default MBeanServer or by name) or creation
 *       of an MBeanServer.</li>
 *   <li>Default logging of events using java.util.Logging</li>
 *   <li>Provision of a simple statistic implementation suitable for most application uses.</li>
 * </ul>
 */
public abstract class AbstractManagementStrategy implements ManagementStrategy {
    
    /**
     * The default domain name to be used when creating a MBeanServer
     */
    public static final String DEFAULT_DOMAIN = "org.fusesource";
    
    private boolean usePlatformMBeanServer = true;
    
    private String mbeanServerDefaultDomain = DEFAULT_DOMAIN;
    
    private MBeanServer mbeanServer;
    
    private LogManager logManager = LogManager.getLogManager();
    

    //
    // ManagementStrategy interface
    //
    
    public void manageObject(Object managedObject) throws Exception  {
        ObjectName name = getManagedObjectName(managedObject, null, ObjectName.class);
        if (name != null) {
            manageNamedObject(managedObject, name);
        }
    }
    
    public void unmanageObject(Object managedObject) throws Exception {
        ObjectName name = getManagedObjectName(managedObject, null, ObjectName.class);
        if (name != null) {
            unmanageNamedObject(name);
        }
    }
    
    /**
     * A default implementation of notify that logs events to java.util.logging using the event class name
     * for the logger, a Level.FINE as the log level and the event source and event toString() to create
     * a loggable message.
     * <p>
     * Applications that use other logging systems (such as commons logging or log4j will typically
     * override this implementation of notify(EventObject).
     */
    public void notify(EventObject event) throws Exception {
        Logger logger = this.logManager.getLogger(event.getClass().getName());
        if (logger != null && logger.isLoggable(Level.FINE)) {
            logger.fine(event.getSource() + ": " + event.toString());
        }
    }
    
    public Statistic createStatistic(String name, Object owner, UpdateMode updateMode) {
        return new StatisticImpl(updateMode);
    }
    
    
    //
    // Bean setters/getters
    //

    /**
     * Gets the JMX MBeanServer.
     * 
     * @return the mbean server
     */
    @SuppressWarnings("unchecked")
    public synchronized MBeanServer getMbeanServer() {
        if (this.mbeanServer == null) {
            if (this.usePlatformMBeanServer) {
                this.mbeanServer = ManagementFactory.getPlatformMBeanServer();
            } else {
                List mbeanServers = MBeanServerFactory.findMBeanServer(this.mbeanServerDefaultDomain);
                if (mbeanServers.size() == 0) {
                    this.mbeanServer = MBeanServerFactory.createMBeanServer(this.mbeanServerDefaultDomain);
                } else {
                    this.mbeanServer = (MBeanServer)mbeanServers.get(0);
                }
            }
        }

        return this.mbeanServer;
    }

    /**
     * Sets the JMX MBeanServer.
     * 
     * @param mbeanServer the new mbean server
     */
    public synchronized void setMbeanServer(MBeanServer mbeanServer) {
        if (this.mbeanServer != null) {
            throw new IllegalStateException("MBeanServer has already been set");
        }
        
        if (mbeanServer == null) {
            throw new IllegalArgumentException("MBeanServer cannot be null");
        }
        
        this.mbeanServer = mbeanServer;
    }

    /**
     * Gets the value of the usePlatformMBeanServer property.
     * 
     * @return whether to use platform mbean server or not
     */
    public boolean getUsePlatformMBeanServer() {
        return this.usePlatformMBeanServer;
    }

    /**
     * Sets the value of the usePlatformMBeanServer property.
     * 
     * @param usePlatformMBeanServer If true, then use the JVM's platform MBeanServer. If false, find or
     *                               create an MBeanServer using the mbeanServerDefaultDomain property.
     */
    public void setUsePlatformMBeanServer(boolean usePlatformMBeanServer) {
        this.usePlatformMBeanServer = usePlatformMBeanServer;
    }

    /**
     * Gets the mbeanServerDefaultDomain property.
     * 
     * @return The MBeanServerDefaultDomainName.
     */
    public String getMbeanServerDefaultDomain() {
        return this.mbeanServerDefaultDomain;
    }

    /**
     * Sets the mbeanServerDefaultDomain property.
     * <p/>
     * If the MBeanServer is not explicitly set and using the default platform MBeanServer has been disabled,
     * then an MBeanServer with the given default domain name will be discovered or created.
     * 
     * @param mbeanServerDefaultDomain The MBeanServerDefaultDomainName.
     */
    public void setMbeanServerDefaultDomain(String mbeanServerDefaultDomain) {
        this.mbeanServerDefaultDomain = mbeanServerDefaultDomain;
    }
}
