/*
 * (c) 2001-2009, Progress Software Corporation and/or its subsidiaries or affiliates.  All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.fusesource.commons.management;

import java.util.EventObject;

import org.fusesource.commons.management.Statistic.UpdateMode;

/**
 * A ManagementStrategy provides an abstraction for a high level, common set of management functions:
 * <ul>
 *   <li>(Un)Registering object that expose some aspect of management (e.g. management API, statistics,
 *       etc.)</li>
 *   <li>Notification of events significant to enterprise management</li>
 *   <li>Creation of statistics for recording application performance</li>
 * </ul>
 * <p>
 * ManagementStrategy implementations decide how such functions are exposed (e.g. via JMX or another
 * technology). Applications should allow for injection of a strategy implementation.
 */
public interface ManagementStrategy {
    
    /**
     * Adds a managed object allowing the ManagementStrategy implementation to record or expose
     * the object as it sees fit.
     * 
     * @param managedObject the managed object
     * @throws Exception can be thrown if the object could not be added
     */
    void manageObject(Object managedObject) throws Exception;

    /**
     * Adds a managed object allowing the ManagementStrategy implementation
     * to record or expose the object as it sees fit.
     *
     * @param managedObject the managed object
     * @param preferedName representing the preferred name, maybe a String, or a JMX ObjectName
     * @throws Exception can be thrown if the object could not be added
     */
    void manageNamedObject(Object managedObject, Object preferedName) throws Exception;

    /**
     * Construct an object name, where either the object to be managed and/or
     * a custom name component are provided
     *
     * @param managableObject the object to be managed
     * @param customName a custom name component
     * @param nameType the name type required
     * @return an object name of the required type if supported, otherwise <tt>null</tt>
     * @throws Exception can be thrown if the object name could not be created
     */
    <T> T getManagedObjectName(Object managableObject, String customName, Class<T> nameType) throws Exception;

    /**
     * Removes the managed object.
     * 
     * @param managedObject the managed object
     * @throws Exception can be thrown if the object could not be removed
     */
    void unmanageObject(Object managedObject) throws Exception;

    /**
     * Removes a managed object by name.
     *
     * @param name an object name previously created by this strategy.
     * @throws Exception can be thrown if the object could not be removed
     */
    void unmanageNamedObject(Object name) throws Exception;

    /**
     * Determines if an object or name is managed.
     *
     * @param managableObject the object to consider
     * @param name the name to consider
     * @return <tt>true</tt> if the given object or name is managed
     */
    boolean isManaged(Object managableObject, Object name);

    /**
     * Management events provide a single model for capturing information about execution points in the
     * application code. Management strategy implementations decide if and where to record these events.
     * Applications communicate events to management strategy implementations via the notify(EventObject) 
     * method.
     * 
     * @param event the event
     * @throws Exception can be thrown if the notification failed
     */
    void notify(EventObject event) throws Exception;
    
    /**
     * Creates a statistic; a representation of raw statistical data.
     * 
     * @param name The unique name assigned to the statistic usage.
     * @param owner The object whose identity is to be associated with the metric.
     * @param updateMode The update mode to be assigned to the returned statistic.
     * @return the statistic
     * 
     * @see Statistic.UpdateMode
     */
    Statistic createStatistic(String name, Object owner, UpdateMode updateMode);
}
