# pylint: disable=line-too-long, invalid-name, missing-function-docstring, missing-module-docstring, superfluous-parens
# pylint: disable=wrong-import-position, unused-import, unused-wildcard-import, wildcard-import, wrong-import-order, missing-class-docstring
from __future__ import annotations
from typing import List, Optional
from datetime import date
from datetime import time
from datetime import datetime
from decimal import Decimal
from pydantic import Field
from rosetta.runtime.utils import *

__all__ = ['PriceSchedule']

from cdm.base.math.MeasureSchedule import MeasureSchedule

class PriceSchedule(MeasureSchedule):
  """
  Specifies the price of a financial instrument in a trade as a schedule of measures. A price generically expresses the value of an exchange as a ratio: it measures the amount of one thing needed to be exchanged for 1 unit of another thing (e.g. cash in a specific currency in exchange for a bond or share). This generic representation can be used to support any type of financial price beyond just cash price: e.g. an interest rate, a foreign exchange rate, etc. This data type is generically based on a schedule and can also be used to represent a price as a single value.
  """
  perUnitOf: Optional[UnitType] = Field(None, description="Provides an attribute to define the unit of the thing being priced. For example, {amount, unitOfAmount, PerUnitOfAmount} = [10, EUR, Shares] = (10.00 EUR/SHARE) * (300,000 SHARES) = EUR 3,000,000.00 (Shares cancel out in the calculation).")
  """
  Provides an attribute to define the unit of the thing being priced. For example, {amount, unitOfAmount, PerUnitOfAmount} = [10, EUR, Shares] = (10.00 EUR/SHARE) * (300,000 SHARES) = EUR 3,000,000.00 (Shares cancel out in the calculation).
  """
  priceType: PriceTypeEnum = Field(..., description="Specifies the price type as an enumeration: interest rate, exchange rate, asset price etc. This attribute is mandatory so that prices can always be clasiffied according to their type. The price type implies some constraints on the price's units.")
  """
  Specifies the price type as an enumeration: interest rate, exchange rate, asset price etc. This attribute is mandatory so that prices can always be clasiffied according to their type. The price type implies some constraints on the price's units.
  """
  priceExpression: Optional[PriceExpressionEnum] = Field(None, description="(Optionally) Specifies whether the price is expressed in absolute or percentage terms.")
  """
  (Optionally) Specifies whether the price is expressed in absolute or percentage terms.
  """
  composite: Optional[PriceComposite] = Field(None, description="(Optionally) Specifies the underlying price components if the price can be expressed as a composite: e.g. dirty price = clean price + accrued.")
  """
  (Optionally) Specifies the underlying price components if the price can be expressed as a composite: e.g. dirty price = clean price + accrued.
  """
  arithmeticOperator: Optional[ArithmeticOperationEnum] = Field(None, description="(Optionally) When the price is to be understood as an operator to apply to an observable, i.e. a spread, multiplier or min/max.")
  """
  (Optionally) When the price is to be understood as an operator to apply to an observable, i.e. a spread, multiplier or min/max.
  """
  cashPrice: Optional[CashPrice] = Field(None, description="(Optionally when the price type is cash) Additional attributes that further define a cash price, e.g. what type of fee it is.")
  """
  (Optionally when the price type is cash) Additional attributes that further define a cash price, e.g. what type of fee it is.
  """
  
  @rosetta_condition
  def condition_0_UnitOfAmountExists(self):
    """
    Requires that a unit of amount must be specified for price unless price type is Variance, Volatility or Correlation.
    """
    def _then_fn0():
      return (((_resolve_rosetta_attr(self, "unit")) is None) and ((_resolve_rosetta_attr(self, "perUnitOf")) is None))
    
    def _else_fn0():
      return (((_resolve_rosetta_attr(self, "unit")) is not None) and ((_resolve_rosetta_attr(self, "perUnitOf")) is not None))
    
    return if_cond_fn(((all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "VARIANCE")) or all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "VOLATILITY"))) or all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "CORRELATION"))), _then_fn0, _else_fn0)
  
  @rosetta_condition
  def condition_1_PositiveAssetPrice(self):
    """
    Requires that per FpML rules, the FX rate must be a positive value.
    """
    def _then_fn0():
      return all_elements(_resolve_rosetta_attr(self, "value"), ">", 0)
    
    def _else_fn0():
      return True
    
    return if_cond_fn(((all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "EXCHANGE_RATE")) or all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "ASSET_PRICE"))) and ((_resolve_rosetta_attr(self, "arithmeticOperator")) is None)), _then_fn0, _else_fn0)
  
  @rosetta_condition
  def condition_2_PositiveSpotRate(self):
    """
    Requires that per FpML rules, the spot rate must be a positive value, for example for FX or Commodities.
    """
    def _then_fn0():
      return all_elements(_resolve_rosetta_attr(_resolve_rosetta_attr(self, "composite"), "baseValue"), ">", 0)
    
    def _else_fn0():
      return True
    
    return if_cond_fn(((all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "EXCHANGE_RATE")) or all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "ASSET_PRICE"))) and ((_resolve_rosetta_attr(_resolve_rosetta_attr(self, "composite"), "baseValue")) is not None)), _then_fn0, _else_fn0)
  
  @rosetta_condition
  def condition_3_PositiveCashPrice(self):
    """
    Requires that any price expressed as a cash price and generating a cashflow must be positive
    """
    def _then_fn0():
      return (all_elements(_resolve_rosetta_attr(self, "value"), ">", 0) or all_elements(_resolve_rosetta_attr(_resolve_rosetta_attr(self, "datedValue"), "value"), ">", 0))
    
    def _else_fn0():
      return True
    
    return if_cond_fn(all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "CASH_PRICE")), _then_fn0, _else_fn0)
  
  @rosetta_condition
  def condition_4_CurrencyUnitForInterestRate(self):
    """
    Requires that the unit of amount for an interest rate must be a currency.
    """
    def _then_fn0():
      return ((_resolve_rosetta_attr(_resolve_rosetta_attr(self, "unit"), "currency")) is not None)
    
    def _else_fn0():
      return True
    
    return if_cond_fn(all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "INTEREST_RATE")), _then_fn0, _else_fn0)
  
  @rosetta_condition
  def condition_5_Choice(self):
    """
    The price can be specified mutually exclusively as a cashflow, an operator (e.g. a spread or multiplier), or a composite.
    """
    return self.check_one_of_constraint('cashPrice', 'arithmeticOperator', 'composite', necessity=False)
  
  @rosetta_condition
  def condition_6_CashPrice(self):
    """
    If a cash price type is specified, the price type must be cash, otherwise it must be non-cash.
    """
    def _then_fn0():
      return all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "CASH_PRICE"))
    
    def _else_fn0():
      return True
    
    return if_cond_fn(((_resolve_rosetta_attr(self, "cashPrice")) is not None), _then_fn0, _else_fn0)
  
  @rosetta_condition
  def condition_7_ArithmeticOperator(self):
    """
    Operator must not be subtract or divide.
    """
    return (any_elements(_resolve_rosetta_attr(self, "arithmeticOperator"), "<>", _resolve_rosetta_attr(ArithmeticOperationEnum, "SUBTRACT")) and any_elements(_resolve_rosetta_attr(self, "arithmeticOperator"), "<>", _resolve_rosetta_attr(ArithmeticOperationEnum, "DIVIDE")))
  
  @rosetta_condition
  def condition_8_SpreadPrice(self):
    """
    A spread type can only be specified when the price type is an asset price or an interest rate.
    """
    def _then_fn0():
      return (all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "ASSET_PRICE")) or all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "INTEREST_RATE")))
    
    def _else_fn0():
      return True
    
    return if_cond_fn(all_elements(_resolve_rosetta_attr(self, "arithmeticOperator"), "=", _resolve_rosetta_attr(ArithmeticOperationEnum, "ADD")), _then_fn0, _else_fn0)
  
  @rosetta_condition
  def condition_9_ForwardPoint(self):
    """
    If composite operand type is ForwardPoint then the price type must be ExchangeRate.
    """
    def _then_fn0():
      return all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "EXCHANGE_RATE"))
    
    def _else_fn0():
      return True
    
    return if_cond_fn(all_elements(_resolve_rosetta_attr(_resolve_rosetta_attr(self, "composite"), "operandType"), "=", _resolve_rosetta_attr(PriceOperandEnum, "FORWARD_POINT")), _then_fn0, _else_fn0)
  
  @rosetta_condition
  def condition_10_AccruedInterest(self):
    """
    If composite operand type is AccruedInterest then the price type must be AssetPrice.
    """
    def _then_fn0():
      return all_elements(_resolve_rosetta_attr(self, "priceType"), "=", _resolve_rosetta_attr(PriceTypeEnum, "ASSET_PRICE"))
    
    def _else_fn0():
      return True
    
    return if_cond_fn(all_elements(_resolve_rosetta_attr(_resolve_rosetta_attr(self, "composite"), "operandType"), "=", _resolve_rosetta_attr(PriceOperandEnum, "ACCRUED_INTEREST")), _then_fn0, _else_fn0)

from cdm.base.math.UnitType import UnitType
from cdm.observable.asset.PriceTypeEnum import PriceTypeEnum
from cdm.observable.asset.PriceExpressionEnum import PriceExpressionEnum
from cdm.observable.asset.PriceComposite import PriceComposite
from cdm.base.math.ArithmeticOperationEnum import ArithmeticOperationEnum
from cdm.observable.asset.CashPrice import CashPrice
from cdm.observable.asset.PriceOperandEnum import PriceOperandEnum

PriceSchedule.update_forward_refs()
