package cdm.product.asset.floatingrate.functions;

import cdm.base.math.functions.Max;
import cdm.observable.asset.Price;
import cdm.product.asset.NegativeInterestRateTreatmentEnum;
import cdm.product.asset.RateTreatmentEnum;
import cdm.product.asset.floatingrate.FloatingRateProcessingDetails;
import cdm.product.asset.floatingrate.FloatingRateProcessingDetails.FloatingRateProcessingDetailsBuilder;
import cdm.product.asset.floatingrate.FloatingRateProcessingParameters;
import cdm.product.common.schedule.CalculationPeriodBase;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.expression.MapperMaths;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import java.math.BigDecimal;
import java.util.Optional;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(ApplyFloatingRateProcessing.ApplyFloatingRateProcessingDefault.class)
public abstract class ApplyFloatingRateProcessing implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected ApplyFloatingRatePostSpreadProcessing applyFloatingRatePostSpreadProcessing;
	@Inject protected ApplyUSRateTreatment applyUSRateTreatment;
	@Inject protected Max max;

	/**
	* @param processing THe parameters to be used for processing, such as multipliers, spreads, cap rates, etc.
	* @param rawRate The floating rate prior to treatment, either a single term rate, or a calculated rate such as an OIS or lookback compounded rate.
	* @param calculationPeriod The calculation period for with the processing need to be performed.
	* @param isInitialPeriod Is this the initial calculation period of the payout?
	* @return details Results are details of the rate treatment.
	*/
	public FloatingRateProcessingDetails evaluate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
		FloatingRateProcessingDetails.FloatingRateProcessingDetailsBuilder detailsBuilder = doEvaluate(processing, rawRate, calculationPeriod, isInitialPeriod);
		
		final FloatingRateProcessingDetails details;
		if (detailsBuilder == null) {
			details = null;
		} else {
			details = detailsBuilder.build();
			objectValidator.validate(FloatingRateProcessingDetails.class, details);
		}
		
		return details;
	}

	protected abstract FloatingRateProcessingDetails.FloatingRateProcessingDetailsBuilder doEvaluate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> multiplier(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> multiplied(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> multipliedRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> treatedRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<NegativeInterestRateTreatmentEnum> negativeTreatment(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> negativeTreatedRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> spreadRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> added(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> ratePlusSpread(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> negativeTreatedRatePlusSpread(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<Boolean> doInitialRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> initialRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> initialRatePluSpread(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	protected abstract Mapper<BigDecimal> initialRatePlusSpread(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod);

	public static class ApplyFloatingRateProcessingDefault extends ApplyFloatingRateProcessing {
		@Override
		protected FloatingRateProcessingDetails.FloatingRateProcessingDetailsBuilder doEvaluate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			FloatingRateProcessingDetails.FloatingRateProcessingDetailsBuilder details = FloatingRateProcessingDetails.builder();
			return assignOutput(details, processing, rawRate, calculationPeriod, isInitialPeriod);
		}
		
		protected FloatingRateProcessingDetails.FloatingRateProcessingDetailsBuilder assignOutput(FloatingRateProcessingDetails.FloatingRateProcessingDetailsBuilder details, FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			details
				.setProcessingParameters(MapperS.of(processing).get());
			
			details
				.setRawRate(MapperS.of(rawRate).get());
			
			details
				.setProcessedRate(MapperUtils.runSingle(() -> {
					if (areEqual(MapperS.of(doInitialRate(processing, rawRate, calculationPeriod, isInitialPeriod).get()), MapperS.of(Boolean.valueOf(true)), CardinalityOperator.All).getOrDefault(false)) {
						return MapperS.of(initialRatePluSpread(processing, rawRate, calculationPeriod, isInitialPeriod).get());
					}
					else {
						return MapperS.of(applyFloatingRatePostSpreadProcessing.evaluate(MapperS.of(ratePlusSpread(processing, rawRate, calculationPeriod, isInitialPeriod).get()).get(), MapperS.of(processing).get()));
					}
				}).get());
			
			details
				.setSpreadExclusiveRate(MapperUtils.runSingle(() -> {
					if (areEqual(MapperS.of(doInitialRate(processing, rawRate, calculationPeriod, isInitialPeriod).get()), MapperS.of(Boolean.valueOf(true)), CardinalityOperator.All).getOrDefault(false)) {
						return MapperS.of(initialRate(processing, rawRate, calculationPeriod, isInitialPeriod).get());
					}
					else {
						return MapperS.of(applyFloatingRatePostSpreadProcessing.evaluate(MapperS.of(negativeTreatedRate(processing, rawRate, calculationPeriod, isInitialPeriod).get()).get(), MapperS.of(processing).get()));
					}
				}).get());
			
			return Optional.ofNullable(details)
				.map(o -> o.prune())
				.orElse(null);
		}
		
		@Override
		protected Mapper<BigDecimal> multiplier(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperS.of(processing).<BigDecimal>map("getMultiplier", floatingRateProcessingParameters -> floatingRateProcessingParameters.getMultiplier());
		}
		
		@Override
		protected Mapper<BigDecimal> multiplied(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperMaths.<BigDecimal, BigDecimal, BigDecimal>multiply(MapperS.of(rawRate), MapperS.of(multiplier(processing, rawRate, calculationPeriod, isInitialPeriod).get()));
		}
		
		@Override
		protected Mapper<BigDecimal> multipliedRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(multiplier(processing, rawRate, calculationPeriod, isInitialPeriod).get())).getOrDefault(false)) {
					return MapperS.of(multiplied(processing, rawRate, calculationPeriod, isInitialPeriod).get());
				}
				else {
					return MapperS.of(rawRate);
				}
			});
		}
		
		@Override
		protected Mapper<BigDecimal> treatedRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperMaths.<BigDecimal, BigDecimal, BigDecimal>multiply(MapperS.of(applyUSRateTreatment.evaluate(MapperS.of(multipliedRate(processing, rawRate, calculationPeriod, isInitialPeriod).get()).get(), MapperS.of(processing).<RateTreatmentEnum>map("getTreatment", floatingRateProcessingParameters -> floatingRateProcessingParameters.getTreatment()).get(), MapperS.of(calculationPeriod).get())), MapperS.of(new BigDecimal("1.0")));
		}
		
		@Override
		protected Mapper<NegativeInterestRateTreatmentEnum> negativeTreatment(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperS.of(processing).<NegativeInterestRateTreatmentEnum>map("getNegativeTreatment", floatingRateProcessingParameters -> floatingRateProcessingParameters.getNegativeTreatment());
		}
		
		@Override
		protected Mapper<BigDecimal> negativeTreatedRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperUtils.runSingle(() -> {
				if (areEqual(MapperS.of(negativeTreatment(processing, rawRate, calculationPeriod, isInitialPeriod).get()), MapperS.of(NegativeInterestRateTreatmentEnum.ZERO_INTEREST_RATE_EXCLUDING_SPREAD_METHOD), CardinalityOperator.All).getOrDefault(false)) {
					return MapperS.of(max.evaluate(MapperS.of(new BigDecimal("0.0")).get(), MapperS.of(treatedRate(processing, rawRate, calculationPeriod, isInitialPeriod).get()).get()));
				}
				else {
					return MapperS.of(treatedRate(processing, rawRate, calculationPeriod, isInitialPeriod).get());
				}
			});
		}
		
		@Override
		protected Mapper<BigDecimal> spreadRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperS.of(processing).<BigDecimal>map("getSpread", floatingRateProcessingParameters -> floatingRateProcessingParameters.getSpread());
		}
		
		@Override
		protected Mapper<BigDecimal> added(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperMaths.<BigDecimal, BigDecimal, BigDecimal>add(MapperS.of(negativeTreatedRate(processing, rawRate, calculationPeriod, isInitialPeriod).get()), MapperS.of(spreadRate(processing, rawRate, calculationPeriod, isInitialPeriod).get()));
		}
		
		@Override
		protected Mapper<BigDecimal> ratePlusSpread(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(spreadRate(processing, rawRate, calculationPeriod, isInitialPeriod).get())).getOrDefault(false)) {
					return MapperS.of(added(processing, rawRate, calculationPeriod, isInitialPeriod).get());
				}
				else {
					return MapperS.of(negativeTreatedRate(processing, rawRate, calculationPeriod, isInitialPeriod).get());
				}
			});
		}
		
		@Override
		protected Mapper<BigDecimal> negativeTreatedRatePlusSpread(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperUtils.runSingle(() -> {
				if (areEqual(MapperS.of(negativeTreatment(processing, rawRate, calculationPeriod, isInitialPeriod).get()), MapperS.of(NegativeInterestRateTreatmentEnum.ZERO_INTEREST_RATE_METHOD), CardinalityOperator.All).getOrDefault(false)) {
					return MapperS.of(max.evaluate(MapperS.of(new BigDecimal("0.0")).get(), MapperS.of(ratePlusSpread(processing, rawRate, calculationPeriod, isInitialPeriod).get()).get()));
				}
				else {
					return MapperS.of(ratePlusSpread(processing, rawRate, calculationPeriod, isInitialPeriod).get());
				}
			});
		}
		
		@Override
		protected Mapper<Boolean> doInitialRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperUtils.runSingle(() -> {
				if (areEqual(MapperS.of(isInitialPeriod), MapperS.of(Boolean.valueOf(true)), CardinalityOperator.All).and(exists(MapperS.of(processing).<Price>map("getInitialRate", floatingRateProcessingParameters -> floatingRateProcessingParameters.getInitialRate()))).getOrDefault(false)) {
					return MapperS.of(Boolean.valueOf(true));
				}
				else {
					return MapperS.of(Boolean.valueOf(false));
				}
			});
		}
		
		@Override
		protected Mapper<BigDecimal> initialRate(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperS.of(processing).<Price>map("getInitialRate", floatingRateProcessingParameters -> floatingRateProcessingParameters.getInitialRate()).<BigDecimal>map("getValue", measureBase -> measureBase.getValue());
		}
		
		@Override
		protected Mapper<BigDecimal> initialRatePluSpread(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperMaths.<BigDecimal, BigDecimal, BigDecimal>add(MapperS.of(initialRate(processing, rawRate, calculationPeriod, isInitialPeriod).get()), MapperS.of(spreadRate(processing, rawRate, calculationPeriod, isInitialPeriod).get()));
		}
		
		@Override
		protected Mapper<BigDecimal> initialRatePlusSpread(FloatingRateProcessingParameters processing, BigDecimal rawRate, CalculationPeriodBase calculationPeriod, Boolean isInitialPeriod) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(spreadRate(processing, rawRate, calculationPeriod, isInitialPeriod).get())).getOrDefault(false)) {
					return MapperS.of(initialRatePluSpread(processing, rawRate, calculationPeriod, isInitialPeriod).get());
				}
				else {
					return MapperS.of(initialRate(processing, rawRate, calculationPeriod, isInitialPeriod).get());
				}
			});
		}
	}
}
