package cdm.observable.asset.calculatedrate.functions;

import cdm.base.datetime.BusinessCenterEnum;
import cdm.base.datetime.BusinessCenters;
import cdm.base.datetime.functions.AppendDateToList;
import cdm.base.datetime.functions.GetAllBusinessCenters;
import cdm.observable.asset.calculatedrate.FloatingRateCalculationParameters;
import cdm.observable.asset.calculatedrate.ObservationShiftCalculation;
import cdm.observable.asset.calculatedrate.OffsetCalculation;
import cdm.product.common.schedule.CalculationPeriodBase;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import com.rosetta.model.lib.records.Date;
import java.util.ArrayList;
import java.util.List;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(DetermineWeightingDates.DetermineWeightingDatesDefault.class)
public abstract class DetermineWeightingDates implements RosettaFunction {
	
	// RosettaFunction dependencies
	//
	@Inject protected AppendDateToList appendDateToList;
	@Inject protected GenerateObservationDates generateObservationDates;
	@Inject protected GetAllBusinessCenters getAllBusinessCenters;

	/**
	* @param calculationParams Floating rate definition for the calculated rate.
	* @param observationDates 
	* @param observationPeriod The resulting observation period.
	* @param adjustedCalculationPeriod The calculation period for which the rate is being computed, after any adjustment.
	* @param lockoutDays The number of lockout day.
	* @return weightingDates 
	*/
	public List<Date> evaluate(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays) {
		List<Date> weightingDates = doEvaluate(calculationParams, observationDates, observationPeriod, adjustedCalculationPeriod, lockoutDays);
		
		return weightingDates;
	}

	protected abstract List<Date> doEvaluate(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays);

	protected abstract Mapper<? extends ObservationShiftCalculation> obsShift(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays);

	protected abstract Mapper<? extends OffsetCalculation> lookback(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays);

	protected abstract Mapper<BusinessCenterEnum> businessCenters(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays);

	protected abstract Mapper<Date> baseWeightingDates(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays);

	protected abstract Mapper<? extends CalculationPeriodBase> wtPeriod(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays);

	protected abstract Mapper<Date> weightingDatesAll(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays);

	public static class DetermineWeightingDatesDefault extends DetermineWeightingDates {
		@Override
		protected List<Date> doEvaluate(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays) {
			List<Date> weightingDates = new ArrayList<>();
			return assignOutput(weightingDates, calculationParams, observationDates, observationPeriod, adjustedCalculationPeriod, lockoutDays);
		}
		
		protected List<Date> assignOutput(List<Date> weightingDates, FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays) {
			List<Date> addVar = MapperC.<Date>of(weightingDatesAll(calculationParams, observationDates, observationPeriod, adjustedCalculationPeriod, lockoutDays).getMulti()).getMulti();
			weightingDates.addAll(addVar);
			
			return weightingDates;
		}
		
		@Override
		protected Mapper<? extends ObservationShiftCalculation> obsShift(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays) {
			return MapperS.of(calculationParams).<ObservationShiftCalculation>map("getObservationShiftCalculation", floatingRateCalculationParameters -> floatingRateCalculationParameters.getObservationShiftCalculation());
		}
		
		@Override
		protected Mapper<? extends OffsetCalculation> lookback(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays) {
			return MapperS.of(calculationParams).<OffsetCalculation>map("getLookbackCalculation", floatingRateCalculationParameters -> floatingRateCalculationParameters.getLookbackCalculation());
		}
		
		@Override
		protected Mapper<BusinessCenterEnum> businessCenters(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays) {
			return MapperC.<BusinessCenterEnum>of(getAllBusinessCenters.evaluate(MapperS.of(calculationParams).<BusinessCenters>map("getApplicableBusinessDays", floatingRateCalculationParameters -> floatingRateCalculationParameters.getApplicableBusinessDays()).get()));
		}
		
		@Override
		protected Mapper<Date> baseWeightingDates(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays) {
			return MapperUtils.runMulti(() -> {
				if (exists(MapperS.of(obsShift(calculationParams, observationDates, observationPeriod, adjustedCalculationPeriod, lockoutDays).get())).getOrDefault(false)) {
					return MapperC.<Date>of(observationDates);
				}
				else {
					return MapperC.<Date>of(generateObservationDates.evaluate(MapperS.of(adjustedCalculationPeriod).get(), MapperC.<BusinessCenterEnum>of(businessCenters(calculationParams, observationDates, observationPeriod, adjustedCalculationPeriod, lockoutDays).getMulti()).getMulti(), MapperS.of(lockoutDays).get()));
				}
			});
		}
		
		@Override
		protected Mapper<? extends CalculationPeriodBase> wtPeriod(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays) {
			return MapperUtils.runSinglePolymorphic(() -> {
				if (exists(MapperS.of(lookback(calculationParams, observationDates, observationPeriod, adjustedCalculationPeriod, lockoutDays).get())).getOrDefault(false)) {
					return MapperS.of(adjustedCalculationPeriod);
				}
				else {
					return MapperS.of(observationPeriod);
				}
			});
		}
		
		@Override
		protected Mapper<Date> weightingDatesAll(FloatingRateCalculationParameters calculationParams, List<Date> observationDates, CalculationPeriodBase observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, Integer lockoutDays) {
			return MapperC.<Date>of(appendDateToList.evaluate(MapperC.<Date>of(baseWeightingDates(calculationParams, observationDates, observationPeriod, adjustedCalculationPeriod, lockoutDays).getMulti()).getMulti(), MapperS.of(wtPeriod(calculationParams, observationDates, observationPeriod, adjustedCalculationPeriod, lockoutDays).get()).<Date>map("getAdjustedEndDate", calculationPeriodBase -> calculationPeriodBase.getAdjustedEndDate()).get()));
		}
	}
}
