/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.watcher.watch;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.Version;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.admin.indices.refresh.RefreshRequest;
import org.elasticsearch.action.admin.indices.refresh.RefreshResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.common.ParseField;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.common.xcontent.json.JsonXContent;
import org.elasticsearch.index.IndexNotFoundException;
import org.elasticsearch.index.engine.DocumentMissingException;
import org.elasticsearch.indices.TypeMissingException;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.xpack.common.stats.Counters;
import org.elasticsearch.xpack.security.InternalClient;
import org.elasticsearch.xpack.watcher.actions.ActionWrapper;
import org.elasticsearch.xpack.watcher.support.Exceptions;
import org.elasticsearch.xpack.watcher.support.init.proxy.WatcherClientProxy;
import org.elasticsearch.xpack.watcher.support.xcontent.WatcherParams;
import org.elasticsearch.xpack.watcher.trigger.schedule.Schedule;
import org.elasticsearch.xpack.watcher.trigger.schedule.ScheduleTrigger;
import org.elasticsearch.xpack.watcher.watch.Watch;
import org.elasticsearch.xpack.watcher.watch.WatchStoreUtils;

public class WatchStore
extends AbstractComponent {
    public static final String INDEX = ".watches";
    public static final String DOC_TYPE = "doc";
    public static final String LEGACY_DOC_TYPE = "watch";
    private final WatcherClientProxy client;
    private final Watch.Parser watchParser;
    private final ConcurrentMap<String, Watch> watches;
    private final AtomicBoolean started = new AtomicBoolean(false);
    private final int scrollSize;
    private final TimeValue scrollTimeout;

    @Inject
    public WatchStore(Settings settings, InternalClient client, Watch.Parser watchParser) {
        this(settings, new WatcherClientProxy(settings, client), watchParser);
    }

    public WatchStore(Settings settings, WatcherClientProxy client, Watch.Parser watchParser) {
        super(settings);
        this.client = client;
        this.watchParser = watchParser;
        this.watches = ConcurrentCollections.newConcurrentMap();
        this.scrollTimeout = settings.getAsTime("xpack.watcher.watch.scroll.timeout", TimeValue.timeValueSeconds((long)30L));
        this.scrollSize = settings.getAsInt("xpack.watcher.watch.scroll.size", Integer.valueOf(100));
    }

    public void start(ClusterState state) throws Exception {
        if (this.started.get()) {
            this.logger.debug("watch store already started");
            return;
        }
        try {
            IndexMetaData indexMetaData = WatchStoreUtils.getConcreteIndex(INDEX, state.metaData());
            int count = this.loadWatches(indexMetaData);
            this.logger.debug("loaded [{}] watches from the watches index [{}]", (Object)count, (Object)indexMetaData.getIndex().getName());
        }
        catch (IndexNotFoundException indexMetaData) {
        }
        catch (Exception e) {
            this.logger.debug(() -> new ParameterizedMessage("failed to load watches for watch index [{}]", (Object)INDEX), (Throwable)e);
            this.watches.clear();
            throw e;
        }
        this.started.set(true);
    }

    public boolean validate(ClusterState state) {
        try {
            IndexMetaData indexMetaData = WatchStoreUtils.getConcreteIndex(INDEX, state.metaData());
            if (indexMetaData.getState() == IndexMetaData.State.CLOSE) {
                this.logger.debug("watch index [{}] is marked as closed, watcher cannot be started", (Object)indexMetaData.getIndex().getName());
                return false;
            }
            return state.routingTable().index(indexMetaData.getIndex()).allPrimaryShardsActive();
        }
        catch (IndexNotFoundException e) {
            return true;
        }
        catch (IllegalStateException e) {
            this.logger.trace(() -> new ParameterizedMessage("error getting index meta data [{}]: ", (Object)INDEX), (Throwable)e);
            return false;
        }
    }

    public boolean started() {
        return this.started.get();
    }

    public void stop() {
        if (this.started.compareAndSet(true, false)) {
            this.watches.clear();
            this.logger.info("stopped watch store");
        }
    }

    public Watch get(String id) {
        this.ensureStarted();
        return (Watch)this.watches.get(id);
    }

    public WatchPut put(Watch watch) throws IOException {
        IndexResponse response;
        this.ensureStarted();
        try {
            IndexRequest indexRequest = this.createIndexRequest(DOC_TYPE, watch);
            response = this.client.index(indexRequest, (TimeValue)null);
        }
        catch (TypeMissingException e) {
            IndexRequest indexRequest = this.createIndexRequest(LEGACY_DOC_TYPE, watch);
            response = this.client.index(indexRequest, (TimeValue)null);
        }
        watch.status().version(response.getVersion());
        watch.version(response.getVersion());
        Watch previous = this.watches.put(watch.id(), watch);
        if (!this.started()) {
            this.watches.remove(watch.id());
        }
        return new WatchPut(previous, watch, response);
    }

    private IndexRequest createIndexRequest(String docType, Watch watch) throws IOException {
        IndexRequest indexRequest = new IndexRequest(INDEX, docType, watch.id());
        boolean useOldStatus = LEGACY_DOC_TYPE.equals(docType);
        BytesReference source = watch.toXContent(XContentFactory.jsonBuilder(), (ToXContent.Params)WatcherParams.builder().put("include_status", true).put("write_status_with_underscore", useOldStatus).build()).bytes();
        indexRequest.source(BytesReference.toBytes((BytesReference)source), XContentType.JSON);
        indexRequest.version(-3L);
        indexRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        return indexRequest;
    }

    public void updateStatus(Watch watch) throws IOException {
        this.updateStatus(watch, false);
    }

    public void updateStatus(Watch watch, boolean refresh) throws IOException {
        this.ensureStarted();
        if (!watch.status().dirty()) {
            return;
        }
        UpdateResponse response = null;
        try {
            try {
                response = this.client.update(this.createUpdateRequest(watch, DOC_TYPE, Watch.Field.STATUS, refresh));
            }
            catch (DocumentMissingException e) {
                response = this.client.update(this.createUpdateRequest(watch, LEGACY_DOC_TYPE, Watch.Field.STATUS_V5, refresh));
            }
        }
        catch (DocumentMissingException e) {
            this.logger.warn("Watch [{}] was deleted during watch execution, not updating watch status", (Object)watch.id());
        }
        if (response != null) {
            watch.status().version(response.getVersion());
            watch.version(response.getVersion());
            watch.status().resetDirty();
        }
    }

    private UpdateRequest createUpdateRequest(Watch watch, String type, ParseField statusFieldName, boolean refresh) throws IOException {
        XContentBuilder source = JsonXContent.contentBuilder().startObject().field(statusFieldName.getPreferredName(), (ToXContent)watch.status(), ToXContent.EMPTY_PARAMS).endObject();
        UpdateRequest updateRequest = new UpdateRequest(INDEX, type, watch.id());
        updateRequest.doc(source);
        updateRequest.version(watch.version());
        if (refresh) {
            updateRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        }
        return updateRequest;
    }

    public WatchDelete delete(String id) {
        this.ensureStarted();
        Watch watch = (Watch)this.watches.remove(id);
        DeleteRequest request = (DeleteRequest)new DeleteRequest(INDEX, DOC_TYPE, id).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        DeleteResponse response = this.client.delete(request);
        if (response.getResult() == DocWriteResponse.Result.NOT_FOUND) {
            response = this.client.delete((DeleteRequest)new DeleteRequest(INDEX, LEGACY_DOC_TYPE, id).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE));
        }
        if (watch != null) {
            watch.version(response.getVersion());
        }
        return new WatchDelete(response);
    }

    public Collection<Watch> watches() {
        return this.watches.values();
    }

    public Collection<Watch> activeWatches() {
        HashSet<Watch> watches = new HashSet<Watch>();
        for (Watch watch : this.watches()) {
            if (!watch.status().state().isActive()) continue;
            watches.add(watch);
        }
        return watches;
    }

    public Map<String, Object> usageStats() {
        Counters counters = new Counters("count.total", "count.active");
        for (Watch watch : this.watches.values()) {
            Object type;
            boolean isActive = watch.status().state().isActive();
            this.addToCounters("count", isActive, counters);
            if (watch.trigger() != null) {
                this.addToCounters("watch.trigger._all", isActive, counters);
                if ("schedule".equals(watch.trigger().type())) {
                    Schedule schedule = ((ScheduleTrigger)watch.trigger()).getSchedule();
                    this.addToCounters("watch.trigger.schedule._all", isActive, counters);
                    this.addToCounters("watch.trigger.schedule." + schedule.type(), isActive, counters);
                }
            }
            if (watch.input() != null) {
                type = watch.input().type();
                this.addToCounters("watch.input._all", isActive, counters);
                this.addToCounters("watch.input." + (String)type, isActive, counters);
            }
            if (watch.condition() != null) {
                type = watch.condition().type();
                this.addToCounters("watch.condition._all", isActive, counters);
                this.addToCounters("watch.condition." + (String)type, isActive, counters);
            }
            for (ActionWrapper actionWrapper : watch.actions()) {
                String type2 = actionWrapper.action().type();
                this.addToCounters("watch.action." + type2, isActive, counters);
                if (actionWrapper.transform() == null) continue;
                String transformType = actionWrapper.transform().type();
                this.addToCounters("watch.transform." + transformType, isActive, counters);
            }
            if (watch.transform() != null) {
                type = watch.transform().type();
                this.addToCounters("watch.transform." + (String)type, isActive, counters);
            }
            if (watch.metadata() == null || watch.metadata().size() <= 0) continue;
            this.addToCounters("watch.metadata", isActive, counters);
        }
        return counters.toMap();
    }

    private void addToCounters(String name, boolean isActive, Counters counters) {
        counters.inc(name + ".total");
        if (isActive) {
            counters.inc(name + ".active");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int loadWatches(IndexMetaData indexMetaData) {
        assert (this.watches.isEmpty()) : "no watches should reside, but there are [" + this.watches.size() + "] watches.";
        RefreshResponse refreshResponse = this.client.refresh(new RefreshRequest(new String[]{INDEX}));
        if (refreshResponse.getSuccessfulShards() < indexMetaData.getNumberOfShards()) {
            throw Exceptions.illegalState("not all required shards have been refreshed", new Object[0]);
        }
        boolean upgradeSource = indexMetaData.getCreationVersion().before(Version.V_5_0_0_alpha1);
        int count = 0;
        SearchRequest searchRequest = new SearchRequest(new String[]{INDEX}).preference("_primary").scroll(this.scrollTimeout).source(new SearchSourceBuilder().size(this.scrollSize).sort((SortBuilder)SortBuilders.fieldSort((String)"_doc")).version(Boolean.valueOf(true)));
        SearchResponse response = this.client.search(searchRequest, null);
        try {
            if (response.getTotalShards() != response.getSuccessfulShards()) {
                throw new ElasticsearchException("Partial response while loading watches", new Object[0]);
            }
            while (response.getHits().getHits().length != 0) {
                for (SearchHit hit : response.getHits()) {
                    String id = hit.getId();
                    try {
                        BytesReference source = hit.getSourceRef();
                        Watch watch = this.watchParser.parse(id, true, source, XContentFactory.xContentType((BytesReference)source), upgradeSource);
                        watch.status().version(hit.getVersion());
                        watch.version(hit.getVersion());
                        this.watches.put(id, watch);
                        ++count;
                    }
                    catch (Exception e) {
                        this.logger.error(() -> new ParameterizedMessage("couldn't load watch [{}], ignoring it...", (Object)id), (Throwable)e);
                    }
                }
                response = this.client.searchScroll(response.getScrollId(), this.scrollTimeout);
            }
        }
        finally {
            this.client.clearScroll(response.getScrollId());
        }
        return count;
    }

    private void ensureStarted() {
        if (!this.started.get()) {
            throw new IllegalStateException("watch store not started");
        }
    }

    public void clearWatchesInMemory() {
        this.watches.clear();
    }

    public class WatchDelete {
        private final DeleteResponse response;

        public WatchDelete(DeleteResponse response) {
            this.response = response;
        }

        public DeleteResponse deleteResponse() {
            return this.response;
        }
    }

    public class WatchPut {
        private final Watch previous;
        private final Watch current;
        private final IndexResponse response;

        public WatchPut(Watch previous, Watch current, IndexResponse response) {
            this.current = current;
            this.previous = previous;
            this.response = response;
        }

        public Watch current() {
            return this.current;
        }

        public Watch previous() {
            return this.previous;
        }

        public IndexResponse indexResponse() {
            return this.response;
        }
    }
}

