/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.support;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingRequest;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingResponse;
import org.elasticsearch.action.admin.indices.template.put.PutIndexTemplateRequest;
import org.elasticsearch.action.admin.indices.template.put.PutIndexTemplateResponse;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.AliasOrIndex;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.IndexTemplateMetaData;
import org.elasticsearch.cluster.metadata.MappingMetaData;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.compress.CompressedXContent;
import org.elasticsearch.common.inject.internal.Nullable;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.xcontent.XContent;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.common.xcontent.json.JsonXContent;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.security.InternalClient;
import org.elasticsearch.xpack.template.TemplateUtils;

public class IndexLifecycleManager
extends AbstractComponent {
    private static final String SECURITY_VERSION_STRING = "security-version";
    public static final String TEMPLATE_VERSION_PATTERN = Pattern.quote("${security.template.version}");
    private static final int MAX_MIGRATE_ATTEMPTS = 10;
    private final String indexName;
    private final String templateName;
    private final InternalClient client;
    private final IndexDataMigrator migrator;
    private final ClusterService clusterService;
    private final ThreadPool threadPool;
    private final AtomicBoolean templateCreationPending = new AtomicBoolean(false);
    private final AtomicBoolean updateMappingPending = new AtomicBoolean(false);
    private final AtomicReference<UpgradeState> migrateDataState = new AtomicReference<UpgradeState>(UpgradeState.NOT_STARTED);
    private final AtomicInteger migrateDataAttempts = new AtomicInteger(0);
    private volatile boolean templateIsUpToDate;
    private volatile boolean indexExists;
    private volatile boolean indexAvailable;
    private volatile boolean canWriteToIndex;
    private volatile boolean mappingIsUpToDate;
    private volatile Version mappingVersion;
    public static final IndexDataMigrator NULL_MIGRATOR = (version, listener) -> listener.onResponse((Object)false);

    public IndexLifecycleManager(Settings settings, InternalClient client, ClusterService clusterService, ThreadPool threadPool, String indexName, String templateName, IndexDataMigrator migrator) {
        super(settings);
        this.client = client;
        this.indexName = indexName;
        this.templateName = templateName;
        this.migrator = migrator;
        this.clusterService = clusterService;
        this.threadPool = threadPool;
    }

    public boolean isTemplateUpToDate() {
        return this.templateIsUpToDate;
    }

    public boolean isTemplateCreationPending() {
        return this.templateCreationPending.get();
    }

    public boolean isMappingUpToDate() {
        return this.mappingIsUpToDate;
    }

    public Version getMappingVersion() {
        return this.mappingVersion;
    }

    public boolean checkMappingVersion(Predicate<Version> requiredVersion) {
        return this.mappingVersion == null || requiredVersion.test(this.mappingVersion);
    }

    public boolean isMappingUpdatePending() {
        return this.updateMappingPending.get();
    }

    public boolean indexExists() {
        return this.indexExists;
    }

    public boolean isAvailable() {
        return this.indexAvailable;
    }

    public boolean isWritable() {
        return this.canWriteToIndex;
    }

    public UpgradeState getMigrationState() {
        return this.migrateDataState.get();
    }

    public void clusterChanged(ClusterChangedEvent event) {
        ClusterState state = event.state();
        this.processClusterState(state);
    }

    private void processClusterState(ClusterState state) {
        assert (state != null);
        this.indexExists = IndexLifecycleManager.resolveConcreteIndex(this.indexName, state.metaData()) != null;
        this.indexAvailable = this.checkIndexAvailable(state);
        this.templateIsUpToDate = this.checkTemplateExistsAndIsUpToDate(state);
        this.mappingIsUpToDate = this.checkIndexMappingUpToDate(state);
        this.canWriteToIndex = this.templateIsUpToDate && this.mappingIsUpToDate;
        this.mappingVersion = this.oldestIndexMappingVersion(state);
        if (state.nodes().isLocalNodeElectedMaster()) {
            if (!this.templateIsUpToDate) {
                this.updateTemplate();
            }
            if (this.indexAvailable && !this.mappingIsUpToDate) {
                this.migrateData(state, this::updateMapping);
            }
        }
    }

    private boolean checkIndexAvailable(ClusterState state) {
        IndexRoutingTable routingTable = this.getIndexRoutingTable(state);
        if (routingTable != null && routingTable.allPrimaryShardsActive()) {
            return true;
        }
        this.logger.debug("Security index [{}] is not yet active", (Object)this.indexName);
        return false;
    }

    private IndexRoutingTable getIndexRoutingTable(ClusterState clusterState) {
        IndexMetaData metaData = IndexLifecycleManager.resolveConcreteIndex(this.indexName, clusterState.metaData());
        if (metaData == null) {
            return null;
        }
        return clusterState.routingTable().index(metaData.getIndex());
    }

    private boolean checkTemplateExistsAndIsUpToDate(ClusterState state) {
        return IndexLifecycleManager.checkTemplateExistsAndVersionMatches(this.templateName, state, this.logger, arg_0 -> ((Version)Version.CURRENT).equals(arg_0));
    }

    public static boolean checkTemplateExistsAndVersionMatches(String templateName, ClusterState state, Logger logger, Predicate<Version> predicate) {
        IndexTemplateMetaData templateMeta = (IndexTemplateMetaData)state.metaData().templates().get((Object)templateName);
        if (templateMeta == null) {
            return false;
        }
        ImmutableOpenMap mappings = templateMeta.getMappings();
        for (Object typeMapping : mappings.values().toArray()) {
            CompressedXContent typeMappingXContent = (CompressedXContent)typeMapping;
            try {
                Map typeMappingMap = (Map)XContentHelper.convertToMap((BytesReference)new BytesArray(typeMappingXContent.uncompressed()), (boolean)false, (XContentType)XContentType.JSON).v2();
                assert (typeMappingMap.size() == 1);
                String key = (String)typeMappingMap.keySet().iterator().next();
                Map mappingMap = (Map)typeMappingMap.get(key);
                if (IndexLifecycleManager.containsCorrectVersion(mappingMap, predicate)) continue;
                return false;
            }
            catch (ElasticsearchParseException e) {
                logger.error((Message)new ParameterizedMessage("Cannot parse the template [{}]", (Object)templateName), (Throwable)e);
                throw new IllegalStateException("Cannot parse the template " + templateName, e);
            }
        }
        return true;
    }

    private static boolean containsCorrectVersion(Map<String, Object> typeMappingMap, Predicate<Version> predicate) {
        Map meta = (Map)typeMappingMap.get("_meta");
        if (meta == null) {
            return false;
        }
        return predicate.test(Version.fromString((String)((String)meta.get(SECURITY_VERSION_STRING))));
    }

    private boolean checkIndexMappingUpToDate(ClusterState clusterState) {
        return this.checkIndexMappingVersionMatches(clusterState, arg_0 -> ((Version)Version.CURRENT).equals(arg_0));
    }

    private boolean checkIndexMappingVersionMatches(ClusterState clusterState, Predicate<Version> predicate) {
        return IndexLifecycleManager.checkIndexMappingVersionMatches(this.indexName, clusterState, this.logger, predicate);
    }

    public static boolean checkIndexMappingVersionMatches(String indexName, ClusterState clusterState, Logger logger, Predicate<Version> predicate) {
        return IndexLifecycleManager.loadIndexMappingVersions(indexName, clusterState, logger).stream().allMatch(predicate);
    }

    private Version oldestIndexMappingVersion(ClusterState clusterState) {
        Set<Version> versions = IndexLifecycleManager.loadIndexMappingVersions(this.indexName, clusterState, this.logger);
        return versions.stream().min(Comparator.comparingInt(v -> v.id)).orElse(null);
    }

    private static Set<Version> loadIndexMappingVersions(String indexName, ClusterState clusterState, Logger logger) {
        HashSet<Version> versions = new HashSet<Version>();
        IndexMetaData indexMetaData = IndexLifecycleManager.resolveConcreteIndex(indexName, clusterState.metaData());
        if (indexMetaData != null) {
            for (Object object : indexMetaData.getMappings().values().toArray()) {
                MappingMetaData mappingMetaData = (MappingMetaData)object;
                if (mappingMetaData.type().equals("_default_")) continue;
                versions.add(IndexLifecycleManager.readMappingVersion(indexName, mappingMetaData, logger));
            }
        }
        return versions;
    }

    private static IndexMetaData resolveConcreteIndex(String indexOrAliasName, MetaData metaData) {
        AliasOrIndex aliasOrIndex = (AliasOrIndex)metaData.getAliasAndIndexLookup().get(indexOrAliasName);
        if (aliasOrIndex != null) {
            List indices = aliasOrIndex.getIndices();
            if (aliasOrIndex.isAlias() && indices.size() > 1) {
                throw new IllegalStateException("Alias [" + indexOrAliasName + "] points to more than one index: " + indices.stream().map(imd -> imd.getIndex().getName()).collect(Collectors.toList()));
            }
            return (IndexMetaData)indices.get(0);
        }
        return null;
    }

    private static Version readMappingVersion(String indexName, MappingMetaData mappingMetaData, Logger logger) {
        try {
            Map meta = (Map)mappingMetaData.sourceAsMap().get("_meta");
            if (meta == null) {
                logger.info("Missing _meta field in mapping [{}] of index [{}]", (Object)mappingMetaData.type(), (Object)indexName);
                return Version.V_2_3_0;
            }
            return Version.fromString((String)((String)meta.get(SECURITY_VERSION_STRING)));
        }
        catch (IOException e) {
            logger.error((Message)new ParameterizedMessage("Cannot parse the mapping for index [{}]", (Object)indexName), (Throwable)e);
            throw new ElasticsearchException("Cannot parse the mapping for index [{}]", (Throwable)e, new Object[]{indexName});
        }
    }

    private void updateTemplate() {
        if (this.templateCreationPending.compareAndSet(false, true)) {
            this.putTemplate();
        }
    }

    private boolean migrateData(ClusterState state, final Runnable andThen) {
        if (this.migrateDataState.compareAndSet(UpgradeState.NOT_STARTED, UpgradeState.IN_PROGRESS)) {
            final Version previousVersion = this.oldestIndexMappingVersion(state);
            this.migrator.performUpgrade(previousVersion, new ActionListener<Boolean>(){

                public void onResponse(Boolean upgraded) {
                    IndexLifecycleManager.this.migrateDataState.set(UpgradeState.COMPLETE);
                    andThen.run();
                }

                public void onFailure(Exception e) {
                    IndexLifecycleManager.this.migrateDataState.set(UpgradeState.FAILED);
                    int attempts = IndexLifecycleManager.this.migrateDataAttempts.incrementAndGet();
                    IndexLifecycleManager.this.logger.error((Message)new ParameterizedMessage("failed to upgrade security [{}] data from version [{}] (Attempt {} of {})", new Object[]{IndexLifecycleManager.this.indexName, previousVersion, attempts, 10}), (Throwable)e);
                    if (attempts < 10) {
                        TimeValue retry = TimeValue.timeValueMillis((long)((long)Math.pow(attempts, 5.0)));
                        IndexLifecycleManager.this.logger.info("Will attempt upgrade again in {}", (Object)retry);
                        IndexLifecycleManager.this.threadPool.schedule(retry, "same", () -> IndexLifecycleManager.this.retryDataMigration());
                    } else {
                        IndexLifecycleManager.this.logger.error("Security migration has failed after {} attempts. Restart the master node to try again.", (Object)10);
                    }
                }

                public String toString() {
                    return this.getClass() + "{" + IndexLifecycleManager.this.indexName + " migrator}";
                }
            });
            return true;
        }
        if (this.migrateDataState.get() == UpgradeState.COMPLETE) {
            andThen.run();
        }
        return false;
    }

    private void retryDataMigration() {
        if (this.migrateDataState.compareAndSet(UpgradeState.FAILED, UpgradeState.NOT_STARTED)) {
            this.processClusterState(this.clusterService.state());
        }
    }

    private void updateMapping() {
        if (this.updateMappingPending.compareAndSet(false, true)) {
            this.putMappings();
        }
    }

    private void putMappings() {
        Map typeMappingMap;
        String template = TemplateUtils.loadTemplate("/" + this.templateName + ".json", Version.CURRENT.toString(), TEMPLATE_VERSION_PATTERN);
        try {
            typeMappingMap = XContentHelper.convertToMap((XContent)JsonXContent.jsonXContent, (String)template, (boolean)false);
        }
        catch (ElasticsearchParseException e) {
            this.updateMappingPending.set(false);
            this.logger.error((Message)new ParameterizedMessage("failed to parse index template {}", (Object)this.templateName), (Throwable)e);
            throw new ElasticsearchException("failed to parse index template {}", (Throwable)e, new Object[]{this.templateName});
        }
        ConcurrentMap updateResults = ConcurrentCollections.newConcurrentMap();
        Map typeMappings = (Map)typeMappingMap.get("mappings");
        int expectedResults = typeMappings.size();
        for (String type : typeMappings.keySet()) {
            Map typeMapping = (Map)typeMappings.get(type);
            this.putMapping(updateResults, expectedResults, type, typeMapping);
        }
    }

    private void putMapping(final Map<String, PutMappingResponse> updateResults, final int expectedResults, final String type, Map<String, Object> typeMapping) {
        this.logger.debug("updating mapping of the [{}] index for type [{}]", (Object)this.indexName, (Object)type);
        PutMappingRequest putMappingRequest = (PutMappingRequest)this.client.admin().indices().preparePutMapping(new String[]{this.indexName}).setSource(typeMapping).setType(type).request();
        this.client.admin().indices().putMapping(putMappingRequest, (ActionListener)new ActionListener<PutMappingResponse>(){

            public void onResponse(PutMappingResponse putMappingResponse) {
                if (!putMappingResponse.isAcknowledged()) {
                    IndexLifecycleManager.this.updateMappingPending.set(false);
                    throw new ElasticsearchException("update mapping for type [{}] in index [{}] was not acknowledged", new Object[]{type, IndexLifecycleManager.this.indexName});
                }
                updateResults.put(type, putMappingResponse);
                if (updateResults.size() == expectedResults) {
                    IndexLifecycleManager.this.updateMappingPending.set(false);
                }
            }

            public void onFailure(Exception e) {
                IndexLifecycleManager.this.updateMappingPending.set(false);
                IndexLifecycleManager.this.logger.warn(() -> new ParameterizedMessage("failed to update mapping for type [{}] on index [{}]", (Object)type, (Object)IndexLifecycleManager.this.indexName), (Throwable)e);
            }

            public String toString() {
                return this.getClass() + "{" + IndexLifecycleManager.this.indexName + " PutMapping}";
            }
        });
    }

    private void putTemplate() {
        this.logger.debug("putting the template [{}]", (Object)this.templateName);
        String template = TemplateUtils.loadTemplate("/" + this.templateName + ".json", Version.CURRENT.toString(), TEMPLATE_VERSION_PATTERN);
        PutIndexTemplateRequest putTemplateRequest = (PutIndexTemplateRequest)this.client.admin().indices().preparePutTemplate(this.templateName).setSource((BytesReference)new BytesArray(template.getBytes(StandardCharsets.UTF_8)), XContentType.JSON).request();
        this.client.admin().indices().putTemplate(putTemplateRequest, (ActionListener)new ActionListener<PutIndexTemplateResponse>(){

            public void onResponse(PutIndexTemplateResponse putIndexTemplateResponse) {
                IndexLifecycleManager.this.templateCreationPending.set(false);
                if (!putIndexTemplateResponse.isAcknowledged()) {
                    throw new ElasticsearchException("put template [{}] was not acknowledged", new Object[]{IndexLifecycleManager.this.templateName});
                }
                IndexLifecycleManager.this.templateIsUpToDate = true;
            }

            public void onFailure(Exception e) {
                IndexLifecycleManager.this.templateCreationPending.set(false);
                IndexLifecycleManager.this.logger.warn((Message)new ParameterizedMessage("failed to put template [{}]", (Object)IndexLifecycleManager.this.templateName), (Throwable)e);
            }

            public String toString() {
                return this.getClass() + "{" + IndexLifecycleManager.this.indexName + " PutTemplate}";
            }
        });
    }

    public static interface IndexDataMigrator {
        public void performUpgrade(@Nullable Version var1, ActionListener<Boolean> var2);
    }

    public static enum UpgradeState {
        NOT_STARTED,
        IN_PROGRESS,
        COMPLETE,
        FAILED;

    }
}

