/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.crypto;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Objects;
import java.util.regex.Pattern;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.Version;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.env.Environment;
import org.elasticsearch.xpack.XPackPlugin;
import org.elasticsearch.xpack.security.Security;
import org.elasticsearch.xpack.security.authc.support.CharArrays;

public class CryptoService
extends AbstractComponent {
    public static final String KEY_ALGO = "HmacSHA512";
    public static final int KEY_SIZE = 1024;
    static final String FILE_NAME = "system_key";
    static final String HMAC_ALGO = "HmacSHA1";
    static final String DEFAULT_ENCRYPTION_ALGORITHM = "AES/CTR/NoPadding";
    static final String DEFAULT_KEY_ALGORITH = "AES";
    static final String ENCRYPTED_TEXT_PREFIX = "::es_encrypted::";
    static final int DEFAULT_KEY_LENGTH = 128;
    static final int RANDOM_KEY_SIZE = 128;
    private static final Pattern SIG_PATTERN = Pattern.compile("^\\$\\$[0-9]+\\$\\$[^\\$]*\\$\\$.+");
    private static final byte[] HKDF_APP_INFO = "es-security-crypto-service".getBytes(StandardCharsets.UTF_8);
    private static final Setting<Boolean> SYSTEM_KEY_REQUIRED_SETTING = Setting.boolSetting((String)Security.setting("system_key.required"), (boolean)false, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    private static final Setting<String> ENCRYPTION_ALGO_SETTING = new Setting(Security.setting("encryption.algorithm"), s -> "AES/CTR/NoPadding", s -> s, new Setting.Property[]{Setting.Property.NodeScope});
    private static final Setting<Integer> ENCRYPTION_KEY_LENGTH_SETTING = Setting.intSetting((String)Security.setting("encryption_key.length"), (int)128, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    private static final Setting<String> ENCRYPTION_KEY_ALGO_SETTING = new Setting(Security.setting("encryption_key.algorithm"), "AES", s -> s, new Setting.Property[]{Setting.Property.NodeScope});
    private final SecureRandom secureRandom = new SecureRandom();
    private final String encryptionAlgorithm;
    private final String keyAlgorithm;
    private final int keyLength;
    private final int ivLength;
    private final Path keyFile;
    private final SecretKey randomKey;
    private final String randomKeyBase64;
    private final SecretKey encryptionKey;
    private final SecretKey systemKey;
    private final SecretKey signingKey;

    public CryptoService(Settings settings, Environment env) throws IOException {
        super(settings);
        this.encryptionAlgorithm = (String)ENCRYPTION_ALGO_SETTING.get(settings);
        this.keyLength = (Integer)ENCRYPTION_KEY_LENGTH_SETTING.get(settings);
        this.ivLength = this.keyLength / 8;
        this.keyAlgorithm = (String)ENCRYPTION_KEY_ALGO_SETTING.get(settings);
        if (this.keyLength % 8 != 0) {
            throw new IllegalArgumentException("invalid key length [" + this.keyLength + "]. value must be a multiple of 8");
        }
        this.keyFile = CryptoService.resolveSystemKey(env);
        this.systemKey = CryptoService.readSystemKey(this.keyFile, (Boolean)SYSTEM_KEY_REQUIRED_SETTING.get(settings));
        this.randomKey = CryptoService.generateSecretKey(128);
        this.randomKeyBase64 = Base64.getUrlEncoder().encodeToString(this.randomKey.getEncoded());
        this.signingKey = CryptoService.createSigningKey(this.systemKey, this.randomKey);
        try {
            this.encryptionKey = CryptoService.encryptionKey(this.systemKey, this.keyLength, this.keyAlgorithm);
        }
        catch (NoSuchAlgorithmException nsae) {
            throw new ElasticsearchException("failed to start crypto service. could not load encryption key", (Throwable)nsae, new Object[0]);
        }
        if (this.systemKey != null) {
            this.logger.info("system key [{}] has been loaded", (Object)this.keyFile.toAbsolutePath());
        }
    }

    public static byte[] generateKey() {
        return CryptoService.generateSecretKey(1024).getEncoded();
    }

    static SecretKey generateSecretKey(int keyLength) {
        try {
            KeyGenerator generator = KeyGenerator.getInstance(KEY_ALGO);
            generator.init(keyLength);
            return generator.generateKey();
        }
        catch (NoSuchAlgorithmException e) {
            throw new ElasticsearchException("failed to generate key", (Throwable)e, new Object[0]);
        }
    }

    public static Path resolveSystemKey(Environment env) {
        return XPackPlugin.resolveConfigFile(env, FILE_NAME);
    }

    static SecretKey createSigningKey(@Nullable SecretKey systemKey, SecretKey randomKey) {
        assert (randomKey != null);
        if (systemKey != null) {
            return systemKey;
        }
        byte[] keyBytes = HmacSHA1HKDF.extractAndExpand(null, randomKey.getEncoded(), HKDF_APP_INFO, 128);
        assert (keyBytes.length * 8 == 1024);
        return new SecretKeySpec(keyBytes, KEY_ALGO);
    }

    private static SecretKey readSystemKey(Path file, boolean required) throws IOException {
        if (Files.exists(file, new LinkOption[0])) {
            byte[] bytes = Files.readAllBytes(file);
            return new SecretKeySpec(bytes, KEY_ALGO);
        }
        if (required) {
            throw new FileNotFoundException("[" + file + "] must be present with a valid key");
        }
        return null;
    }

    public String sign(String text, Version version) throws IOException {
        if (version.before(Version.V_5_4_0)) {
            String sigStr = CryptoService.signInternal(text, this.signingKey);
            return "$$" + sigStr.length() + "$$" + (this.systemKey == this.signingKey ? "" : this.randomKeyBase64) + "$$" + sigStr + text;
        }
        if (this.systemKey != null) {
            String sigStr = CryptoService.signInternal(text, this.systemKey);
            return "$$" + sigStr.length() + "$$$$" + sigStr + text;
        }
        return text;
    }

    public String unsignAndVerify(String signedText, Version version) {
        String text;
        String receivedSignature;
        String[] pieces;
        if (version == null ? (pieces = signedText.split("\\$\\$")).length == 4 && !pieces[2].equals("") : version.before(Version.V_5_4_0)) {
            return this.unsignAndVerifyLegacy(signedText);
        }
        if (this.systemKey == null) {
            if (this.isSigned(signedText)) {
                throw new IllegalArgumentException("tampered signed text");
            }
            return signedText;
        }
        pieces = signedText.split("\\$\\$");
        if (pieces.length != 4 || !pieces[0].equals("") || !pieces[2].equals("")) {
            this.logger.debug("received signed text [{}] with [{}] parts", (Object)signedText, (Object)pieces.length);
            throw new IllegalArgumentException("tampered signed text");
        }
        try {
            int length = Integer.parseInt(pieces[1]);
            receivedSignature = pieces[3].substring(0, length);
            text = pieces[3].substring(length);
        }
        catch (Exception e) {
            this.logger.error("error occurred while parsing signed text", (Throwable)e);
            throw new IllegalArgumentException("tampered signed text");
        }
        try {
            String sig = CryptoService.signInternal(text, this.systemKey);
            if (CharArrays.constantTimeEquals(sig, receivedSignature)) {
                return text;
            }
        }
        catch (Exception e) {
            this.logger.error("error occurred while verifying signed text", (Throwable)e);
            throw new IllegalStateException("error while verifying the signed text");
        }
        throw new IllegalArgumentException("tampered signed text");
    }

    private String unsignAndVerifyLegacy(String signedText) {
        SecretKey signingKey;
        String text;
        String receivedSignature;
        String base64RandomKey;
        if (!signedText.startsWith("$$") || signedText.length() < 2) {
            throw new IllegalArgumentException("tampered signed text");
        }
        String[] pieces = signedText.split("\\$\\$");
        if (pieces.length != 4 || !pieces[0].equals("")) {
            this.logger.debug("received signed text [{}] with [{}] parts", (Object)signedText, (Object)pieces.length);
            throw new IllegalArgumentException("tampered signed text");
        }
        try {
            int length = Integer.parseInt(pieces[1]);
            base64RandomKey = pieces[2];
            receivedSignature = pieces[3].substring(0, length);
            text = pieces[3].substring(length);
        }
        catch (Exception e) {
            this.logger.error("error occurred while parsing signed text", (Throwable)e);
            throw new IllegalArgumentException("tampered signed text");
        }
        if (base64RandomKey.isEmpty()) {
            if (this.systemKey == null) {
                this.logger.debug("received signed text without random key information and no system key is present");
                throw new IllegalArgumentException("tampered signed text");
            }
            signingKey = this.systemKey;
        } else {
            byte[] randomKeyBytes;
            if (this.systemKey != null) {
                this.logger.debug("received signed text with random key information but a system key is present");
                throw new IllegalArgumentException("tampered signed text");
            }
            try {
                randomKeyBytes = Base64.getUrlDecoder().decode(base64RandomKey);
            }
            catch (IllegalArgumentException e) {
                this.logger.error("error occurred while decoding key data", (Throwable)e);
                throw new IllegalStateException("error while verifying the signed text");
            }
            if (randomKeyBytes.length * 8 != 128) {
                this.logger.debug("incorrect random key data length. received [{}] bytes", (Object)randomKeyBytes.length);
                throw new IllegalArgumentException("tampered signed text");
            }
            SecretKeySpec randomKey = new SecretKeySpec(randomKeyBytes, KEY_ALGO);
            signingKey = CryptoService.createSigningKey(this.systemKey, randomKey);
        }
        try {
            String sig = CryptoService.signInternal(text, signingKey);
            if (CharArrays.constantTimeEquals(sig, receivedSignature)) {
                return text;
            }
        }
        catch (Exception e) {
            this.logger.error("error occurred while verifying signed text", (Throwable)e);
            throw new IllegalStateException("error while verifying the signed text");
        }
        throw new IllegalArgumentException("tampered signed text");
    }

    public boolean isSigned(String text) {
        return SIG_PATTERN.matcher(text).matches();
    }

    public char[] encrypt(char[] chars) {
        SecretKey key = this.encryptionKey;
        if (key == null) {
            this.logger.warn("encrypt called without a key, returning plain text. run syskeygen and copy same key to all nodes to enable encryption");
            return chars;
        }
        byte[] charBytes = CharArrays.toUtf8Bytes(chars);
        String base64 = Base64.getEncoder().encodeToString(this.encryptInternal(charBytes, key));
        return ENCRYPTED_TEXT_PREFIX.concat(base64).toCharArray();
    }

    public char[] decrypt(char[] chars) {
        byte[] bytes;
        if (this.encryptionKey == null) {
            return chars;
        }
        if (!this.isEncrypted(chars)) {
            return chars;
        }
        String encrypted = new String(chars, ENCRYPTED_TEXT_PREFIX.length(), chars.length - ENCRYPTED_TEXT_PREFIX.length());
        try {
            bytes = Base64.getDecoder().decode(encrypted);
        }
        catch (IllegalArgumentException e) {
            throw new ElasticsearchException("unable to decode encrypted data", (Throwable)e, new Object[0]);
        }
        byte[] decrypted = this.decryptInternal(bytes, this.encryptionKey);
        return CharArrays.utf8BytesToChars(decrypted);
    }

    public boolean isEncrypted(char[] chars) {
        return CharArrays.charsBeginsWith(ENCRYPTED_TEXT_PREFIX, chars);
    }

    public boolean isEncryptionEnabled() {
        return this.encryptionKey != null;
    }

    public boolean isSystemKeyPresent() {
        return this.systemKey != null;
    }

    private byte[] encryptInternal(byte[] bytes, SecretKey key) {
        byte[] iv = new byte[this.ivLength];
        this.secureRandom.nextBytes(iv);
        Cipher cipher = CryptoService.cipher(1, this.encryptionAlgorithm, key, iv);
        try {
            byte[] encrypted = cipher.doFinal(bytes);
            byte[] output = new byte[iv.length + encrypted.length];
            System.arraycopy(iv, 0, output, 0, iv.length);
            System.arraycopy(encrypted, 0, output, iv.length, encrypted.length);
            return output;
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new ElasticsearchException("error encrypting data", (Throwable)e, new Object[0]);
        }
    }

    private byte[] decryptInternal(byte[] bytes, SecretKey key) {
        if (bytes.length < this.ivLength) {
            this.logger.error("received data for decryption with size [{}] that is less than IV length [{}]", (Object)bytes.length, (Object)this.ivLength);
            throw new IllegalArgumentException("invalid data to decrypt");
        }
        byte[] iv = new byte[this.ivLength];
        System.arraycopy(bytes, 0, iv, 0, this.ivLength);
        byte[] data = new byte[bytes.length - this.ivLength];
        System.arraycopy(bytes, this.ivLength, data, 0, bytes.length - this.ivLength);
        Cipher cipher = CryptoService.cipher(2, this.encryptionAlgorithm, key, iv);
        try {
            return cipher.doFinal(data);
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new IllegalStateException("error decrypting data", e);
        }
    }

    static Mac createMac(SecretKey key) {
        try {
            Mac mac = HmacSHA1Provider.hmacSHA1();
            mac.init(key);
            return mac;
        }
        catch (Exception e) {
            throw new ElasticsearchException("could not initialize mac", (Throwable)e, new Object[0]);
        }
    }

    private static String signInternal(String text, SecretKey key) throws IOException {
        Mac mac = CryptoService.createMac(key);
        byte[] sig = mac.doFinal(text.getBytes(StandardCharsets.UTF_8));
        return Base64.getUrlEncoder().encodeToString(sig);
    }

    static Cipher cipher(int mode, String encryptionAlgorithm, SecretKey key, byte[] initializationVector) {
        try {
            Cipher cipher = Cipher.getInstance(encryptionAlgorithm);
            cipher.init(mode, (Key)key, new IvParameterSpec(initializationVector));
            return cipher;
        }
        catch (Exception e) {
            throw new ElasticsearchException("error creating cipher", (Throwable)e, new Object[0]);
        }
    }

    static SecretKey encryptionKey(SecretKey systemKey, int keyLength, String algorithm) throws NoSuchAlgorithmException {
        if (systemKey == null) {
            return null;
        }
        byte[] bytes = systemKey.getEncoded();
        if (bytes.length * 8 < keyLength) {
            throw new IllegalArgumentException("at least " + keyLength + " bits should be provided as key data");
        }
        MessageDigest messageDigest = MessageDigest.getInstance("SHA-256");
        byte[] digest = messageDigest.digest(bytes);
        assert (digest.length == 32);
        if (digest.length * 8 < keyLength) {
            throw new IllegalArgumentException("requested key length is too large");
        }
        byte[] truncatedDigest = Arrays.copyOfRange(digest, 0, keyLength / 8);
        return new SecretKeySpec(truncatedDigest, algorithm);
    }

    public static void addSettings(List<Setting<?>> settings) {
        settings.add(ENCRYPTION_KEY_LENGTH_SETTING);
        settings.add(ENCRYPTION_KEY_ALGO_SETTING);
        settings.add(ENCRYPTION_ALGO_SETTING);
        settings.add(SYSTEM_KEY_REQUIRED_SETTING);
    }

    private static class HmacSHA1HKDF {
        private static final int HMAC_SHA1_BYTE_LENGTH = 20;
        private static final String HMAC_SHA1_ALGORITHM = "HmacSHA1";

        private HmacSHA1HKDF() {
        }

        static byte[] extractAndExpand(@Nullable SecretKey salt, byte[] ikm, @Nullable byte[] info, int outputLength) {
            Objects.requireNonNull(ikm, "the input keying material must not be null");
            if (outputLength < 1) {
                throw new IllegalArgumentException("output length must be positive int >= 1");
            }
            if (outputLength > 5100) {
                throw new IllegalArgumentException("output length must be <= 255*20");
            }
            if (salt == null) {
                salt = new SecretKeySpec(new byte[20], "HmacSHA1");
            }
            if (info == null) {
                info = new byte[]{};
            }
            Mac mac = CryptoService.createMac(salt);
            byte[] keyBytes = mac.doFinal(ikm);
            SecretKeySpec pseudoRandomKey = new SecretKeySpec(keyBytes, "HmacSHA1");
            int n = outputLength % 20 == 0 ? outputLength / 20 : outputLength / 20 + 1;
            byte[] hashRound = new byte[]{};
            ByteBuffer generatedBytes = ByteBuffer.allocate(Math.multiplyExact(n, 20));
            try {
                mac.init(pseudoRandomKey);
            }
            catch (InvalidKeyException e) {
                throw new ElasticsearchException("failed to initialize the mac", (Throwable)e, new Object[0]);
            }
            for (int roundNum = 1; roundNum <= n; ++roundNum) {
                mac.reset();
                mac.update(hashRound);
                mac.update(info);
                mac.update((byte)roundNum);
                hashRound = mac.doFinal();
                generatedBytes.put(hashRound);
            }
            byte[] result = new byte[outputLength];
            generatedBytes.rewind();
            generatedBytes.get(result, 0, outputLength);
            return result;
        }
    }

    private static class HmacSHA1Provider {
        private static final ThreadLocal<Mac> MAC = ThreadLocal.withInitial(() -> {
            try {
                return Mac.getInstance(CryptoService.HMAC_ALGO);
            }
            catch (NoSuchAlgorithmException e) {
                throw new IllegalStateException("could not create Mac instance with algorithm [HmacSHA1]", e);
            }
        });

        private HmacSHA1Provider() {
        }

        private static Mac hmacSHA1() {
            Mac instance = MAC.get();
            instance.reset();
            return instance;
        }
    }
}

