/*
 *  Copyright Terracotta, Inc.
 *  Copyright IBM Corp. 2024, 2025
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package com.tc.management;

import com.tc.management.beans.L2MBeanNames;
import com.tc.util.Assert;
import com.tc.util.UUID;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.Collections;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.management.MBeanServerConnection;
import javax.management.MBeanServerInvocationHandler;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.QueryExp;

public abstract class TerracottaManagement {

  private static final ManagementResources MANAGEMENT_RESOURCES = new ManagementResources();

  public static class Type {

    private static final Map<String, Type> typesByName      = Collections.synchronizedMap(new HashMap<String, Type>());
    public static final Type Client        = new Type(MANAGEMENT_RESOURCES.getDsoClientType());
    public static final Type Server           = new Type(MANAGEMENT_RESOURCES.getTerracottaServerType());

    private final String     type;

    private Type(String type) {
      this.type = type;
      typesByName.put(type, this);
    }

    @Override
    public String toString() {
      return type;
    }

    static Type getType(String name) {
      return typesByName.get(name);
    }
  }

  public static class Subsystem {

    private static final Map<String, Subsystem>      subsystemByName  = Collections.synchronizedMap(new HashMap<String, Subsystem>());
    public static final Subsystem Logging          = new Subsystem(MANAGEMENT_RESOURCES.getLoggingSubsystem());
    public static final Subsystem Statistics       = new Subsystem(MANAGEMENT_RESOURCES.getStatisticsSubsystem());
    public static final Subsystem None             = new Subsystem(MANAGEMENT_RESOURCES.getNoneSubsystem());

    private final String          subsystem;

    private Subsystem(String subsystem) {
      this.subsystem = subsystem;
      subsystemByName.put(subsystem, this);
    }

    @Override
    public String toString() {
      return subsystem;
    }

    static Subsystem getSubsystem(String name) {
      return subsystemByName.get(name);
    }
  }

  public static interface MBeanKeys {
    public static final String TYPE            = "type";
    public static final String MBEAN_NODE      = "node";
    public static final String MBEAN_NODE_NAME = "node-name";
    public static final String SUBSYSTEM       = "subsystem";
    public static final String NAME            = "name";
  }

  public enum MBeanDomain {
    PUBLIC(MANAGEMENT_RESOURCES.getPublicMBeanDomain());

    private final String value;

    private MBeanDomain(String value) {
      this.value = value;
    }

    @Override
    public String toString() {
      return value;
    }
  }

  private static final String COMMA           = ",";
  private static final String COLON           = ":";
  private static final String EQUALS          = "=";
  private static final String UNDERSCORE      = "_";

  private static final String NODE_PREFIX_KEY = "clients";
  private static final String NODE_PREFIX     = NODE_PREFIX_KEY + EQUALS + "Clients";

  public static ObjectName createObjectName(Type type, String uiFriendlyName,
                                            MBeanDomain domain) throws MalformedObjectNameException {
    Assert.assertNotNull(domain);
    final StringBuffer objName = new StringBuffer(domain.toString());
    objName.append(COLON);
    objName.append(MBeanKeys.NAME).append(EQUALS).append(uiFriendlyName);
    if (type != null) {
      objName.append(COMMA).append(MBeanKeys.TYPE).append(EQUALS).append(type);
    }
    try {
      return new ObjectName(objName.toString());
    } catch (MalformedObjectNameException mal) {
      throw new MalformedObjectNameException(objName.toString() + " " + mal.getMessage());
    }
  }

  private static void addNodeInfo(StringBuffer objName, InetSocketAddress addr) {
    objName.append(COMMA).append(MBeanKeys.MBEAN_NODE).append(EQUALS).append(buildNodeId(addr));
  }

  public static String buildNodeId(InetSocketAddress addr) {
    String remoteHost = addr.getAddress().getCanonicalHostName();
    int remotePort = addr.getPort();
    return remoteHost + UNDERSCORE + remotePort;
  }

  public static ObjectName addNodeInfo(ObjectName objName, InetSocketAddress addr)
      throws MalformedObjectNameException {
    if (objName.getKeyProperty(MBeanKeys.MBEAN_NODE) != null) {
      Hashtable<String, String> kpl = objName.getKeyPropertyList();
      kpl.remove(MBeanKeys.MBEAN_NODE);
      objName = ObjectName.getInstance(objName.getDomain(), kpl);
    }
    StringBuffer sb = new StringBuffer(objName.getCanonicalName());
    if (objName.getKeyProperty(NODE_PREFIX_KEY) == null) {
      sb.append(COMMA).append(NODE_PREFIX);
    }
    addNodeInfo(sb, addr);
    return new ObjectName(sb.toString());
  }

  private static void addNodeInfo(StringBuffer objName, UUID id) {
    objName.append(COMMA).append(MBeanKeys.MBEAN_NODE).append(EQUALS).append(id);
  }

  public static ObjectName addNodeInfo(ObjectName objName, UUID id) throws MalformedObjectNameException {
    if (objName.getKeyProperty(MBeanKeys.MBEAN_NODE) != null) {
      Hashtable<String, String> kpl = objName.getKeyPropertyList();
      kpl.remove(MBeanKeys.MBEAN_NODE);
      objName = ObjectName.getInstance(objName.getDomain(), kpl);
    }
    StringBuffer sb = new StringBuffer(objName.getCanonicalName());
    if (objName.getKeyProperty(NODE_PREFIX_KEY) == null) {
      sb.append(COMMA).append(NODE_PREFIX);
    }
    addNodeInfo(sb, id);
    return new ObjectName(sb.toString());
  }

  public abstract Object findMBean(ObjectName objectName, Class<?> mBeanInterface) throws Exception;

  public static final Object findMBean(ObjectName objectName, Class<?> mBeanInterface,
                                       MBeanServerConnection mBeanServer) throws IOException {
    final Set<ObjectInstance> matchingBeans = mBeanServer.queryMBeans(objectName, null);
    final Iterator<ObjectInstance> beanPos = matchingBeans.iterator();
    if (beanPos.hasNext()) { return MBeanServerInvocationHandler.newProxyInstance(mBeanServer, objectName,
                                                                                  mBeanInterface, false); }
    return null;
  }

  public static final QueryExp matchAllTerracottaMBeans(UUID id) {
    try {
      QueryExp query = new ObjectName(MBeanDomain.PUBLIC + ":*,node=" + id);
      return query;
    } catch (MalformedObjectNameException e) {
      throw new RuntimeException(e);
    }
  }

  public static final String quoteIfNecessary(String objectNamePart) {
    if (objectNamePart.matches("[,=:*?\"']")) { return ObjectName.quote(objectNamePart); }
    return objectNamePart;
  }

  public static final Set<ObjectName> getAllL2DumperMBeans(MBeanServerConnection mbs) throws MalformedObjectNameException,
      NullPointerException, IOException {
    return mbs.queryNames(new ObjectName(L2MBeanNames.DUMPER.getCanonicalName() + ",*"), null);
  }
}
