/**
 * Copyright (c) 2015 itemis AG (http://www.itemis.eu) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.xtext.web.server.contentassist;

import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.ExecutorService;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.ide.editor.contentassist.ContentAssistContext;
import org.eclipse.xtext.ide.editor.contentassist.ContentAssistEntry;
import org.eclipse.xtext.ide.editor.contentassist.IIdeContentProposalAcceptor;
import org.eclipse.xtext.ide.editor.contentassist.IdeContentProposalProvider;
import org.eclipse.xtext.ide.editor.contentassist.antlr.ContentAssistContextFactory;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.util.ITextRegion;
import org.eclipse.xtext.util.concurrent.CancelableUnitOfWork;
import org.eclipse.xtext.web.server.InvalidRequestException;
import org.eclipse.xtext.web.server.contentassist.ContentAssistResult;
import org.eclipse.xtext.web.server.model.IXtextWebDocument;
import org.eclipse.xtext.web.server.model.XtextWebDocumentAccess;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Service class for content assist proposals.
 */
@Singleton
@SuppressWarnings("all")
public class ContentAssistService {
  public static final int DEFAULT_PROPOSALS_LIMIT = 1000;
  
  @Inject
  private Provider<ContentAssistContextFactory> contextFactoryProvider;
  
  @Inject
  private ExecutorService executorService;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  @Inject
  private IdeContentProposalProvider proposalProvider;
  
  /**
   * Create content assist proposals at the given caret offset. This document read operation
   * is scheduled with higher priority, so currently running operations may be canceled.
   * The document processing is rescheduled as background work afterwards.
   */
  public ContentAssistResult createProposals(final XtextWebDocumentAccess document, final ITextRegion selection, final int caretOffset, final int proposalsLimit) throws InvalidRequestException {
    final String[] stateIdWrapper = new String[1];
    final CancelableUnitOfWork<ContentAssistContext[], IXtextWebDocument> _function = new CancelableUnitOfWork<ContentAssistContext[], IXtextWebDocument>() {
      @Override
      public ContentAssistContext[] exec(final IXtextWebDocument it, final CancelIndicator cancelIndicator) throws Exception {
        ContentAssistContext[] _xblockexpression = null;
        {
          stateIdWrapper[0] = it.getStateId();
          _xblockexpression = ContentAssistService.this.getContexts(it, selection, caretOffset);
        }
        return _xblockexpression;
      }
    };
    final ContentAssistContext[] contexts = document.<ContentAssistContext[]>priorityReadOnly(_function);
    return this.createProposals(((List<ContentAssistContext>)Conversions.doWrapArray(contexts)), stateIdWrapper[0], proposalsLimit);
  }
  
  /**
   * Apply a text update and then create content assist proposals. This document read operation
   * is scheduled with higher priority, so currently running operations may be canceled.
   * The document processing is rescheduled as background work afterwards.
   */
  public ContentAssistResult createProposalsWithUpdate(final XtextWebDocumentAccess document, final String deltaText, final int deltaOffset, final int deltaReplaceLength, final ITextRegion selection, final int caretOffset, final int proposalsLimit) throws InvalidRequestException {
    final String[] stateIdWrapper = new String[1];
    final CancelableUnitOfWork<ContentAssistContext[], IXtextWebDocument> _function = new CancelableUnitOfWork<ContentAssistContext[], IXtextWebDocument>() {
      @Override
      public ContentAssistContext[] exec(final IXtextWebDocument it, final CancelIndicator cancelIndicator) throws Exception {
        ContentAssistContext[] _xblockexpression = null;
        {
          it.setDirty(true);
          it.createNewStateId();
          stateIdWrapper[0] = it.getStateId();
          it.updateText(deltaText, deltaOffset, deltaReplaceLength);
          _xblockexpression = ContentAssistService.this.getContexts(it, selection, caretOffset);
        }
        return _xblockexpression;
      }
    };
    final ContentAssistContext[] contexts = document.<ContentAssistContext[]>modify(_function);
    return this.createProposals(((List<ContentAssistContext>)Conversions.doWrapArray(contexts)), stateIdWrapper[0], proposalsLimit);
  }
  
  public ContentAssistContext[] getContexts(final IXtextWebDocument document, final ITextRegion selection, final int caretOffset) {
    ContentAssistContext[] _xblockexpression = null;
    {
      int _length = document.getText().length();
      boolean _greaterThan = (caretOffset > _length);
      if (_greaterThan) {
        return new ContentAssistContext[] {};
      }
      ContentAssistContextFactory _get = this.contextFactoryProvider.get();
      final Procedure1<ContentAssistContextFactory> _function = (ContentAssistContextFactory it) -> {
        it.setPool(this.executorService);
      };
      final ContentAssistContextFactory contextFactory = ObjectExtensions.<ContentAssistContextFactory>operator_doubleArrow(_get, _function);
      _xblockexpression = contextFactory.create(document.getText(), selection, caretOffset, document.getResource());
    }
    return _xblockexpression;
  }
  
  /**
   * Invoke the proposal provider and put the results into a {@link ContentAssistResult} object.
   */
  protected ContentAssistResult createProposals(final List<ContentAssistContext> contexts, final String stateId, final int proposalsLimit) {
    final ContentAssistResult result = new ContentAssistResult(stateId);
    boolean _isEmpty = contexts.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final HashSet<Pair<Integer, ContentAssistEntry>> proposals = new HashSet<Pair<Integer, ContentAssistEntry>>();
      final IIdeContentProposalAcceptor acceptor = new IIdeContentProposalAcceptor() {
        @Override
        public void accept(final ContentAssistEntry entry, final int priority) {
          if ((entry != null)) {
            String _proposal = entry.getProposal();
            boolean _tripleEquals = (_proposal == null);
            if (_tripleEquals) {
              throw new IllegalArgumentException("proposal must not be null.");
            }
            Pair<Integer, ContentAssistEntry> _mappedTo = Pair.<Integer, ContentAssistEntry>of(Integer.valueOf(priority), entry);
            proposals.add(_mappedTo);
          }
        }
        
        @Override
        public boolean canAcceptMoreProposals() {
          int _size = proposals.size();
          return (_size < proposalsLimit);
        }
      };
      this.proposalProvider.createProposals(contexts, acceptor);
      final Comparator<Pair<Integer, ContentAssistEntry>> _function = (Pair<Integer, ContentAssistEntry> p1, Pair<Integer, ContentAssistEntry> p2) -> {
        final int prioResult = p2.getKey().compareTo(p1.getKey());
        if ((prioResult != 0)) {
          return prioResult;
        }
        String _elvis = null;
        String _label = p1.getValue().getLabel();
        if (_label != null) {
          _elvis = _label;
        } else {
          String _proposal = p1.getValue().getProposal();
          _elvis = _proposal;
        }
        final String s1 = _elvis;
        String _elvis_1 = null;
        String _label_1 = p2.getValue().getLabel();
        if (_label_1 != null) {
          _elvis_1 = _label_1;
        } else {
          String _proposal_1 = p2.getValue().getProposal();
          _elvis_1 = _proposal_1;
        }
        final String s2 = _elvis_1;
        return s1.compareTo(s2);
      };
      final Function1<Pair<Integer, ContentAssistEntry>, ContentAssistEntry> _function_1 = (Pair<Integer, ContentAssistEntry> it) -> {
        return it.getValue();
      };
      result.getEntries().addAll(ListExtensions.<Pair<Integer, ContentAssistEntry>, ContentAssistEntry>map(IterableExtensions.<Pair<Integer, ContentAssistEntry>>sortWith(proposals, _function), _function_1));
    }
    return result;
  }
  
  @Pure
  public IdeContentProposalProvider getProposalProvider() {
    return this.proposalProvider;
  }
}
