/*******************************************************************************
 *  Copyright (c) 2006, 2007 Oracle. All rights reserved. This
 *  program and the accompanying materials are made available under the terms of
 *  the Eclipse Public License v1.0 which accompanies this distribution, and is
 *  available at http://www.eclipse.org/legal/epl-v10.html
 *  
 *  Contributors: Oracle. - initial API and implementation
 *******************************************************************************/
package org.eclipse.jpt.ui.internal.xml.structure;

import java.util.ArrayList;
import java.util.Collection;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.edit.provider.ChangeNotifier;
import org.eclipse.emf.edit.provider.ComposeableAdapterFactory;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.IChangeNotifier;
import org.eclipse.emf.edit.provider.IEditingDomainItemProvider;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.INotifyChangedListener;
import org.eclipse.emf.edit.provider.IStructuredItemContentProvider;
import org.eclipse.emf.edit.provider.ITreeItemContentProvider;
import org.eclipse.jpt.core.internal.content.orm.util.OrmAdapterFactory;
import org.eclipse.ui.services.IDisposable;

/**
 * This is the factory that is used to provide the interfaces needed to support Viewers.
 * The adapters generated by this factory convert EMF adapter notifications into calls to {@link #fireNotifyChanged fireNotifyChanged}.
 * The adapters also support Eclipse property sheets.
 * Note that most of the adapters are shared among multiple instances.
 */
public class JpaCoreXmlItemProviderAdapterFactory 
	extends OrmAdapterFactory 
	implements ComposeableAdapterFactory,
		IChangeNotifier, 
		IDisposable 
{
	/**
	 * This keeps track of the root adapter factory that delegates to this adapter factory.
	 */
	protected ComposedAdapterFactory parentAdapterFactory;

	/**
	 * This is used to implement {@link org.eclipse.emf.edit.provider.IChangeNotifier}.
	 */
	protected IChangeNotifier changeNotifier = new ChangeNotifier();

	/**
	 * This keeps track of all the supported types checked by {@link #isFactoryForType isFactoryForType}.
	 */
	protected Collection supportedTypes = new ArrayList();

	public JpaCoreXmlItemProviderAdapterFactory() {
		supportedTypes.add(IEditingDomainItemProvider.class);
		supportedTypes.add(IStructuredItemContentProvider.class);
		supportedTypes.add(ITreeItemContentProvider.class);
		supportedTypes.add(IItemLabelProvider.class);
	}
	
	
	protected XmlRootContentNodeItemProvider xmlRootContentNodeItemProvider;
	
	public Adapter createXmlRootContentNodeAdapter() {
		if (xmlRootContentNodeItemProvider == null) {
			xmlRootContentNodeItemProvider = new XmlRootContentNodeItemProvider(this);
		}

		return xmlRootContentNodeItemProvider;
	}
	
	protected EntityMappingsItemProvider entityMappingsItemProvider;

	/**
	 * This creates an adapter for a {@link org.eclipse.jpt.core.internal.content.orm.EntityMappingsImpl}.
	 */
	public Adapter createEntityMappingsAdapter() {
		if (entityMappingsItemProvider == null) {
			entityMappingsItemProvider = new EntityMappingsItemProvider(this);
		}

		return entityMappingsItemProvider;
	}
	
	/**
	 * This keeps track of the one adapter used for all {@link org.eclipse.jpt.core.internal.content.orm.XmlPersistentType} instances.
	 */
	protected XmlPersistentTypeItemProvider xmlPersistentTypeItemProvider;
	
	/**
	 * This creates an adapter for a {@link org.eclipse.jpt.core.internal.content.orm.XmlPersistentType}.
	 */
	public Adapter createXmlPersistentTypeAdapter() {
		if (xmlPersistentTypeItemProvider == null) {
			xmlPersistentTypeItemProvider = new XmlPersistentTypeItemProvider(
					this);
		}
		
		return xmlPersistentTypeItemProvider;
	}
	
	/**
	 * This keeps track of the one adapter used for all {@link org.eclipse.jpt.core.internal.content.orm.XmlPersistentAttribute} instances.
	 */
	protected XmlPersistentAttributeItemProvider xmlPersistentAttributeItemProvider;
	
	/**
	 * This creates an adapter for a {@link org.eclipse.jpt.core.internal.content.orm.XmlPersistentAttribute}.
	 */
	public Adapter createXmlPersistentAttributeAdapter() {
		if (xmlPersistentAttributeItemProvider == null) {
			xmlPersistentAttributeItemProvider = new XmlPersistentAttributeItemProvider(
					this);
		}
		
		return xmlPersistentAttributeItemProvider;
	}
	
	/**
	 * This returns the root adapter factory that contains this factory.
	 */
	public ComposeableAdapterFactory getRootAdapterFactory() {
		return parentAdapterFactory == null ? this : parentAdapterFactory
				.getRootAdapterFactory();
	}
	
	/**
	 * This sets the composed adapter factory that contains this factory.
	 */
	public void setParentAdapterFactory(
			ComposedAdapterFactory parentAdapterFactory) {
		this.parentAdapterFactory = parentAdapterFactory;
	}
	
	public boolean isFactoryForType(Object type) {
		return supportedTypes.contains(type) || super.isFactoryForType(type);
	}
	
	/**
	 * This implementation substitutes the factory itself as the key for the adapter.
	 */
	public Adapter adapt(Notifier notifier, Object type) {
		return super.adapt(notifier, this);
	}
	
	public Object adapt(Object object, Object type) {
		if (isFactoryForType(type)) {
			Object adapter = super.adapt(object, type);
			if (!(type instanceof Class)
					|| (((Class) type).isInstance(adapter))) {
				return adapter;
			}
		}
		
		return null;
	}
	
	/**
	 * This adds a listener.
	 */
	public void addListener(INotifyChangedListener notifyChangedListener) {
		changeNotifier.addListener(notifyChangedListener);
	}
	
	/**
	 * This removes a listener.
	 */
	public void removeListener(INotifyChangedListener notifyChangedListener) {
		changeNotifier.removeListener(notifyChangedListener);
	}
	
	/**
	 * This delegates to {@link #changeNotifier} and to {@link #parentAdapterFactory}.
	 */
	public void fireNotifyChanged(Notification notification) {
		changeNotifier.fireNotifyChanged(notification);
		
		if (parentAdapterFactory != null) {
			parentAdapterFactory.fireNotifyChanged(notification);
		}
	}
	
	/**
	 * This disposes all of the item providers created by this factory. 
	 */
	public void dispose() {
		if (xmlRootContentNodeItemProvider != null)
			xmlRootContentNodeItemProvider.dispose();
		if (entityMappingsItemProvider != null)
			entityMappingsItemProvider.dispose();
		if (xmlPersistentTypeItemProvider != null)
			xmlPersistentTypeItemProvider.dispose();
		if (xmlPersistentAttributeItemProvider != null)
			xmlPersistentAttributeItemProvider.dispose();
	}
}
