/*******************************************************************************
 * Copyright (c) 2005 Sybase, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Sybase, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.sqltools.result.internal.ui.view;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.datatools.sqltools.result.IResultSetObject;
import org.eclipse.datatools.sqltools.result.IResultSetRow;
import org.eclipse.datatools.sqltools.result.Parameter;
import org.eclipse.datatools.sqltools.result.ResultSetObject;
import org.eclipse.datatools.sqltools.result.XMLResultSetObject;
import org.eclipse.datatools.sqltools.result.internal.ui.export.actions.ExportAllResultSetsAction;
import org.eclipse.datatools.sqltools.result.internal.ui.export.actions.ExportResultSetAction;
import org.eclipse.datatools.sqltools.result.internal.ui.export.actions.PrintResultSetAction;
import org.eclipse.datatools.sqltools.result.internal.ui.export.actions.SaveAllResultSetsAction;
import org.eclipse.datatools.sqltools.result.internal.ui.export.actions.SaveResultSetAction;
import org.eclipse.datatools.sqltools.result.internal.ui.viewer.ResultSetViewer;
import org.eclipse.datatools.sqltools.result.internal.utils.Messages;
import org.eclipse.datatools.sqltools.result.internal.utils.StatusTextProvider;
import org.eclipse.datatools.sqltools.result.internal.utils.UIUtil;
import org.eclipse.datatools.sqltools.result.model.IResultInstance;
import org.eclipse.datatools.sqltools.result.model.ResultItem;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;

/**
 * This UI component displays the execution result on multiple tabs in grid mode(using tables to display result sets).
 * 
 * @author Dafan Yang
 *  
 */
public class MultipleTabsGridSection extends MultipleTabsModeSection
{
    private Text             _statusView;
    private Text             _messageView;
    private static final int PARAM_NAME      = 0;
    private static final int PARAM_TYPE      = 1;
    private static final int PARAM_DATA_TYPE = 2;
    private static final int PARAM_VALUE     = 3;
    private static final int PARAM_VALUE_OUT = 4;
    private Table            _paramTable;
    
    public MultipleTabsGridSection(Composite composite, ResultsView view)
    {
        super(composite, view);
    }

    public MultipleTabsGridSection(Composite composite, IResultInstance instance, ResultsView view)
    {
        super(composite, instance, view);
    }

    protected Control createStatusItem(CTabFolder ctf)
    {
        _statusView = new Text(ctf, SWT.MULTI | SWT.V_SCROLL | SWT.H_SCROLL);
        _statusView.setEditable(false);
        return _statusView;
    }

    protected Control createMessageItem(CTabFolder ctf)
    {
        _messageView = new Text(ctf, SWT.MULTI | SWT.V_SCROLL | SWT.H_SCROLL);
        _messageView.setEditable(false);
        return _messageView;
    }
    
    public void showStatusDetail(IResultInstance instance)
    {
        if (instance == null)
        {
            _statusView.setText(""); //$NON-NLS-1$
        }
        else
        {
            String text = StatusTextProvider.getStatusText(instance);
            _statusView.setText(text);
        }
    }
    
    protected void showMessageDetail(IResultInstance instance)
    {
        if(_splitMessages)
        {
            // will display messages in multiple tabs
            return; 
        }
        if (instance == null)
        {
            _messageView.setText(""); //$NON-NLS-1$
        }
        else
        {
            int count = instance.getItemCount();
            StringBuffer sb = new StringBuffer(""); //$NON-NLS-1$
            for (int i = 0; i < count; i++)
            {
                ResultItem item = instance.getItem(i);
                if(item.getResultType() == ResultItem.PLAIN_TEXT)
                {
                    sb.append(item.getResultObject());
                }
            }
            _messageView.setText(sb.toString());
        }
    }
    
    protected void showParameterDetail(IResultInstance instance)
    {
        if(instance.getParameters() == null)
        {
            return;
        }
        Iterator iter = instance.getParameters().iterator();
        ArrayList newList = new ArrayList();
        while(iter.hasNext())
        {
            // only Parameter type is accepted
            Object obj = iter.next();
            if(obj != null && obj instanceof Parameter)
            {
                newList.add(obj);
            }
        }        
        appendAndShowParameters(newList);
    }

    protected void createViewerForItem(Composite composite, ResultItem item)
    {
        int resultType = item.getResultType();
        switch (resultType)
        {
            case ResultItem.PLAIN_TEXT:
                Text message = new Text(composite, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL);
                message.setLayoutData(new GridData(GridData.FILL_BOTH));
                message.setEditable(false);
                message.setText((String) item.getResultObject());
                break;
            case ResultItem.UPDATE_COUNT:

                break;
            case ResultItem.RESULT_SET:
                IResultSetObject result = (IResultSetObject) item.getResultObject();

                //treat result set object and xml result
                if(result instanceof XMLResultSetObject)
                {
                    createTextforResultSet(composite, result);
                }
                if(result instanceof ResultSetObject)
                {
                    createTableViewerForResultSet(composite, result);
                }
                else
                {
                    // to be extended
                }
                break;
            default:
                break;
        }

    }

    protected void appendStatusView(String text)
    {
        _statusView.append(text);
    }

    protected void appendMessageView(String text)
    {
        _messageView.append(text);
    }
    
    /**
     * Creates table viewer for result set object
     * 
     * @param composite the parent composite
     * @param result the result set object
     */
    protected void createTableViewerForResultSet(Composite composite, IResultSetObject result)
    {
        new ResultSetViewer(composite, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL
            | SWT.FULL_SELECTION, _resultInstance, result, _displayRowNumber);

        Label label = new Label(composite, SWT.NONE);

        GridData labelGridData = new GridData();
        labelGridData.grabExcessHorizontalSpace = true;
        labelGridData.grabExcessVerticalSpace = false;
        labelGridData.heightHint = UIUtil.convertHeightInCharsToPixels(1, _parent);
        label.setLayoutData(labelGridData);

        int totalRowCount = result.getTotalRowCount();
        int rowCount = result.getRowCount();
        if (totalRowCount == rowCount)
        {
            label.setText(NLS.bind(Messages.ResultSection_resultset_tooltip1, new Object[]
			{
			    String.valueOf(rowCount)
			})); //$NON-NLS-1$
        }
        else
        {
            label.setText(NLS.bind(Messages.ResultSection_resultset_tooltip, new Object[]
			{
			    String.valueOf(totalRowCount), String //$NON-NLS-1$
				                .valueOf(rowCount)
			}));
        }
    }

    protected void createTextforResultSet(Composite composite, IResultSetObject result)
    {
        Text txt = new Text(composite, SWT.MULTI | SWT.READ_ONLY | SWT.V_SCROLL | SWT.H_SCROLL);
        txt.setLayoutData(new GridData(GridData.FILL_BOTH));
        StringBuffer buf = new StringBuffer();

        Iterator iter = result.getDisplayRecords();
        while (iter != null && iter.hasNext())
        {
            IResultSetRow row = (IResultSetRow) iter.next();
            // in fact, there is only one column
            for (int i = 0; i < row.getData().length; i++)
            {
                buf.append(row.getData(i));
            }
        }
        String s = buf.toString();
        txt.setText(s);        

        //Register context menu for save or export result set
        MenuManager mgr = new MenuManager();
        
        MenuManager saveMgr = new MenuManager(Messages.Save_name); 
        saveMgr.add(new SaveResultSetAction(txt.getShell(), result));
        saveMgr.add(new SaveAllResultSetsAction(txt.getShell(), _resultInstance));
        
        MenuManager exportMgr = new MenuManager(Messages.Export_name); 
        exportMgr.add(new ExportResultSetAction(txt.getShell(), result));
        exportMgr.add(new ExportAllResultSetsAction(txt.getShell(), _resultInstance));
        
        MenuManager printMgr = new MenuManager(Messages.Print_name); 
        printMgr.add(new PrintResultSetAction(result, _parent));
        printMgr.add(new PrintResultSetAction(_resultInstance, _parent));
        
        mgr.add(saveMgr);
        mgr.add(exportMgr);
        mgr.add(printMgr);  
        
        Menu menu = mgr.createContextMenu(txt);
        txt.setMenu(menu);
    }

    public void onInstanceReseted()
    {
        super.onInstanceReseted();
        _statusView.setText(""); //$NON-NLS-1$
    }
    
    private void createTableForParameters(Composite comp)
    {
        TableViewer paramViewer = new TableViewer(comp, SWT.V_SCROLL | SWT.FULL_SELECTION);
        _paramTable = paramViewer.getTable();
        
        _paramTable.setLinesVisible(true);
        _paramTable.setHeaderVisible(true);
        _paramTable.setLayoutData(new GridData(GridData.FILL_BOTH));
        
        TableColumn nameColumn = new TableColumn(_paramTable, SWT.NONE);
        nameColumn.setText(Messages.MultipleTabsGridSection_parameter_name); 
        
        TableColumn typeColumn = new TableColumn(_paramTable, SWT.NONE);
        typeColumn.setText(Messages.MultipleTabsGridSection_parameter_type); 
        
        TableColumn dataTypeColumn = new TableColumn(_paramTable, SWT.NONE);
        dataTypeColumn.setText(Messages.MultipleTabsGridSection_parameter_datatype); 
        
        TableColumn valueColumn = new TableColumn(_paramTable, SWT.NONE);
        valueColumn.setText(Messages.MultipleTabsGridSection_value); 
        
        TableColumn outValueColumn = new TableColumn(_paramTable, SWT.NONE);
        outValueColumn.setText(Messages.MultipleTabsGridSection_value_out);
        
        int defaultWidth = 0;

        int columnCount = _paramTable.getColumnCount();
        for (int i=0; i < columnCount; i++)
        {
            TableColumn column = _paramTable.getColumn(i);
            column.pack();
            defaultWidth = defaultWidth + column.getWidth() + _paramTable.getGridLineWidth();
        }

        int moreWidth = comp.getParent().getBounds().width - 2 - defaultWidth;
        if (moreWidth > 0)
        {
            for (int i=0; i < columnCount; i++)
            {
                TableColumn col = _paramTable.getColumn(i);
                col.setWidth(col.getWidth() + moreWidth / columnCount);
            }
        }
        _paramTable.pack();
    }

    protected void appendAndShowParameters(List params)
    {
        if(_paramsItem == null)
        {
            _paramsItem = new CTabItem(_tabFolder, SWT.NONE, _numberStaticTab);
            _paramsItem.setText(Messages.MultipleTabsGridSection_parameter); 
            _numberStaticTab++;
            _isParamShown = true;
            _paramTabNumber = _numberStaticTab - 1;
            createTableForParameters(_tabFolder);
            _paramsItem.setControl(_paramTable);
        }
        // overwrite the orginal parameters
        fillDataIntoParamsTable(params);
    }
    
    private void fillDataIntoParamsTable(List params)
    {
        _paramTable.removeAll();
        Iterator iter = params.iterator();
        while(iter.hasNext())
        {
            Parameter param = (Parameter)iter.next();
            TableItem item = new TableItem(_paramTable, SWT.NONE);
            item.setText(PARAM_NAME, param.getParamName());
            item.setText(PARAM_TYPE, param.getParamType());
            item.setText(PARAM_DATA_TYPE, param.getParamDataType());
            item.setText(PARAM_VALUE, param.getParamValue());
            item.setText(PARAM_VALUE_OUT, param.getParamOutValue());
        }
    }
}
