/*******************************************************************************
 * Copyright (c) 2004, 2007 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  Actuate Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.datatools.enablement.oda.xml.util;

/**
 * 
 */

public class XPathParserUtil
{

	// XPath String
	public static final String SLASH = "/";
	public static final String DOUBLE_SLASH = "//";
	public static final String EMPTY_STRING = "";
	public static final String WILDCARD_PREDICATE_NUMBER = "[\\d+]";

	// RE String
	public static final String RE_XPATH_WILDCARD = "\\Q*\\E";
	public static final String RE_OPENING_BRACKET = "\\Q[\\E";
	public static final String RE_CLOSING_BRACKET = "\\Q]\\E";
	public static final String RE_SQUARE_BRACKETS = "\\Q[\\E.+\\Q]\\E";
	public static final String RE_NODETEST_WITH_PREDICATE_NUM_AND_OPTIONAL_ATTR = ".+\\Q[\\E\\d+\\Q]\\E.*";
	public static final String RE_WITH_PREDICATE_ONLY = "\\Q[\\E.*\\Q]\\E";//* is used in case of []
	public static final String RE_PREDICATE_WILDCARD = "\\Q[\\E\\*\\Q]\\E";// [*]

	// REPL String that will be casted to regular expressions after the
	// replacement
	public static final String REPL_XPATH_WILDCARD = ".+";
	public static final String REPL_DOUBLE_SLASH = "(/|/.+/)";
	public static final String REPL_OPENING_BRACKET = "\\\\Q[\\\\E";
	public static final String REPL_CLOSING_BRACKET = "\\\\Q]\\\\E";

	/**
	 * Supports locations including AbsolutePath,
	 * RelativePath,Predicates,Wildcard i.e. the combination of /,//,[],*
	 * 
	 * @param string:XPath:/A[n]/B[n]/.../Z[n][@]
	 * @param regex
	 * @return
	 */
	public static boolean match( String string, String regex )
	{
		// XPath
		if ( !checkXPath( string, regex ) )
			return false;

		if ( !checkStepLength( string, regex ) )
			return false;

		// MedicatedRE
		regex = toMediatedRE( regex );
		if ( !( checkLastStep( string, regex ) ) )
			return false;

		// RE
		regex = toRE( regex );
		return string.matches( regex );
	}

	private static boolean checkXPath( String string, String regex )
	{
		if ( regex == null || string == null )
			return false;
		if ( !regex.startsWith( SLASH ) )
			return false;
		if ( string.equals( regex ) )
			return true;

		return true;
	}

	private static boolean checkStepLength( String string, String regex )
	{
		boolean absolute = regex.indexOf( DOUBLE_SLASH ) == -1 ? true : false;
		if ( absolute )
		{
			if ( regex.split( SLASH ).length != string.split( SLASH ).length )
				return false;

		}
		else
		{
			regex = regex.replaceAll( DOUBLE_SLASH, SLASH );
			if ( regex.split( SLASH ).length > string.split( SLASH ).length )
				return false;
		}

		return true;
	}

	/**
	 * Change to a RE-like string, leaving // and [] intact
	 * 
	 * @param regex
	 * @return
	 */
	private static String toMediatedRE( String regex )
	{
		// replace [*] with "", * with .+
		regex = regex.replaceAll( RE_PREDICATE_WILDCARD, EMPTY_STRING );
		regex = regex.replaceAll( RE_XPATH_WILDCARD, REPL_XPATH_WILDCARD );

		StringBuffer buffer = new StringBuffer( regex.trim( ) );

		// locate the first step /
		int start = buffer.toString( ).startsWith( DOUBLE_SLASH ) ? 2 : 1;
		int end = 0;
		String step = "";

		while ( buffer.indexOf( SLASH, start ) != -1 )
		{
			end = buffer.indexOf( SLASH, start );
			step = buffer.toString( ).substring( start, end );

			if ( !step.matches( RE_NODETEST_WITH_PREDICATE_NUM_AND_OPTIONAL_ATTR ) )
			{
				buffer.replace( start, end, qualifyStep( step ) );
				end += WILDCARD_PREDICATE_NUMBER.length( );
			}

			start = ( String.valueOf( buffer.toString( ).charAt( end + 1 ) )
					.equals( SLASH ) ? 2 : 1 )
					+ end;
		}

		// lastStep
		step = buffer.toString( ).substring( start );
		if ( !step.matches( RE_NODETEST_WITH_PREDICATE_NUM_AND_OPTIONAL_ATTR ) )
		{
			buffer.replace( start,
					buffer.toString( ).length( ),
					qualifyStep( step ) );
		}

		return buffer.toString( );
	}

	private static String qualifyStep( String step )
	{
		StringBuffer buffer = new StringBuffer( step );

		// for the step that has no nodetest, insert .+ as nodetest
		if ( buffer.toString( ).matches( RE_WITH_PREDICATE_ONLY ) )
			buffer.insert( 0, REPL_XPATH_WILDCARD );

		buffer.insert( buffer.toString( ).replaceAll( RE_SQUARE_BRACKETS,
				EMPTY_STRING ).length( ), WILDCARD_PREDICATE_NUMBER );

		return buffer.toString( );
	}

	private static boolean checkLastStep( String string, String regex )
	{
		String[] stringSplits = string.substring( string.lastIndexOf( SLASH ) + 1 )
				.split( RE_OPENING_BRACKET );
		String[] regexSplits = regex.substring( regex.lastIndexOf( SLASH ) + 1 )
				.split( RE_OPENING_BRACKET );

		if ( ( !stringSplits[0].equals( regexSplits[0] ) && !regexSplits[0].equals( REPL_XPATH_WILDCARD ) )
				|| stringSplits.length != regexSplits.length )
			return false;

		return true;
	}

	/**
	 * Change to the regular expression with which a string can test whether it
	 * matches or not
	 * 
	 * @param regex
	 * @return
	 */
	private static String toRE( String regex )
	{
		regex = regex.replaceAll( DOUBLE_SLASH, REPL_DOUBLE_SLASH );
		regex = regex.replaceAll( RE_OPENING_BRACKET, REPL_OPENING_BRACKET );
		regex = regex.replaceAll( RE_CLOSING_BRACKET, REPL_CLOSING_BRACKET );

		return regex;
	}

}
