/*******************************************************************************
 * Copyright (c) 2007 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Brian Fitzpatrick - initial API and implementation
 ******************************************************************************/
package org.eclipse.datatools.enablement.jdt.dbunit.internal.ui;

import java.util.Arrays;
import java.util.Comparator;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.datatools.connectivity.ICategory;
import org.eclipse.datatools.connectivity.IConnection;
import org.eclipse.datatools.connectivity.IConnectionFactoryProvider;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.datatools.connectivity.IProfileListener;
import org.eclipse.datatools.connectivity.ProfileConnectionManager;
import org.eclipse.datatools.connectivity.ProfileManager;
import org.eclipse.datatools.connectivity.internal.ConnectionProfileManager;
import org.eclipse.datatools.connectivity.internal.ui.dialogs.ExceptionHandler;
import org.eclipse.datatools.connectivity.internal.ui.wizards.CPWizardNode;
import org.eclipse.datatools.connectivity.internal.ui.wizards.NewCPWizard;
import org.eclipse.datatools.connectivity.internal.ui.wizards.ProfileWizardProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

/**
 * This class provides a simple UI for use in a property page or wizard, showing
 * a list of database connection profiles.
 * 
 * @author brianf
 *
 */
public class ConnectionProfileGroup
{

	public static final String JDBC_CATEGORY = "org.eclipse.datatools.connectivity.db.category"; //$NON-NLS-1$
	
	private Cursor _cursor;
	private Combo _DBCPCombo;
	private boolean _readOnly = false;
	private Button _mModify;
	private Button _mConnect;
	private Composite _cpGroup;
	private String _cpName = null;

	// local category/provider ID stash
	private String _category = null;
	private String _providerID = null;

	
	// change listeners
	private ListenerList changeListeners;
	
	private IConnection _connection = null;
	private Throwable _connectException = null;

	public ConnectionProfileGroup(Composite parent) {
		this(parent, false, false, true, 1);
	}
	/**
	 * Constructor
	 */
	public ConnectionProfileGroup(Composite parent, boolean labelVisible, boolean newButtonVisible, boolean connectButtonVisible, int numColumnsForParent )
	{
		this.changeListeners = new ListenerList();

		// connection profile group
		this._cpGroup = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		int numColumns = 1;
		if (labelVisible) numColumns++;
		if (newButtonVisible) numColumns++;
		if (connectButtonVisible) numColumns++;
		
		layout.numColumns = numColumns;
		layout.marginWidth = 0;
		this._cpGroup.setLayout(layout);
		
		GridData cGridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL
				| GridData.GRAB_HORIZONTAL);
		if (numColumnsForParent > 1) {
			cGridData.horizontalSpan = numColumnsForParent;
			if (labelVisible)
				cGridData.horizontalSpan = cGridData.horizontalSpan - 1;
			if (newButtonVisible)
				cGridData.horizontalSpan = cGridData.horizontalSpan - 1;
			if (connectButtonVisible)
				cGridData.horizontalSpan = cGridData.horizontalSpan - 1;
		}
		this._cpGroup.setLayoutData(cGridData);

		if (labelVisible) {
			Label label = new Label(this._cpGroup, SWT.NONE);
			label.setText(DbUnitMessages.ConnectionProfileGroup_Profile_Combo_Label);
		}

		// CP combo
		this._DBCPCombo = new Combo(this._cpGroup, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL);
		this._DBCPCombo.setLayoutData(data);
		this._DBCPCombo.setEnabled(!this._readOnly);

		if (newButtonVisible) {
			this._mModify = new Button(this._cpGroup, SWT.PUSH);
			this._mModify.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
			this._mModify.setText(DbUnitMessages.ConnectionProfileGroup_New_Button_Label);
			this._mModify.addSelectionListener( new NewButtonSelectionChangedListener());
			this._mModify.setVisible(newButtonVisible);
		}

		if (connectButtonVisible) {
			this._mConnect = new Button(this._cpGroup, SWT.PUSH);
			this._mConnect.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
			this._mConnect.setText(DbUnitMessages.ConnectionProfileGroup_Connect_Button_Label);
			this._mConnect.addSelectionListener( new ConnectButtonSelectionChangedListener(this) );
			this._mConnect.setEnabled(false);
			this._mConnect.setVisible(connectButtonVisible);
		}

		ComboSelectionListener listener = new ComboSelectionListener(this);
		
		this._DBCPCombo.addModifyListener( listener);
		this._DBCPCombo.addSelectionListener( listener );
		
	}

	/**
	 * Select (highlight) the given profile in the combobox of database profiles
	 * 
	 * @param profileName name of the profile to highlight
	 */
	public void setSelectedProfile(String profileName)
	{
		if (this._DBCPCombo.getItemCount() == 0) {
			this._mConnect.setEnabled(false);
			return;
		}
		
		if (profileName != null) {
			this._cpName = profileName;
			this._DBCPCombo.setText(profileName);
		}
		else
			this._mConnect.setEnabled(false);
	}
	
	/**
	 * @return selected connection profile name 
	 */
	public String getSelectedProfile()
	{
		if (this._DBCPCombo.getItemCount() == 0)
			return null;
		
		if (this._DBCPCombo.getSelectionIndex() == -1) 
			return null;
		
		return this._DBCPCombo.getItem(this._DBCPCombo.getSelectionIndex());
	}

	/**
	 * @return
	 */
	public boolean isReadOnly() {
		return this._readOnly;
	}
	
	/**
	 * @return
	 */
	public IConnection getConnection() {
		return this._connection;
	}
	
	/**
	 * @return
	 */
	public Throwable getConnectionException() {
		return this._connectException;
	}
	
	/**
	 * @param flag
	 */
	public void setIsReadOnly ( boolean flag ) {
		this._readOnly = flag;
		if (this._DBCPCombo != null) {
			this._DBCPCombo.setEnabled(!this._readOnly);
			this._mModify.setEnabled(!this._readOnly);
			if (this._DBCPCombo.getSelectionIndex() == -1)
				this._mConnect.setEnabled(false);
		}
	}
	
	/**
	 * If we changed, fire a changed event.
	 * @param source
	 */
	private void fireChangedEvent(Object source)
	{
		ChangeEvent e = new ChangeEvent(source);
		// inform any listeners of the resize event
		Object[] listeners = this.changeListeners.getListeners();
		for (int i = 0; i < listeners.length; ++i)
		{
			((ChangeListener) listeners[i]).stateChanged(e);
		}

		if (this._DBCPCombo.getSelectionIndex() > -1) {
			if (this._mConnect != null)
				this._mConnect.setEnabled(true);
		}
	}

	/**
	 * Add a change listener
	 * @param listener
	 */
	public void addChangeListener(ChangeListener listener)
	{
		this.changeListeners.add(listener);
	}

	/**
	 * Remove a change listener.
	 * @param listener
	 */
	public void removeChangeListener(ChangeListener listener)
	{
		this.changeListeners.remove(listener);
	}

	/**
	 * Get the provider category
	 * @return
	 */
	public String getCategory() {
		return this._category;
	}
	
	/**
	 * Set the provider category
	 * @param category
	 */
	public void setCategory (String category) {
		this._category = category;
	}
	
	/**
	 * Load the combo with the list of profiles
	 */
	public void loadProfiles() {
		this._DBCPCombo.removeAll();
		fillConnectionProfiles();
	}
	
	/**
	 *  Set the list of profiles
	 */
	public void fillConnectionProfiles()
	{
		IConnectionProfile[] profiles = null;
		
		if (this._category != null) 
			profiles = ProfileManager.getInstance().
				getProfilesByCategory(this._category);
		else if (this._providerID != null)
			profiles = ProfileManager.getInstance().
				getProfileByProviderID(this._providerID);
		else 
			profiles = ProfileManager.getInstance().
				getProfiles();

		// sort the list
		Arrays.sort(profiles, new Comparator() {

			public int compare(Object arg0, Object arg1) {
				IConnectionProfile profile1 = (IConnectionProfile) arg0;
				IConnectionProfile profile2 = (IConnectionProfile) arg1;
				String profileName1 = profile1.getName();
				String profileName2 = profile2.getName();
				
				int temp = profileName1.compareToIgnoreCase(profileName2);
				
				return temp;
			}
		});
		
		this._DBCPCombo.removeAll();
		
		for (int i = 0; i < profiles.length; i++) 
		{
			IConnectionProfile profile = profiles[i];
			String profileName = profile.getName();
			this._DBCPCombo.add(profileName);
		}
		// Does it make sense to default the selection? Otherwise need to show an error
		// if no profile selected...
//		if (!(profiles.length == 0))
//			this._DBCPCombo.select(0);
	}

	/**
     * Listener for combo selection events
     * @author brianf
     */
    private class ComboSelectionListener implements SelectionListener, ModifyListener  {

    	private ConnectionProfileGroup parent;
    	
    	/**
    	 * @param combo
    	 */
    	public ComboSelectionListener ( ConnectionProfileGroup combo ) {
    		this.parent = combo;
    	}
    	
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		public void widgetSelected(SelectionEvent e) {
			if (ConnectionProfileGroup.this._cpName != null) {
				if (ConnectionProfileGroup.this._cpName.equals(
						ConnectionProfileGroup.this._DBCPCombo.getText())) {
					return;
				}
				else {
					ConnectionProfileGroup.this._cpName =
						ConnectionProfileGroup.this._DBCPCombo.getText();
				}
			}
			ConnectionProfileGroup.this._connection = null;
	    	fireChangedEvent(this.parent);
		}

		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		public void widgetDefaultSelected(SelectionEvent e) {
			widgetSelected(e);
		}

		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
		 */
		public void modifyText(ModifyEvent e) {
	    	fireChangedEvent(this.parent);
		}
		
	}

    /**
     * Listener for edit button clicks events
     * @author brianf
     */
    public class NewButtonSelectionChangedListener implements SelectionListener {

    	/**
    	 * Constructor
    	 */
    	public NewButtonSelectionChangedListener ( ) {
    		// empty
    	}
    	
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		public void widgetSelected(SelectionEvent e) {
			Shell newShell = new Shell();
	        NewCPWizard wizard;
	        WizardDialog wizardDialog;

	        if (ConnectionProfileGroup.this._category != null) {
	        	final ICategory category = (ICategory) ConnectionProfileManager.getInstance().getCategory(ConnectionProfileGroup.this._category);
	     		ViewerFilter viewerFilter = new ViewerFilter() {
		 			public boolean select(Viewer viewer, Object parentElement,
		 					Object element) {
		 				if (category == null) return true;
		 				CPWizardNode wizardNode = (CPWizardNode) element;
		 				if (wizardNode.getProvider() instanceof ProfileWizardProvider) {		 				
			 				ICategory cat = ConnectionProfileManager
                                    .getInstance().getProvider(
                                            ((ProfileWizardProvider) wizardNode
                                                    .getProvider())
                                                    .getProfile())
                                    .getCategory();
			 				// Only display wizards of a specific category
			 				if (cat.getId().equals(category.getId()))
			 					return true;
		 				}
	 					return false;
		 			}
		 		};

		 		wizard = new NewCPWizard(viewerFilter, null);
	        }
	        else {
		        wizard = new NewCPWizard();
	        }
	        wizardDialog = new WizardDialog(newShell, wizard);
	        wizardDialog.setBlockOnOpen(true);
	        
	        GroupProfileListener listener = new GroupProfileListener();
	        
	        ProfileManager.getInstance().addProfileListener( listener );
	        
	        wizardDialog.open();
	        
	        loadProfiles();
	        
	        if (listener.getNewProfileName() != null) {
	        	setSelectedProfile(listener.getNewProfileName());
	        }
	        
			newShell.dispose();
		}

		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		public void widgetDefaultSelected(SelectionEvent e) {
			widgetSelected(e);
		}

	}

    /**
     * Inner listener to listen for new profiles
     * @author brianf
     */
    private class GroupProfileListener implements IProfileListener {

    	private String newProfileName = null;
    	
    	/**
    	 * Get the new profile name
    	 * @return
    	 */
    	public String getNewProfileName() {
    		return this.newProfileName;
    	}
    	
		/* (non-Javadoc)
		 * @see com.sybase.stf.servers.core.IProfileListener#profileAdded(com.sybase.stf.servers.core.IConnectionProfile)
		 */
		public void profileAdded(IConnectionProfile profile) {
			this.newProfileName = profile.getName();
		}

		/* (non-Javadoc)
		 * @see com.sybase.stf.servers.core.IProfileListener#profileDeleted(com.sybase.stf.servers.core.IConnectionProfile)
		 */
		public void profileDeleted(IConnectionProfile profile) {
			// empty
		}

		/* (non-Javadoc)
		 * @see com.sybase.stf.servers.core.IProfileListener#profileChanged(com.sybase.stf.servers.core.IConnectionProfile)
		 */
		public void profileChanged(IConnectionProfile profile) {
			// empty
		}
    	
    }

    /**
     * 
     */
    private void internalGetConnection() {
		IConnectionProfile icp = ProfileManager.getInstance().getProfileByName(getSelectedProfile());
		IConnectionFactoryProvider factoryProvider = icp.getProvider().getConnectionFactory("java.sql.Connection"); //$NON-NLS-1$
		IConnection connect = null;
		if (factoryProvider != null) {
			ProfileConnectionManager.getProfileConnectionManagerInstance().manageProfileConnection(icp, "java.sql.Connection", this); //$NON-NLS-1$
			if (!icp.isConnected()) {
				IStatus status = icp.connect();
				switch (status.getSeverity()) {
				case IStatus.OK:
				case IStatus.INFO:
				case IStatus.WARNING:
					break;
				case IStatus.ERROR:
					icp.setConnected(false);
					break;
			}
			}
			connect = ProfileConnectionManager.getProfileConnectionManagerInstance().getConnection(icp, "java.sql.Connection"); //$NON-NLS-1$
		}
		
		if (connect != null) {
			if (connect.getConnectException() != null) {
				ConnectionProfileGroup.this._connectException = connect.getConnectException();
				connect.getConnectException().printStackTrace();
				ConnectionProfileGroup.this._connection = null;
				fireChangedEvent(this);
				ExceptionHandler.showException(ConnectionProfileGroup.this._DBCPCombo.getShell(), 
						DbUnitMessages.ConnectionProfileGroup_Connection_Error_Title,
						DbUnitMessages.ConnectionProfileGroup_Connection_Error_Text,
						connect.getConnectException());
			}
			else {
				ConnectionProfileGroup.this._connection = connect;
				ConnectionProfileGroup.this._connectException = null;
				fireChangedEvent(this);
			}
		}
    }
    
    /**
     * 
     */
    public void forceGetConnectionNow() {
    	if (getSelectedProfile() != null && getSelectedProfile().length() > 0)
    		internalGetConnection();
    }
    
    /**
     * Listener for connect button clicks events
     * @author brianf
     */
    public class ConnectButtonSelectionChangedListener implements SelectionListener {

    	/**
    	 * Constructor
    	 */
    	public ConnectButtonSelectionChangedListener ( ConnectionProfileGroup combo ) {
    		// empty
    	}
    	
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		public void widgetSelected(SelectionEvent e) {
			internalGetConnection();
		}

		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		public void widgetDefaultSelected(SelectionEvent e) {
			widgetSelected(e);
		}

	}

	/**
	 * 
	 */
	public void setWaitCursor() {
		// Change the cursor to an hourglass.
		//
		if (this._cursor == null) {
			this._cursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT );
			Display.getDefault().getActiveShell().setCursor(this._cursor);
		}
	}
	
	/**
	 * 
	 */
	public void setPointerCursor() 
	{
		// Change the cursor back...
		//
		if (this._cursor != null) 
		{
			Display.getDefault().getActiveShell().setCursor(null);
			this._cursor.dispose();
			this._cursor = null;
		}
	}
	
	/**
	 * @return
	 */
	public boolean canConnect() {
		return this._mConnect.getEnabled();
	}
	
	/**
	 * @return
	 */
	public boolean canNew() {
		return this._mModify.getEnabled();
	}
	
	/**
	 * @param flag
	 */
	public void setShowConnectButton ( boolean flag ) {
		this._mConnect.setVisible(flag);
	}
}
