/*******************************************************************************
 * Copyright (c) 2007 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Brian Fitzpatrick - initial API and implementation
 ******************************************************************************/
package org.eclipse.datatools.enablement.jdt.dbunit.internal.export;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IPath;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.datatools.connectivity.internal.ui.wizards.BaseWizardPage;
import org.eclipse.datatools.connectivity.sqm.core.connection.DatabaseConnectionRegistry;
import org.eclipse.datatools.enablement.jdt.dbunit.Activator;
import org.eclipse.datatools.enablement.jdt.dbunit.internal.ui.DbUnitMessages;
import org.eclipse.datatools.modelbase.sql.schema.Catalog;
import org.eclipse.datatools.modelbase.sql.schema.Database;
import org.eclipse.datatools.modelbase.sql.schema.SQLObject;
import org.eclipse.datatools.modelbase.sql.schema.Schema;
import org.eclipse.datatools.modelbase.sql.tables.Table;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.SaveAsDialog;

public class ExportDBUDataSetObjectSelectionPage extends BaseWizardPage {

	private Object mIncomingObject;
	private boolean mShowCatalog = false;
	private boolean mShowSchema = false;
	private IConnectionProfile mProfile = null;
	private Object[] mCheckedObjects = null;
	private String mFileName = null;
	private IPath mFilePath = null;
	
	private CheckboxTreeViewer mCTV = null;
	private TreeContentProvider mTCP = null;
	private TreeLabelProvider mTLP = null;
	private Text mDataSetFile;
	
	/**
	 * @param name
	 */
	public ExportDBUDataSetObjectSelectionPage(String name) {
		super(name);
		this.setTitle(DbUnitMessages.ExportDBUDataSetObjectSelectionPage_title);
		this.setDescription(DbUnitMessages.ExportDBUDataSetObjectSelectionPage_description);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		
		ExportDBUDataSetWizard wizard = (ExportDBUDataSetWizard) getWizard();
		this.mIncomingObject = wizard.getIncomingObject();
		this.mProfile = getProfileFromIncomingObject(this.mIncomingObject);
		GridData gd;
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout gl = new GridLayout(3, false);
		composite.setLayout(gl);
		// Connection profile name
		Label nameLabel = new Label(composite, SWT.NONE);
		nameLabel.setText(DbUnitMessages.ExportDBUDataSetObjectSelectionPage_TableList_title);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 3;
		nameLabel.setLayoutData(gd);
		
		mCTV = new CheckboxTreeViewer(composite, SWT.V_SCROLL | SWT.BORDER );
		gd = new GridData(GridData.FILL_BOTH | GridData.HORIZONTAL_ALIGN_BEGINNING);
		gd.horizontalSpan = 2;
		gd.verticalSpan = 5;
		mCTV.getControl().setLayoutData(gd);
		mTCP = new TreeContentProvider();
		mCTV.setContentProvider(mTCP);
		mTLP = new TreeLabelProvider();
		mCTV.setLabelProvider(mTLP);
		mCTV.setInput(this.mIncomingObject);
		mCTV.addCheckStateListener(new ICheckStateListener(){
			public void checkStateChanged(CheckStateChangedEvent event) {
				ExportDBUDataSetObjectSelectionPage.this.mCheckedObjects = 
					ExportDBUDataSetObjectSelectionPage.this.mCTV.getCheckedElements();
				ExportDBUDataSetObjectSelectionPage.this.setPageComplete(
						ExportDBUDataSetObjectSelectionPage.this.isValid());
			}
		});
		
//		Composite filler = new Composite(composite, SWT.NONE);
//		GridLayout flayout = new GridLayout();
//		filler.setLayout(flayout);
//		GridData fGridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL
//				| GridData.GRAB_HORIZONTAL);
//		filler.setLayoutData(fGridData);
		
		
		Button selectAll = new Button (composite, SWT.PUSH);
		selectAll.setText (DbUnitMessages.ExportDBUDataSetObjectSelectionPage_SelectAll_Button_Title);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		selectAll.setLayoutData(gd);
		selectAll.addSelectionListener(new SelectionListener() {

			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}

			public void widgetSelected(SelectionEvent e) {
				ExportDBUDataSetObjectSelectionPage.this.mCTV.setAllChecked(true);
				ExportDBUDataSetObjectSelectionPage.this.mCheckedObjects = 
					ExportDBUDataSetObjectSelectionPage.this.mCTV.getCheckedElements();
				ExportDBUDataSetObjectSelectionPage.this.setPageComplete(
						ExportDBUDataSetObjectSelectionPage.this.isValid());
			}
		});
		
		Button deselectAll = new Button (composite, SWT.PUSH);
		deselectAll.setText (DbUnitMessages.ExportDBUDataSetObjectSelectionPage_DeselectAll_Button_Title);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		deselectAll.setLayoutData(gd);
		deselectAll.addSelectionListener(new SelectionListener() {

			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}

			public void widgetSelected(SelectionEvent e) {
				ExportDBUDataSetObjectSelectionPage.this.mCTV.setAllChecked(false);
				ExportDBUDataSetObjectSelectionPage.this.mCheckedObjects = 
					ExportDBUDataSetObjectSelectionPage.this.mCTV.getCheckedElements();
				ExportDBUDataSetObjectSelectionPage.this.setPageComplete(
						ExportDBUDataSetObjectSelectionPage.this.isValid());
			}
		});

		Composite composite2 = new Composite(composite, SWT.NULL);
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		layout.marginWidth = 0;
		composite2.setLayout(layout);
		GridData cGridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL
				| GridData.GRAB_HORIZONTAL);
		cGridData.horizontalSpan = 3;
		composite2.setLayoutData(cGridData);
		
		Label initialDataSetLabel = new Label(composite2, SWT.LEFT | SWT.WRAP);
		initialDataSetLabel.setFont(composite.getFont());
		initialDataSetLabel.setText(DbUnitMessages.ExportDBUDataSetObjectSelectionPage_FileName_Label); 
		initialDataSetLabel.setLayoutData(new GridData());
		
		mDataSetFile = new Text(composite2, SWT.BORDER);
		mDataSetFile.setFont(composite.getFont());
		mDataSetFile.setText(DbUnitMessages.ExportDBUDataSetObjectSelectionPage_Initial_Filename); 
		GridData idsGridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL);
		mDataSetFile.setLayoutData(idsGridData);
		mDataSetFile.addKeyListener(new KeyListener() {

			public void keyPressed(KeyEvent e) {
				// ignore
			}

			public void keyReleased(KeyEvent e) {
				String newLocation = ExportDBUDataSetObjectSelectionPage.this.mDataSetFile.getText();
				
				if (newLocation != null && !newLocation.equals(ExportDBUDataSetObjectSelectionPage.this.mFileName)) {
//					ExportDBUDataSetObjectSelectionPage.this.mDataSetFile.setText(newLocation);
					ExportDBUDataSetObjectSelectionPage.this.mFileName = newLocation;
				}

				ExportDBUDataSetObjectSelectionPage.this.setPageComplete(
						ExportDBUDataSetObjectSelectionPage.this.isValid());
			}
		});
		
		Button browseFile = new Button(composite2, SWT.PUSH);
		browseFile.setText(DbUnitMessages.ExportDBUDataSetObjectSelectionPage_Browse_Button_Title);
		browseFile.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL ));
		browseFile.addSelectionListener(new SelectionListener(){

			public void widgetDefaultSelected(SelectionEvent e) {
				IPath newLocation = 
					handleLocationEditButtonPressed(ExportDBUDataSetObjectSelectionPage.this.mDataSetFile.getText());
				
				if (newLocation != null && !newLocation.equals(ExportDBUDataSetObjectSelectionPage.this.mFilePath)) {
					
					ExportDBUDataSetObjectSelectionPage.this.mDataSetFile.setText(newLocation.toPortableString());
					ExportDBUDataSetObjectSelectionPage.this.mFilePath = newLocation;
					ExportDBUDataSetObjectSelectionPage.this.mFileName = newLocation.toPortableString();
				}
				
				ExportDBUDataSetObjectSelectionPage.this.setPageComplete(
						ExportDBUDataSetObjectSelectionPage.this.isValid());
			}

			public void widgetSelected(SelectionEvent e) {
				widgetDefaultSelected(e);
			}
		});
		
		setControl(composite);
		setPageComplete(false);
	}
	
	/*
	 * Browse for a location, with the location passed in as the default
	 */
	private IPath handleLocationEditButtonPressed(String editLocation) {
		SaveAsDialog rsDialog = new SaveAsDialog(getShell());
		rsDialog.setOriginalName(editLocation);
		rsDialog.setTitle(DbUnitMessages.ExportDBUDataSetObjectSelectionPage_FileDialog_Title);
		rsDialog.create();

		int rtn = rsDialog.open();
		if (rtn == Window.OK) {
			IPath selectedPath = rsDialog.getResult();
			return selectedPath;
		}
		return null;
	}
	
	/**
	 * @return
	 */
	public String getFileName() {
		return this.mFileName;
	}
	
	public IPath getFilePath() {
		return this.mFilePath;
	}
	
	/**
	 * @return
	 */
	public IConnectionProfile getProfile() {
		return this.mProfile;
	}
	
	/**
	 * @return
	 */
	public Object[] getSelectedObjects() {
		return this.mCheckedObjects;
	}
	
	/**
	 * @param incoming
	 * @return
	 */
	private IConnectionProfile getProfileFromIncomingObject( Object incoming ) {
		Database db = null;
		if (incoming instanceof Database) {
			db = (Database) incoming;
		}
		else if (incoming instanceof Catalog) {
			db = ((Catalog)incoming).getDatabase();
		}
		else if (incoming instanceof Schema) {
			Schema schema = (Schema) incoming;
			if (schema.getDatabase() != null) {
				db = schema.getDatabase();
			}
			else if (schema.getCatalog() != null) {
				db = schema.getCatalog().getDatabase();
			}
		}
		else if (incoming instanceof Table) {
			Table table = (Table) incoming;
			if (table.getSchema() != null) {
				Schema schema = table.getSchema();
				if (schema.getDatabase() != null)
					db = ((Table)incoming).getSchema().getDatabase();
				else {
					Catalog catalog = schema.getCatalog();
					db = catalog.getDatabase();
				}
			}
			
		}
		return DatabaseConnectionRegistry.getConnectionForDatabase(db).getConnectionProfile();
	}
	
	
	/**
	 * @param table
	 * @param showCatalog
	 * @param showSchema
	 * @return
	 */
	public static String getQualifiedTableName ( Table table, boolean showCatalog, boolean showSchema ) {
		Schema schema = null;
		if (table != null)
			schema = table.getSchema();
		Catalog catalog = null;
		if (schema != null)
			catalog = schema.getCatalog();
		String text = table.getName();
		if (showCatalog && schema != null && schema.getName().length() > 0) 
			text = schema.getName() + "." + text; //$NON-NLS-1$
		if (showSchema && catalog != null && catalog.getName().length() > 0)
			text = catalog.getName() + "." + text; //$NON-NLS-1$
		return text;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.datatools.connectivity.internal.ui.wizards.BaseWizardPage#getSummaryData()
	 */
	public List getSummaryData() {
		List data = new ArrayList();

		if (this.getProfile() != null) {
			data.add(new String[] {
					DbUnitMessages.ExportDBUDataSetObjectSelectionPage_Summary_Selected_Profile,
					getProfile().getName()});
		}

		if (this.getSelectedObjects() != null && this.getSelectedObjects().length > 0) {
			data.add( new String[] {
					DbUnitMessages.ExportDBUDataSetObjectSelectionPage_Summary_Selected_Tables,
					""}); //$NON-NLS-1$
			for (int i = 0; i < this.getSelectedObjects().length; i++) {
				data.add(new String[] {
						"", //$NON-NLS-1$
						getQualifiedTableName((Table) this.getSelectedObjects()[i], mShowCatalog, mShowSchema)});
			}
		}
		
		if (this.getFileName() != null) {
			data.add(new String[] {
					DbUnitMessages.ExportDBUDataSetObjectSelectionPage_Summary_Selected_File,
					this.mFileName});
		}

		return data;
	}

	/**
	 * @return
	 */
	private boolean isValid() {
		boolean fileExists = false;
		if (getFileName() != null && getFileName().length() > 0) {
			IFile blah = Activator.getFile(getFileName());
			IContainer blahParent = Activator.getParentFolder(getFileName());
			if (blah != null) 
				blahParent = (IContainer) blah.getParent();
			if (blah != null && blah.exists()) {
				fileExists = true;
			}
			else if (blahParent != null && blahParent.exists()){
				fileExists = true;
			}
		}
		if (!fileExists) {
			setErrorMessage(DbUnitMessages.ExportDBUDataSetObjectSelectionPage_Error_Must_Select_Valid_File);
		}
		else {
			setErrorMessage(null);
		}
		
		if (fileExists) {
			if (getSelectedObjects() == null || getSelectedObjects().length == 0) {
				setErrorMessage(DbUnitMessages.ExportDBUDataSetObjectSelectionPage_Error_Must_Select_At_Least_One_Table);
			}
			else {
				setErrorMessage(null);
			}
		}
		
		if (getErrorMessage() != null) {
			setPageComplete(false);
			return false;
		}

		return true;
	}

	/**
	 * @author brianf
	 *
	 */
	private class TreeContentProvider implements ITreeContentProvider {
		
		/**
		 * @param list
		 * @param schema
		 */
		private void processSchema ( List list, Schema schema ) {
			for (int i = 0; i < schema.getTables().size(); i++) {
				Table table = (Table) schema.getTables().get(i);
				list.add(table);
			}
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
		 */
		public Object[] getChildren(Object parentElement) {
			ArrayList children = new ArrayList();
			if (parentElement instanceof Database) {
				mShowCatalog = true;
				mShowSchema = true;
				Database database = (Database) parentElement;
				if (database.getCatalogs().size() > 0) {
					Object[] catalogs = database.getCatalogs().toArray();
					for (int i = 0; i < catalogs.length; i++) {
						Object[] schemas = ((Catalog)catalogs[0]).getSchemas().toArray();
						for (int j = 0; j < schemas.length; j++) {
							Schema schema = (Schema) schemas[j];
							processSchema(children, schema);
						}
					}
				}
				else {
					Object[] schemas = database.getSchemas().toArray();
					for (int j = 0; j < schemas.length; j++) {
						Schema schema = (Schema) schemas[j];
						processSchema(children, schema);
					}
				}
			}
			else if (parentElement instanceof Catalog) {
				mShowCatalog = true;
				mShowSchema = true;
				Catalog catalog = (Catalog) parentElement;
				Object[] schemas = catalog.getSchemas().toArray();
				for (int i = 0; i < schemas.length; i++) {
					Schema schema = (Schema) schemas[i];
					processSchema(children, schema);
				}
			}
			else if (parentElement instanceof Schema) {
				mShowSchema = true;
				Schema schema = (Schema) parentElement;
				processSchema(children, schema);
			}
			else if (parentElement instanceof Table) {
				mShowSchema = false;
				Table table = (Table) parentElement;
				children.add(table);
			}
			return children.toArray();
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
		 */
		public Object getParent(Object element) {
			if (element instanceof Table) {
				return ((Table) element).getSchema();
			}
			else if (element instanceof Schema) {
				return ((Schema) element).getCatalog();
			}
			return null;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
		 */
		public boolean hasChildren(Object element) {
			return false;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		 */
		public Object[] getElements(Object inputElement) {
			return getChildren(inputElement);
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose() {
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
		
	}
	
	/**
	 * @author brianf
	 *
	 */
	private class TreeLabelProvider implements ILabelProvider {

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
		 */
		public Image getImage(Object element) {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
		 */
		public String getText(Object element) {
			if (element instanceof Table) {
				Table table = (Table) element;
				return getQualifiedTableName(table, mShowCatalog, mShowSchema);
			}
			else if (element instanceof SQLObject) {
				return ((SQLObject)element).getName();
			}
			return null;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#addListener(org.eclipse.jface.viewers.ILabelProviderListener)
		 */
		public void addListener(ILabelProviderListener listener) {
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
		 */
		public void dispose() {
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#isLabelProperty(java.lang.Object, java.lang.String)
		 */
		public boolean isLabelProperty(Object element, String property) {
			return true;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#removeListener(org.eclipse.jface.viewers.ILabelProviderListener)
		 */
		public void removeListener(ILabelProviderListener listener) {
		}
	}

}
