/*******************************************************************************
 * Copyright (c) 2007 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Bill Siggelkow - initial API and implementation
 * 		see http://www.oreillynet.com/onjava/blog/2005/10/dbunit_made_easy.html
 * 				Brian Fitzpatrick - initial API and implementation
 ******************************************************************************/
package org.eclipse.datatools.enablement.jdt.dbunit.internal.export;

import java.io.File;
import java.io.FileOutputStream;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.sql.DataSource;

import org.dbunit.database.DatabaseConfig;
import org.dbunit.database.DatabaseConnection;
import org.dbunit.database.IDatabaseConnection;
import org.dbunit.database.QueryDataSet;
import org.dbunit.dataset.Column;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.filter.IColumnFilter;
import org.dbunit.dataset.xml.FlatXmlDataSet;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.datatools.enablement.jdt.dbunit.Activator;
import org.eclipse.datatools.enablement.jdt.dbunit.internal.ui.DbUnitMessages;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;

/**
 * Extracts a DBUnit flat XML dataset from a database.
 *
 * @author Bill Siggelkow
 */
public class DBUnitDataExtractor {

    private DataSource dataSource;
    private String dataSetName = "dbunit-dataset.xml"; //$NON-NLS-1$
    private List queryList;
    private List tableList;
    private Map dbUnitProperties;
    private Map dbUnitFeatures;
    private String schema;

    /**
     * A regular expression that is used to get the table name
     * from a SQL 'select' statement.
     * This  pattern matches a string that starts with any characters,
     * followed by the case-insensitive word 'from',
     * followed by a table name of the form 'foo' or 'schema.foo',
     * followed by any number of remaining characters.
     */
    private static final Pattern TABLE_MATCH_PATTERN = Pattern.compile(".*\\s+from\\s+(\\w+(\\.\\w+)?).*", //$NON-NLS-1$
            Pattern.CASE_INSENSITIVE);
//    private static final Logger log = Logger.getLogger(DBUnitDataExtractor.class);

    /**
     * The data source of the database from which the data will be extracted. This property
     * is required.
     *
     * @param ds
     */
    public void setDataSource(DataSource ds) {
        dataSource = ds;
    }

    /**
     * Set the schema.
     * @param schema
     */
    public void setSchema(String schema) {
        this.schema = schema;
    }

    /**
     * Name of the XML file that will be created. Defaults to <code>dbunit-dataset.xml</code>.
     *
     * @param name file name.
     */
    public void setDataSetName(String name) {
        dataSetName = name;
    }

    /**
     * Performs the extraction. If no tables or queries are specified, data from entire database
     * will be extracted. Otherwise, a partial extraction will be performed.
     * @throws Exception
     */
    public void extract(boolean closeWhenDone) throws Exception {
        Connection conn = null;
        try {
            conn = dataSource.getConnection();
            log("Beginning extraction from '"+conn.toString()+"'."); //$NON-NLS-1$ //$NON-NLS-2$
            IDatabaseConnection connection = null;
            if (schema != null)
            	connection = new DatabaseConnection(conn, schema);
            else
            	connection = new DatabaseConnection(conn);
            configConnection((DatabaseConnection) connection);
    	    DatabaseConfig config = connection.getConfig();
    		DatabaseMetaData metadata = conn.getMetaData();
    		// Check if the jdbc driver supports the getPrimaryKeys call 
    		// If if does't, the export will fail, and we need to define a void filter for primary keys
    		try {
    			metadata.getPrimaryKeys(null, null, null);
    		} catch (SQLException e) {
    			// If it doesn't, instruct it to not use the getPrimaryKeys(), define a void filter
    			config.setProperty(
    					DatabaseConfig.PROPERTY_PRIMARY_KEY_FILTER,
    					new MyPrimaryKeyFilter());
    		} catch (Exception e) {
    			// do nothing, it fails but it's not an SQLException, perhaps they don't like the
    			// three null parameters. In any case, possibly supports the operation, because if not
    			// it's easier to simply return an SQLException.
    		}
            if (tableList != null || queryList != null) {
                // partial database export
                QueryDataSet partialDataSet = new QueryDataSet(connection);
                addTables(partialDataSet);
                addQueries(partialDataSet);

        		Path dsPath = new Path(dataSetName);
                IFile dsFile = Activator.getFile(dsPath.toPortableString());
                String hardPath = null;
                if (dsFile != null)
                	hardPath = dsFile.getLocation().makeAbsolute().toOSString();
                else {
                	IContainer dsFolder = Activator.getParentFolder(dsPath.toPortableString());
                	IPath newPath = dsFolder.getLocation().append(dsPath.lastSegment());
                	hardPath = newPath.makeAbsolute().toOSString();
                }
                File file = new File(hardPath);
                if (file != null && !file.exists()) {
                	boolean flag = file.createNewFile();
                	log("New file created ( " + flag + " ): " + file.getAbsolutePath());
                }

                try {
	        		FlatXmlDataSet.write(partialDataSet,
	                        new FileOutputStream(file));
	        		if (dsFile == null)
	        			dsFile = Activator.getFile(file.getAbsolutePath());
	        		if (dsFile != null)
	        			dsFile.refreshLocal(IResource.DEPTH_ONE, null);
                } catch (DataSetException dse) {
                	Activator.log(dse);
                	MessageDialog.openError(new Shell(), 
                			DbUnitMessages.DbUnitDataExtractor_DataSetGenerationError, 
                			DbUnitMessages.DbUnitDataExtractor_DataSetGenerationError_Message);
                }
            } else {
                // full database export
                IDataSet fullDataSet = connection.createDataSet();
                FlatXmlDataSet.write(fullDataSet, new FileOutputStream(dataSetName));
            }
        }
        finally {
        	if (closeWhenDone)
        		if (conn != null) conn.close();
        }
        log("Completed extraction to '"+dataSetName+"'."); //$NON-NLS-1$ //$NON-NLS-2$
    }
    
	/**
     * @throws Exception
     */
    public void extract() throws Exception {
    	extract(true);
    }
    
    /**
     * @param msg
     */
    private void log(String msg) {
    	System.out.println(msg);
    }

    /**
     * List of table names to extract data from.
     *
     * @param list of table names.
     */
    public void setTableList(List list) {
        tableList = list;
    }

    /**
     * List of SQL queries (i.e. 'select' statements) that will be used executed to retrieve
     * the data to be extracted. If the table being queried is also specified in the <code>tableList</code>
     * property, the query will be ignored and all rows will be extracted from that table.
     *
     * @param list of SQL queries.
     */
    public void setQueryList(List list) {
        queryList = list;
    }

    /**
     * @param dbUnitFeatures
     */
    public void setDbUnitFeatures(Map dbUnitFeatures) {
        this.dbUnitFeatures = dbUnitFeatures;
    }

    /**
     * @param dbUnitProperties
     */
    public void setDbUnitProperties(Map dbUnitProperties) {
        this.dbUnitProperties = dbUnitProperties;
    }

    /**
     * @param conn
     */
    private void configConnection(DatabaseConnection conn) {
        DatabaseConfig config = conn.getConfig();
        if (dbUnitProperties != null) {
            for (Iterator k=dbUnitProperties.entrySet().iterator(); k.hasNext(); ) {
                Map.Entry entry = (Map.Entry) k.next();
                String name = (String) entry.getKey();
                Object value = entry.getValue();
                config.setProperty(name, value);
            }
        }
        if (dbUnitFeatures != null) {
            for (Iterator k=dbUnitFeatures.entrySet().iterator(); k.hasNext(); ) {
                Map.Entry entry = (Map.Entry) k.next();
                String name = (String) entry.getKey();
                boolean value = Boolean.valueOf((String)entry.getValue()).booleanValue();
                config.setFeature(name, value);
            }
        }
    }

    /**
     * @param dataSet
     */
    private void addTables(QueryDataSet dataSet) {
        if (tableList == null) return;
        for (Iterator k = tableList.iterator(); k.hasNext(); ) {
            String table = (String) k.next();
            dataSet.addTable(table);
        }
    }

    /**
     * @param dataSet
     */
    private void addQueries(QueryDataSet dataSet) {
        if (queryList == null) return;
        for (Iterator k = queryList.iterator(); k.hasNext(); ) {
            String query = (String) k.next();
            Matcher m = TABLE_MATCH_PATTERN.matcher(query);
            if (!m.matches()) {
                log("Unable to parse query. Ignoring '" + query +"'."); //$NON-NLS-1$ //$NON-NLS-2$
            }
            else {
                String table = m.group(1);
                // only add if the table has not been added
                if (tableList != null && tableList.contains(table)) {
                    log("Table '"+table+"' already added. Ignoring '" + query + "'."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
                } else {
                    dataSet.addTable(table, query);
                }
            }
        }
    }

    //  Void primary key filter to avoid getPrimaryKeys calls in substandard drivers
    class MyPrimaryKeyFilter implements IColumnFilter {

    	/* (non-Javadoc)
    	 * @see org.dbunit.dataset.filter.IColumnFilter#accept(java.lang.String, org.dbunit.dataset.Column)
    	 */
    	public boolean accept(String arg0, Column arg1) {
    		return false;
    	}
    }
}
