/*******************************************************************************
 * Copyright (c) 2007 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Brian Fitzpatrick - initial API and implementation
 ******************************************************************************/

package org.eclipse.datatools.enablement.jdt.dbunit.internal.buildpath;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Arrays;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.datatools.enablement.jdt.dbunit.internal.buildpath.DbUnitContainerInitializer;
import org.eclipse.datatools.enablement.jdt.dbunit.internal.buildpath.DbUnitHomeInitializer;
import org.eclipse.datatools.enablement.jdt.dbunit.internal.buildpath.MyFilenameFilter;
import org.eclipse.datatools.connectivity.drivers.DriverInstance;
import org.eclipse.datatools.connectivity.drivers.DriverManager;
import org.eclipse.datatools.enablement.jdt.classpath.internal.DriverClasspathContainer;
import org.eclipse.datatools.enablement.jdt.dbunit.Activator;
import org.eclipse.jdt.core.IAccessRule;
import org.eclipse.jdt.core.IClasspathAttribute;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.ui.wizards.buildpaths.CPListElement;
import org.eclipse.swt.widgets.Shell;
import org.osgi.framework.Bundle;

/**
 * 
 */
public class BuildPathSupport {
	
	public static final String DBUNIT2_PLUGIN_ID= "org.dbunit"; //$NON-NLS-1$
	public static final String DBUNIT2_LIBS_DIR= "libs"; //$NON-NLS-1$
	
	/**
	 * @param bundleName
	 * @return
	 */
	public static IPath getBundleLocation(String bundleName) {
		Bundle bundle= Activator.getDefault().getBundle(bundleName);
		if (bundle == null)
			return null;
		
		URL local= null;
		try {
			local= FileLocator.toFileURL(bundle.getEntry("/")); //$NON-NLS-1$
		} catch (IOException e) {
			return null;
		}
		String fullPath= new File(local.getPath()).getAbsolutePath();
		return Path.fromOSString(fullPath);
	}
	
	/**
	 * @param bundleName
	 * @return
	 */
	public static IPath getSourceLocation(String bundleName) {
		Bundle bundle= Activator.getDefault().getBundle(bundleName);
		if (bundle == null)
			return null;
			
		String version= (String)bundle.getHeaders().get("Bundle-Version"); //$NON-NLS-1$
		if (version == null) {
			return null;
		}
		bundle= Activator.getDefault().getBundle(DBUNIT2_PLUGIN_ID); //$NON-NLS-1$
		if (bundle == null) {
			return null;
		}
		URL local= null;
		try {
			local= FileLocator.toFileURL(bundle.getEntry("/")); //$NON-NLS-1$
		} catch (IOException e) {
			return null;
		}
		String fullPath= new File(local.getPath()).getAbsolutePath() 
			+ File.separator + "src" + File.separator + bundleName + "_" + version;   //$NON-NLS-1$ //$NON-NLS-2$
		return Path.fromOSString(fullPath);
	}
	
	/**
	 * @return
	 */
	public static IClasspathEntry getDbUnit2ClasspathEntry() {
		return JavaCore.newContainerEntry(DbUnitContainerInitializer.DBUNIT2_PATH);
	}
	
	/**
	 * @return
	 */
	public static IClasspathEntry getDbUnit2LibraryEntry() {
		IPath bundleBase= getBundleLocation(DBUNIT2_PLUGIN_ID);
		if (bundleBase != null) {
			IPath jarLocation= bundleBase.append("dbunit-2.2.jar"); //$NON-NLS-1$
			
			IPath sourceBase= getSourceLocation(DBUNIT2_PLUGIN_ID);
			IPath srcLocation= sourceBase != null ? sourceBase.append("dbunit-2.2-sources.jar") : null; //$NON-NLS-1$
			
			return JavaCore.newLibraryEntry(jarLocation, srcLocation, null);
		}
		return null;
	}
	
	/**
	 * @return
	 */
	public static IClasspathEntry getDbUnit2VariableEntry() {
		IPath jarBundleBase = JavaCore.getClasspathVariable(DbUnitHomeInitializer.DBUNIT_HOME);
//		IPath bundleBase= getBundleLocation(DBUNIT2_PLUGIN_ID);
		if (jarBundleBase != null) {
			IPath jarLocation= jarBundleBase.append("dbunit-2.2.jar"); //$NON-NLS-1$
			
//			IPath sourceBase= getSourceLocation(DBUNIT2_PLUGIN_ID);
			IPath sourceBase = JavaCore.getClasspathVariable(DbUnitHomeInitializer.DBUNIT_SRC_HOME);
			IPath srcLocation= sourceBase != null ? sourceBase.append("dbunit-2.2-sources.jar") : null; //$NON-NLS-1$
			
			IClasspathEntry entry = JavaCore.newLibraryEntry(jarLocation, srcLocation, null, new IAccessRule[0], new IClasspathAttribute[0], true);
			
			return entry;
			
		}
		return null;
	}

	/**
	 * @param driverID
	 * @return
	 */
	public static IClasspathEntry getDriverVariableEntry( String driverID ) {
		DriverInstance di = 
			DriverManager.getInstance().getDriverInstanceByID(driverID);
		if (di != null) {
			DriverClasspathContainer dcc = new DriverClasspathContainer(di.getName());
			if (dcc != null && dcc.getClasspathEntries() != null && dcc.getClasspathEntries().length > 0)
			return dcc.getClasspathEntries()[0];
		}
		return null;
	}
	
	/**
	 * @param path
	 * @param pattern
	 * @return
	 */
	private static String getResolvedFilename ( IPath path, String pattern ) {
		String absolutePath = path.toOSString();
		File folder = new File(absolutePath);
		if (folder.exists()) {
			String[] files = folder.list( new MyFilenameFilter(pattern));
			if (files != null && files.length > 0) {
				Arrays.sort(files);
				return files[0];
			}
		}
		return null;
	}
	
	/**
	 * @param path
	 * @param libName
	 * @return
	 */
	private static IClasspathEntry getEntryForValues ( IPath path, String libName ) {
		String realName = getResolvedFilename(path, libName);
		IPath jarLocation = null;
		if (realName != null)
			jarLocation = path.append(realName);
		else
			jarLocation = path.append(libName);
		return JavaCore.newLibraryEntry(jarLocation, null, null);
	}
	
	/**
	 * @return
	 */
	public static IClasspathEntry[] getOtherDbUnitVariableEntries() {
		IPath jarBundleBase = JavaCore.getClasspathVariable(DbUnitHomeInitializer.DBUNIT_HOME);
		if (jarBundleBase != null) {
			IClasspathEntry[] entries = new IClasspathEntry[6];
			entries[0] = getEntryForValues(jarBundleBase, "commons-collections.*.jar$"); //$NON-NLS-1$
			entries[1] = getEntryForValues(jarBundleBase, "commons-lang.*.jar$"); //$NON-NLS-1$
			entries[2] = getEntryForValues(jarBundleBase, "commons-logging.*.jar$"); //$NON-NLS-1$
			entries[3] = getEntryForValues(jarBundleBase, "poi.*.jar$"); //$NON-NLS-1$
			entries[4] = getEntryForValues(jarBundleBase, "xml-apis.jar"); //$NON-NLS-1$
			entries[5] = getEntryForValues(jarBundleBase, "xercesImpl.jar"); //$NON-NLS-1$
			
			return entries;
		}
		return null;
	}

	/**
	 * @param shell
	 * @param newEntry
	 * @param changedAttributes
	 * @param jproject
	 * @param monitor
	 * @return
	 * @throws CoreException
	 */
	public static IClasspathEntry modifyClasspathEntry(Shell shell, IClasspathEntry newEntry, String[] changedAttributes, IJavaProject jproject, IProgressMonitor monitor) throws CoreException {
		return updateProjectClasspath(shell, jproject, newEntry, changedAttributes, monitor);
	}
	
	
	/**
	 * @param currEntry
	 * @param updatedEntry
	 * @param updatedAttributes
	 * @param jproject
	 * @return
	 */
	private static IClasspathEntry getUpdatedEntry(IClasspathEntry currEntry, IClasspathEntry updatedEntry, String[] updatedAttributes, IJavaProject jproject) {
		if (updatedAttributes == null) {
			return updatedEntry; // used updated entry 'as is'
		}
		CPListElement currElem= CPListElement.createFromExisting(currEntry, jproject);
		CPListElement newElem= CPListElement.createFromExisting(updatedEntry, jproject);
		for (int i= 0; i < updatedAttributes.length; i++) {
			String attrib= updatedAttributes[i];
			currElem.setAttribute(attrib, newElem.getAttribute(attrib));
		}
		return currElem.getClasspathEntry();
	}

	/**
	 * @param shell
	 * @param jproject
	 * @param newEntry
	 * @param changedAttributes
	 * @param monitor
	 * @return
	 * @throws JavaModelException
	 */
	private static IClasspathEntry updateProjectClasspath(Shell shell, IJavaProject jproject, IClasspathEntry newEntry, String[] changedAttributes, IProgressMonitor monitor) throws JavaModelException {
		IClasspathEntry[] oldClasspath= jproject.getRawClasspath();
		int nEntries= oldClasspath.length;
		int entryKind= newEntry.getEntryKind();
		IPath jarPath= newEntry.getPath();
		boolean found= false;
		for (int i= 0; i < nEntries; i++) {
			IClasspathEntry curr= oldClasspath[i];
			if (curr.getEntryKind() == entryKind && curr.getPath().equals(jarPath)) {
				newEntry = getUpdatedEntry(curr, newEntry, changedAttributes, jproject);
				found= true;
			}
		}
		if (!found) {
			return newEntry;
		}
		return null;
	}
	
	/**
	 * @param nativeLibraryPath
	 * @param elem
	 * @param entry
	 * @return
	 */
	public static IClasspathEntry updateNativeLibraryForClassPath ( String nativeLibraryPath, IPackageFragmentRoot elem, IClasspathEntry entry ) {
		try {
			IJavaProject project= elem.getJavaProject();
			if (elem instanceof IPackageFragmentRoot) {
				CPListElement cpElem= CPListElement.createFromExisting(entry, project);
				cpElem.setAttribute(CPListElement.NATIVE_LIB_PATH, nativeLibraryPath);
				IClasspathEntry newEntry= cpElem.getClasspathEntry();
				String[] changedAttributes= { CPListElement.NATIVE_LIB_PATH };
				return BuildPathSupport.modifyClasspathEntry(new Shell(), newEntry, changedAttributes, project, null);
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		return null;
	}

}
