/*******************************************************************************
 * Copyright (c) 2007 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Brian Fitzpatrick - initial API and implementation
 ******************************************************************************/
package org.eclipse.datatools.enablement.hsqldb.ui.properties;

import java.io.File;
import java.util.Properties;
import java.util.StringTokenizer;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.eclipse.core.runtime.Path;
import org.eclipse.datatools.connectivity.ConnectionProfileConstants;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.datatools.connectivity.db.generic.IDBConnectionProfileConstants;
import org.eclipse.datatools.connectivity.db.generic.IDBDriverDefinitionConstants;
import org.eclipse.datatools.connectivity.drivers.DriverInstance;
import org.eclipse.datatools.connectivity.internal.ui.DelimitedStringList;
import org.eclipse.datatools.connectivity.internal.ui.DriverListCombo;
import org.eclipse.datatools.connectivity.ui.wizards.ProfileDetailsPropertyPage;
import org.eclipse.datatools.enablement.hsqldb.ui.IHSQLDBConnectionProfileConstants;
import org.eclipse.datatools.enablement.hsqldb.ui.Messages;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

/**
 * Property page for HSQLDB profiles
 *
 */
public class HSQLDBPropertyPage extends ProfileDetailsPropertyPage {

	private static final String EMPTY_STRING = new String();

	// UI components
	private Text mDatabaseNameText;
	private Text mDBUIDText;
	private Text mDBPWDText;
	private Text mURLText;
	private Button mSaveDBPWDCheckbox;
	private Text mDBFilePathText;
	private Button mBrowseFilePathButton;
	final DriverListCombo combo = new DriverListCombo();
	private DelimitedStringList mDBConnProps;

	// other components
	private DriverInstance mDriverInstance;
	private String mDriverCategory;

	/**
	 * Constructor
	 */
	public HSQLDBPropertyPage() {
		super();
		noDefaultAndApplyButton();
		setDriverCategory(IHSQLDBConnectionProfileConstants.HSQLDB_CATEGORY_ID);
	}

	/**
	 * @see PreferencePage#createContents(Composite)
	 */
	protected void createCustomContents(Composite parent) {
		Composite content = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout(2, false);
		content.setLayout(layout);

		this.combo.setLabelText(Messages.getString(
				"HSQLDBProfileDetailsWizardPage.driverCombo.label")); //$NON-NLS-1$
		this.combo.setCategory(getDriverCategory());
		this.combo.setNullDriverIsValid(false);
		this.combo.createContents(content);

		this.mDatabaseNameText = (Text) createLabelTextPair(
				content,
				Messages.getString(
						"HSQLDBProfileDetailsWizardPage.databaseName.label"), //$NON-NLS-1$
				this.mURLText, SWT.BORDER, GridData.FILL_HORIZONTAL);		

		Label label = new Label(content, SWT.NULL);
		label.setLayoutData(new GridData());
		label.setText(Messages.getString("HSQLDBProfileDetailsWizardPage.0")); //$NON-NLS-1$

		Composite textAndBrowseComposite = new Composite(content, SWT.NULL);
		GridLayout subCompositeLayout = new GridLayout(2, false);
		subCompositeLayout.marginLeft = -5;
		subCompositeLayout.marginRight = -5;
		subCompositeLayout.marginTop = -5;
		subCompositeLayout.marginBottom = -5;
		textAndBrowseComposite.setLayout(subCompositeLayout);
		textAndBrowseComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		this.mDBFilePathText = new Text(textAndBrowseComposite, SWT.BORDER);
		this.mDBFilePathText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		this.mBrowseFilePathButton = new Button(textAndBrowseComposite, SWT.PUSH);
		this.mBrowseFilePathButton.setText(Messages.getString("HSQLDBProfileDetailsWizardPage.1")); //$NON-NLS-1$
		this.mBrowseFilePathButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));

		this.mURLText = (Text) createLabelTextPair(content, Messages.getString(
						"HSQLDBProfileDetailsWizardPage.url.label"), //$NON-NLS-1$
				this.mURLText, SWT.BORDER, GridData.FILL_HORIZONTAL);
		this.mDBUIDText = (Text) createLabelTextPair(content, Messages.getString(
						"HSQLDBProfileDetailsWizardPage.userName.label"), //$NON-NLS-1$ 
				this.mDBUIDText, SWT.BORDER, GridData.FILL_HORIZONTAL);

		this.mDBPWDText = (Text) createLabelTextPair(content, Messages.getString(
						"HSQLDBProfileDetailsWizardPage.password.label"), //$NON-NLS-1$, 
				this.mDBPWDText, SWT.BORDER | SWT.PASSWORD, GridData.FILL_HORIZONTAL);

		this.mSaveDBPWDCheckbox = new Button(content, SWT.CHECK);
		this.mSaveDBPWDCheckbox.setText(Messages
				.getString(
						"HSQLDBProfileDetailsWizardPage.persistpassword.label")); //$NON-NLS-1$
		this.mSaveDBPWDCheckbox.setLayoutData(new GridData(GridData.BEGINNING,
				GridData.CENTER, true, false, 2, 1));

		Composite spacer = new Composite(content, SWT.NULL);
		GridData gdata = new GridData(GridData.FILL_HORIZONTAL);
		gdata.horizontalSpan = 2;
		gdata.heightHint = 20;
		spacer.setLayoutData(gdata);

		label = new Label(content, SWT.NULL);
		gdata = new GridData(GridData.FILL_HORIZONTAL);
		gdata.horizontalSpan = 2;
		label.setLayoutData(gdata);
		label.setText(Messages.getString(
				"HSQLDBProfileDetailsWizardPage.optionalProps.label")); //$NON-NLS-1$

		this.mDBConnProps = new DelimitedStringList(content, SWT.NONE);
		gdata = new GridData(GridData.FILL_HORIZONTAL);
		gdata.horizontalSpan = 2;
		this.mDBConnProps.setLayoutData(gdata);

		this.mDBConnProps.addChangeListener(new ChangeListener() {

			public void stateChanged(ChangeEvent arg0) {
				setErrorMessage(null);
				if (HSQLDBPropertyPage.this.mDBConnProps.getWarning() != null) {
					setErrorMessage(HSQLDBPropertyPage.this.mDBConnProps
							.getWarning());
				}
			}

		});

		this.combo.addChangeListener(new ChangeListener() {

			public void stateChanged(ChangeEvent arg0) {
				HSQLDBPropertyPage.this.mDriverInstance = HSQLDBPropertyPage.this.combo
						.getSelectedDriverInstance();
				setErrorMessage(null);
				if (HSQLDBPropertyPage.this.combo.getErrorMessage() != null) {
					setErrorMessage(HSQLDBPropertyPage.this.combo
							.getErrorMessage());
				}
				else
					HSQLDBPropertyPage.this.mURLText
							.setText(getDriverURL());
				String username = HSQLDBPropertyPage.this
						.getPropertyFromDriverInstance(IDBDriverDefinitionConstants.USERNAME_PROP_ID);
				if (username == null || username.trim().length() == 0)
					username = new String();
				String oldUsername = HSQLDBPropertyPage.this.mDBUIDText
						.getText();
				if (oldUsername == null || oldUsername.trim().length() == 0)
					oldUsername = new String();
				if (oldUsername.length() == 0)
					HSQLDBPropertyPage.this.mDBUIDText
							.setText(username);

				String password = HSQLDBPropertyPage.this
						.getPropertyFromDriverInstance(IDBDriverDefinitionConstants.PASSWORD_PROP_ID);
				if (password == null || password.trim().length() == 0)
					password = new String();
				String oldPassword = HSQLDBPropertyPage.this.mDBPWDText
						.getText();
				if (oldPassword == null || oldPassword.trim().length() == 0)
					oldPassword = new String();
				if (oldPassword.length() == 0)
					HSQLDBPropertyPage.this.mDBPWDText
							.setText(password);
			}

		});

		initControls();

		this.mDBFilePathText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				HSQLDBPropertyPage.this.updateURLFromProps();
			}
		});
		
		this.mBrowseFilePathButton.addSelectionListener( new SelectionListener() {

			public void widgetDefaultSelected(SelectionEvent e) {
				HSQLDBPropertyPage.this.browseForDBFolder();
			}

			public void widgetSelected(SelectionEvent e) {
				HSQLDBPropertyPage.this.browseForDBFolder();
			}
		});

		// return content;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.preference.IPreferencePage#performOk()
	 */
	protected Properties collectProperties() {
		Properties props = new Properties();

		String driverID = EMPTY_STRING;

		if (this.combo.getSelectedDriverInstance() != null) {
			DriverInstance instance = this.combo.getSelectedDriverInstance();
			driverID = instance.getId();
		}

		props.setProperty(ConnectionProfileConstants.PROP_DRIVER_DEFINITION_ID,
				driverID);
		props.setProperty(
				IDBConnectionProfileConstants.CONNECTION_PROPERTIES_PROP_ID,
				this.mDBConnProps.getSelection());	
		props.setProperty(IDBDriverDefinitionConstants.DATABASE_VENDOR_PROP_ID, getPropertyFromDriverInstance(IDBDriverDefinitionConstants.DATABASE_VENDOR_PROP_ID));		
		props.setProperty(IDBDriverDefinitionConstants.DATABASE_VERSION_PROP_ID, getPropertyFromDriverInstance(IDBDriverDefinitionConstants.DATABASE_VERSION_PROP_ID));		
		props.setProperty(IDBDriverDefinitionConstants.DRIVER_CLASS_PROP_ID, getPropertyFromDriverInstance(IDBDriverDefinitionConstants.DRIVER_CLASS_PROP_ID));	
		props.setProperty(IDBDriverDefinitionConstants.DATABASE_NAME_PROP_ID, this.mDatabaseNameText
				.getText());
		props.setProperty(IDBDriverDefinitionConstants.PASSWORD_PROP_ID, this.mDBPWDText
				.getText());
		props.setProperty(IDBDriverDefinitionConstants.USERNAME_PROP_ID, this.mDBUIDText
				.getText());
		props.setProperty(IDBDriverDefinitionConstants.URL_PROP_ID,
				this.mURLText.getText());
		props.setProperty(
				IDBConnectionProfileConstants.SAVE_PASSWORD_PROP_ID, String
						.valueOf(this.mSaveDBPWDCheckbox.getSelection()));

		return props;
	}

	/**
	 * 
	 */
	private void initControls() {
		IConnectionProfile profile = getConnectionProfile();
		String driverID = profile.getBaseProperties().getProperty(
				ConnectionProfileConstants.PROP_DRIVER_DEFINITION_ID);
		if (driverID != null) {
			this.combo.setSelectionToID(driverID);
		}

		String connectionProps = profile.getBaseProperties().getProperty(
				IDBConnectionProfileConstants.CONNECTION_PROPERTIES_PROP_ID);
		if (connectionProps != null) {
			this.mDBConnProps.setSelection(connectionProps);
		}

		String databaseName = profile.getBaseProperties().getProperty(
				IDBDriverDefinitionConstants.DATABASE_NAME_PROP_ID);
		if (databaseName != null) {
			this.mDatabaseNameText.setText(databaseName);
		}
		
		String databasePwd = profile.getBaseProperties().getProperty(
				IDBDriverDefinitionConstants.PASSWORD_PROP_ID);
		if (databasePwd != null) {
			this.mDBPWDText.setText(databasePwd);
		}

		String databaseUid = profile.getBaseProperties().getProperty(
				IDBDriverDefinitionConstants.USERNAME_PROP_ID);
		if (databaseUid != null) {
			this.mDBUIDText.setText(databaseUid);
		}

		String urlText = profile.getBaseProperties().getProperty(
				IDBDriverDefinitionConstants.URL_PROP_ID);
		if (urlText != null) {
			this.mURLText.setText(urlText);
			this.mURLText.setEnabled(false);
		}

		this.mSaveDBPWDCheckbox.setSelection(Boolean.valueOf(
				profile.getBaseProperties().getProperty(
						IDBConnectionProfileConstants.SAVE_PASSWORD_PROP_ID,
						Boolean.FALSE.toString())).booleanValue());

		this.updatePropsFromURL();

		setErrorMessage(null);
		if (this.combo.getErrorMessage() != null) {
			setErrorMessage(this.combo.getErrorMessage());
		}
	}
	
	/**
	 * Browse for a DB folder
	 */
	private void browseForDBFolder() {
		DirectoryDialog dialog = new DirectoryDialog(getShell());
		dialog.setText(Messages
				.getString("HSQLDBProfileDetailsWizardPage.FileDialog.title.filebrowse")); //$NON-NLS-1$
		dialog.setMessage(Messages
				.getString("HSQLDBProfileDetailsWizardPage.FileDialog.msg.filebrowse")); //$NON-NLS-1$

		String dirName = mDBFilePathText.getText();

		File path = new File(dirName);
		if (path.exists()) {
			dialog.setFilterPath(new Path(dirName).toOSString());
		}

		String selectedDirectory = dialog.open();
		if (selectedDirectory != null) {
			this.mDBFilePathText.setText(selectedDirectory);
		}
	}

	/**
	 * Based on the URL, update the UI components accordingly
	 */
	private void updatePropsFromURL() {
		if (this.mDBFilePathText != null && this.mURLText != null) {
			String url = this.mURLText.getText();
			String[] chunks = parseString(url, ";"); //$NON-NLS-1$
			if (chunks.length > 0) {
				String[] chunks2 = parseString(chunks[0], ":"); //$NON-NLS-1$
				if (chunks2.length > 2) {
					String filepath = chunks2[2];
					if (chunks2.length > 3) {
						filepath = filepath + ":" + chunks2[3]; //$NON-NLS-1$
					}
					this.mDBFilePathText.setText(filepath);
				}
			}
		}
	}
	
	/**
	 * Based on the state of the UI components, reset the URL
	 */
	private void updateURLFromProps() {
		if (this.mDBFilePathText != null && this.mURLText != null) {
			String url = this.mURLText.getText();
			String filePath = this.mDBFilePathText.getText();
			//jdbc:derby:C:\DerbyDatabases\MyDB;create=true
			String[] chunks = parseString(url, ";"); //$NON-NLS-1$
			if (chunks.length > 0) {
				String[] chunks2 = parseString(chunks[0], ":"); //$NON-NLS-1$
				if (chunks2.length > 2) {
					url = chunks2[0] + ":" + chunks2[1] + ":" + filePath; //$NON-NLS-1$ //$NON-NLS-2$
					this.mURLText.setText(url);
				}
			}
		}
	}

	/**
	 * Parse a string into an array based on a token
	 * @param str_list
	 * @param token
	 * @return
	 */
	private  String[] parseString(String str_list, String token) {
		StringTokenizer tk = new StringTokenizer(str_list, token);
		String[] pieces = new String[tk.countTokens()];
		int index = 0;
		while (tk.hasMoreTokens())
			pieces[index++] = tk.nextToken();
		return pieces;
	}

	/**
	 * @param propertyID
	 * @return
	 */
	private String getPropertyFromDriverInstance(String propertyID) {
		String returnStr = new String();
		if (this.mDriverInstance != null
				&& this.mDriverInstance.getProperty(propertyID) != null) {
			returnStr = this.mDriverInstance.getProperty(propertyID);
		}
		return returnStr;
	}

	/**
	 * @param parent
	 * @param labelText
	 * @param ctl
	 * @param style
	 * @param gData
	 * @return
	 */
	private Control createLabelTextPair(Composite parent, String labelText,
			Control ctl, int style, int gData) {
		Label label = new Label(parent, SWT.NULL);
		label.setLayoutData(new GridData());
		label.setText(labelText);

		ctl = new Text(parent, style);
		ctl.setLayoutData(new GridData(gData));

		return ctl;
	}

	/**
	 * @return
	 */
	public String getDriverURL() {
		return this.mDriverInstance
				.getProperty(IDBDriverDefinitionConstants.URL_PROP_ID);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.preference.IPreferencePage#isValid()
	 */
	public boolean isValid() {
		if (getErrorMessage() != null) {
			return false;
		}
		return super.isValid();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.DialogPage#setErrorMessage(java.lang.String)
	 */
	public void setErrorMessage(String newMessage) {
		super.setErrorMessage(newMessage);
		isValid();
	}

	/**
	 * @return
	 */
	public String getDriverCategory() {
		return mDriverCategory;
	}

	
	/**
	 * Sets the driver category that should be used for displaying available
	 * driver defnitions.
	 * 
	 * @param driverCategory
	 */
	public void setDriverCategory(String driverCategory) {
		mDriverCategory = driverCategory;
		if (combo != null) {
			combo.setCategory(mDriverCategory);
		}
	}
}
