/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jgit.internal.storage.file;

import java.io.IOException;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.util.Collections;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import org.eclipse.jgit.annotations.NonNull;
import org.eclipse.jgit.internal.JGitText;
import org.eclipse.jgit.internal.storage.file.ByteArrayWindow;
import org.eclipse.jgit.internal.storage.file.ByteWindow;
import org.eclipse.jgit.internal.storage.file.DeltaBaseCache;
import org.eclipse.jgit.internal.storage.file.Pack;
import org.eclipse.jgit.storage.file.WindowCacheConfig;
import org.eclipse.jgit.storage.file.WindowCacheStats;
import org.eclipse.jgit.util.Monitoring;

public class WindowCache {
    private static final Random rng = new Random();
    private static volatile WindowCache cache;
    private static volatile int streamFileThreshold;
    private final CleanupQueue queue;
    private final int tableSize;
    private final AtomicLong clock;
    private final AtomicReferenceArray<Entry> table;
    private final Lock[] locks;
    private final ReentrantLock evictLock;
    private final int evictBatch;
    private final int maxFiles;
    private final long maxBytes;
    private final boolean mmap;
    private final int windowSizeShift;
    private final int windowSize;
    private final StatsRecorder statsRecorder;
    private final StatsRecorderImpl mbean;
    private final AtomicBoolean publishMBean = new AtomicBoolean();
    private boolean useStrongRefs;

    private static final int bits(int newSize) {
        if (newSize < 4096) {
            throw new IllegalArgumentException(JGitText.get().invalidWindowSize);
        }
        if (Integer.bitCount(newSize) != 1) {
            throw new IllegalArgumentException(JGitText.get().windowSizeMustBePowerOf2);
        }
        return Integer.numberOfTrailingZeros(newSize);
    }

    @Deprecated
    public static void reconfigure(WindowCacheConfig cfg) {
        WindowCache nc = new WindowCache(cfg);
        WindowCache oc = cache;
        if (oc != null) {
            oc.removeAll();
        }
        cache = nc;
        streamFileThreshold = cfg.getStreamFileThreshold();
        DeltaBaseCache.reconfigure(cfg);
    }

    static int getStreamFileThreshold() {
        return streamFileThreshold;
    }

    public static WindowCache getInstance() {
        return cache.publishMBeanIfNeeded();
    }

    static final ByteWindow get(Pack pack, long offset) throws IOException {
        WindowCache c = cache;
        ByteWindow r = c.getOrLoad(pack, c.toStart(offset));
        if (c != cache.publishMBeanIfNeeded()) {
            c.removeAll();
        }
        return r;
    }

    static final void purge(Pack pack) {
        cache.removeAll(pack);
    }

    private WindowCache(WindowCacheConfig cfg) {
        this.tableSize = WindowCache.tableSize(cfg);
        int lockCount = WindowCache.lockCount(cfg);
        if (this.tableSize < 1) {
            throw new IllegalArgumentException(JGitText.get().tSizeMustBeGreaterOrEqual1);
        }
        if (lockCount < 1) {
            throw new IllegalArgumentException(JGitText.get().lockCountMustBeGreaterOrEqual1);
        }
        this.clock = new AtomicLong(1L);
        this.table = new AtomicReferenceArray(this.tableSize);
        this.locks = new Lock[lockCount];
        for (int i = 0; i < this.locks.length; ++i) {
            this.locks[i] = new Lock();
        }
        this.evictLock = new ReentrantLock();
        int eb = (int)((double)this.tableSize * 0.1);
        if (64 < eb) {
            eb = 64;
        } else if (eb < 4) {
            eb = 4;
        }
        if (this.tableSize < eb) {
            eb = this.tableSize;
        }
        this.evictBatch = eb;
        this.maxFiles = cfg.getPackedGitOpenFiles();
        this.maxBytes = cfg.getPackedGitLimit();
        this.mmap = cfg.isPackedGitMMAP();
        this.windowSizeShift = WindowCache.bits(cfg.getPackedGitWindowSize());
        this.windowSize = 1 << this.windowSizeShift;
        this.useStrongRefs = cfg.isPackedGitUseStrongRefs();
        this.queue = this.useStrongRefs ? new StrongCleanupQueue(this) : new SoftCleanupQueue(this);
        this.mbean = new StatsRecorderImpl();
        this.statsRecorder = this.mbean;
        this.publishMBean.set(cfg.getExposeStatsViaJmx());
        if (this.maxFiles < 1) {
            throw new IllegalArgumentException(JGitText.get().openFilesMustBeAtLeast1);
        }
        if (this.maxBytes < (long)this.windowSize) {
            throw new IllegalArgumentException(JGitText.get().windowSizeMustBeLesserThanLimit);
        }
    }

    private WindowCache publishMBeanIfNeeded() {
        if (this.publishMBean.getAndSet(false)) {
            Monitoring.registerMBean(this.mbean, "block_cache");
        }
        return this;
    }

    public WindowCacheStats getStats() {
        return this.statsRecorder.getStats();
    }

    public void resetStats() {
        this.mbean.resetCounters();
    }

    private int hash(int packHash, long off) {
        return packHash + (int)(off >>> this.windowSizeShift);
    }

    private ByteWindow load(Pack pack, long offset) throws IOException {
        long startTime = System.nanoTime();
        if (pack.beginWindowCache()) {
            this.statsRecorder.recordOpenFiles(1);
        }
        try {
            if (this.mmap) {
                ByteWindow byteWindow = pack.mmap(offset, this.windowSize);
                return byteWindow;
            }
            ByteArrayWindow w = pack.read(offset, this.windowSize);
            this.statsRecorder.recordLoadSuccess(System.nanoTime() - startTime);
            ByteArrayWindow byteArrayWindow = w;
            return byteArrayWindow;
        }
        catch (IOException | Error | RuntimeException e) {
            this.close(pack);
            this.statsRecorder.recordLoadFailure(System.nanoTime() - startTime);
            throw e;
        }
        finally {
            this.statsRecorder.recordMisses(1);
        }
    }

    private PageRef<ByteWindow> createRef(Pack p, long o, ByteWindow v) {
        PageRef<ByteWindow> ref = this.useStrongRefs ? new StrongRef(p, o, v, this.queue) : new SoftRef(p, o, v, (SoftCleanupQueue)this.queue);
        this.statsRecorder.recordOpenBytes(ref.getPack(), ref.getSize());
        return ref;
    }

    private void clear(PageRef<ByteWindow> ref) {
        this.statsRecorder.recordOpenBytes(ref.getPack(), -ref.getSize());
        this.statsRecorder.recordEvictions(1);
        this.close(ref.getPack());
    }

    private void close(Pack pack) {
        if (pack.endWindowCache()) {
            this.statsRecorder.recordOpenFiles(-1);
        }
    }

    private boolean isFull() {
        return (long)this.maxFiles < this.mbean.getOpenFileCount() || this.maxBytes < this.mbean.getOpenByteCount();
    }

    private long toStart(long offset) {
        return offset >>> this.windowSizeShift << this.windowSizeShift;
    }

    private static int tableSize(WindowCacheConfig cfg) {
        int wsz = cfg.getPackedGitWindowSize();
        long limit = cfg.getPackedGitLimit();
        if (wsz <= 0) {
            throw new IllegalArgumentException(JGitText.get().invalidWindowSize);
        }
        if (limit < (long)wsz) {
            throw new IllegalArgumentException(JGitText.get().windowSizeMustBeLesserThanLimit);
        }
        return (int)Math.min(5L * (limit / (long)wsz) / 2L, 2000000000L);
    }

    private static int lockCount(WindowCacheConfig cfg) {
        return Math.max(cfg.getPackedGitOpenFiles(), 32);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ByteWindow getOrLoad(Pack pack, long position) throws IOException {
        int slot = this.slot(pack, position);
        Entry e1 = this.table.get(slot);
        ByteWindow v = this.scan(e1, pack, position);
        if (v != null) {
            this.statsRecorder.recordHits(1);
            return v;
        }
        Lock lock = this.lock(pack, position);
        synchronized (lock) {
            Entry n;
            Entry e2 = this.table.get(slot);
            if (e2 != e1 && (v = this.scan(e2, pack, position)) != null) {
                this.statsRecorder.recordHits(1);
                return v;
            }
            v = this.load(pack, position);
            PageRef<ByteWindow> ref = this.createRef(pack, position, v);
            this.hit(ref);
            while (!this.table.compareAndSet(slot, e2, n = new Entry(WindowCache.clean(e2), ref))) {
                e2 = this.table.get(slot);
            }
        }
        if (this.evictLock.tryLock()) {
            try {
                this.gc();
                this.evict();
            }
            finally {
                this.evictLock.unlock();
            }
        }
        return v;
    }

    private ByteWindow scan(Entry n, Pack pack, long position) {
        while (n != null) {
            PageRef<ByteWindow> r = n.ref;
            if (r.getPack() == pack && r.getPosition() == position) {
                ByteWindow v = r.get();
                if (v != null) {
                    this.hit(r);
                    return v;
                }
                n.kill();
                break;
            }
            n = n.next;
        }
        return null;
    }

    private void hit(PageRef r) {
        long c = this.clock.get();
        this.clock.compareAndSet(c, c + 1L);
        r.setLastAccess(c);
    }

    private void evict() {
        while (this.isFull()) {
            int ptr = rng.nextInt(this.tableSize);
            Entry old = null;
            int slot = 0;
            int b = this.evictBatch - 1;
            while (b >= 0) {
                if (this.tableSize <= ptr) {
                    ptr = 0;
                }
                Entry e = this.table.get(ptr);
                while (e != null) {
                    if (!(e.dead || old != null && e.ref.getLastAccess() >= old.ref.getLastAccess())) {
                        old = e;
                        slot = ptr;
                    }
                    e = e.next;
                }
                --b;
                ++ptr;
            }
            if (old == null) continue;
            old.kill();
            this.gc();
            Entry e1 = this.table.get(slot);
            this.table.compareAndSet(slot, e1, WindowCache.clean(e1));
        }
    }

    private void removeAll() {
        for (int s = 0; s < this.tableSize; ++s) {
            Entry e1;
            do {
                Entry e = e1 = this.table.get(s);
                while (e != null) {
                    e.kill();
                    e = e.next;
                }
            } while (!this.table.compareAndSet(s, e1, null));
        }
        this.gc();
    }

    private void removeAll(Pack pack) {
        for (int s = 0; s < this.tableSize; ++s) {
            Entry e1 = this.table.get(s);
            boolean hasDead = false;
            Entry e = e1;
            while (e != null) {
                if (e.ref.getPack() == pack) {
                    e.kill();
                    hasDead = true;
                } else if (e.dead) {
                    hasDead = true;
                }
                e = e.next;
            }
            if (!hasDead) continue;
            this.table.compareAndSet(s, e1, WindowCache.clean(e1));
        }
        this.gc();
    }

    private void gc() {
        this.queue.gc();
    }

    private int slot(Pack pack, long position) {
        return (this.hash(pack.hash, position) >>> 1) % this.tableSize;
    }

    private Lock lock(Pack pack, long position) {
        return this.locks[(this.hash(pack.hash, position) >>> 1) % this.locks.length];
    }

    private static Entry clean(Entry top) {
        while (top != null && top.dead) {
            top.ref.kill();
            top = top.next;
        }
        if (top == null) {
            return null;
        }
        Entry n = WindowCache.clean(top.next);
        return n == top.next ? top : new Entry(n, top.ref);
    }

    static {
        WindowCache.reconfigure(new WindowCacheConfig());
    }

    private static final class Lock {
        private Lock() {
        }
    }

    private static class StrongCleanupQueue
    implements CleanupQueue {
        private final WindowCache wc;
        private final ConcurrentLinkedQueue<PageRef<ByteWindow>> queue = new ConcurrentLinkedQueue();

        StrongCleanupQueue(WindowCache wc) {
            this.wc = wc;
        }

        @Override
        public boolean enqueue(PageRef<ByteWindow> r) {
            if (this.queue.contains(r)) {
                return false;
            }
            return this.queue.add(r);
        }

        @Override
        public void gc() {
            PageRef<ByteWindow> r;
            block0: while ((r = this.queue.poll()) != null) {
                Entry e1;
                this.wc.clear(r);
                int s = this.wc.slot(r.getPack(), r.getPosition());
                Entry n = e1 = (Entry)this.wc.table.get(s);
                while (n != null) {
                    if (n.ref == r) {
                        n.dead = true;
                        this.wc.table.compareAndSet(s, e1, WindowCache.clean(e1));
                        continue block0;
                    }
                    n = n.next;
                }
            }
        }
    }

    private static class SoftCleanupQueue
    extends ReferenceQueue<ByteWindow>
    implements CleanupQueue {
        private final WindowCache wc;

        SoftCleanupQueue(WindowCache cache) {
            this.wc = cache;
        }

        @Override
        public boolean enqueue(PageRef<ByteWindow> r) {
            return false;
        }

        @Override
        public void gc() {
            SoftRef r;
            block0: while ((r = (SoftRef)this.poll()) != null) {
                Entry e1;
                this.wc.clear(r);
                int s = this.wc.slot(r.getPack(), r.getPosition());
                Entry n = e1 = (Entry)this.wc.table.get(s);
                while (n != null) {
                    if (n.ref == r) {
                        n.dead = true;
                        this.wc.table.compareAndSet(s, e1, WindowCache.clean(e1));
                        continue block0;
                    }
                    n = n.next;
                }
            }
        }
    }

    private static interface CleanupQueue {
        public boolean enqueue(PageRef<ByteWindow> var1);

        public void gc();
    }

    private static class StrongRef
    implements PageRef<ByteWindow> {
        private ByteWindow referent;
        private final Pack pack;
        private final long position;
        private final int size;
        private long lastAccess;
        private CleanupQueue queue;

        protected StrongRef(Pack pack, long position, ByteWindow v, CleanupQueue queue) {
            this.pack = pack;
            this.position = position;
            this.referent = v;
            this.size = v.size();
            this.queue = queue;
        }

        @Override
        public Pack getPack() {
            return this.pack;
        }

        @Override
        public long getPosition() {
            return this.position;
        }

        @Override
        public int getSize() {
            return this.size;
        }

        @Override
        public long getLastAccess() {
            return this.lastAccess;
        }

        @Override
        public void setLastAccess(long time) {
            this.lastAccess = time;
        }

        @Override
        public ByteWindow get() {
            return this.referent;
        }

        @Override
        public boolean kill() {
            if (this.referent == null) {
                return false;
            }
            this.referent = null;
            return this.queue.enqueue(this);
        }

        @Override
        public boolean isStrongRef() {
            return true;
        }
    }

    private static class SoftRef
    extends SoftReference<ByteWindow>
    implements PageRef<ByteWindow> {
        private final Pack pack;
        private final long position;
        private final int size;
        private long lastAccess;

        protected SoftRef(Pack pack, long position, ByteWindow v, SoftCleanupQueue queue) {
            super(v, queue);
            this.pack = pack;
            this.position = position;
            this.size = v.size();
        }

        @Override
        public Pack getPack() {
            return this.pack;
        }

        @Override
        public long getPosition() {
            return this.position;
        }

        @Override
        public int getSize() {
            return this.size;
        }

        @Override
        public long getLastAccess() {
            return this.lastAccess;
        }

        @Override
        public void setLastAccess(long time) {
            this.lastAccess = time;
        }

        @Override
        public boolean kill() {
            return this.enqueue();
        }

        @Override
        public boolean isStrongRef() {
            return false;
        }
    }

    private static interface PageRef<T> {
        public T get();

        public boolean kill();

        public Pack getPack();

        public long getPosition();

        public int getSize();

        public long getLastAccess();

        public void setLastAccess(long var1);

        public boolean isStrongRef();
    }

    private static class Entry {
        final Entry next;
        final PageRef<ByteWindow> ref;
        volatile boolean dead;

        Entry(Entry n, PageRef<ByteWindow> r) {
            this.next = n;
            this.ref = r;
        }

        final void kill() {
            this.dead = true;
            this.ref.kill();
        }
    }

    static class StatsRecorderImpl
    implements StatsRecorder,
    WindowCacheStats {
        private final LongAdder hitCount = new LongAdder();
        private final LongAdder missCount = new LongAdder();
        private final LongAdder loadSuccessCount = new LongAdder();
        private final LongAdder loadFailureCount = new LongAdder();
        private final LongAdder totalLoadTime = new LongAdder();
        private final LongAdder evictionCount = new LongAdder();
        private final LongAdder openFileCount = new LongAdder();
        private final LongAdder openByteCount = new LongAdder();
        private final Map<String, LongAdder> openByteCountPerRepository = new ConcurrentHashMap<String, LongAdder>();

        @Override
        public void recordHits(int count) {
            this.hitCount.add(count);
        }

        @Override
        public void recordMisses(int count) {
            this.missCount.add(count);
        }

        @Override
        public void recordLoadSuccess(long loadTimeNanos) {
            this.loadSuccessCount.increment();
            this.totalLoadTime.add(loadTimeNanos);
        }

        @Override
        public void recordLoadFailure(long loadTimeNanos) {
            this.loadFailureCount.increment();
            this.totalLoadTime.add(loadTimeNanos);
        }

        @Override
        public void recordEvictions(int count) {
            this.evictionCount.add(count);
        }

        @Override
        public void recordOpenFiles(int delta) {
            this.openFileCount.add(delta);
        }

        @Override
        public void recordOpenBytes(Pack pack, int delta) {
            this.openByteCount.add(delta);
            String repositoryId = StatsRecorderImpl.repositoryId(pack);
            LongAdder la = this.openByteCountPerRepository.computeIfAbsent(repositoryId, k -> new LongAdder());
            la.add(delta);
            if (delta < 0) {
                this.openByteCountPerRepository.computeIfPresent(repositoryId, (k, v) -> v.longValue() == 0L ? null : v);
            }
        }

        private static String repositoryId(Pack pack) {
            return pack.getPackFile().getParentFile().getParentFile().getParent();
        }

        @Override
        public WindowCacheStats getStats() {
            return this;
        }

        @Override
        public long getHitCount() {
            return this.hitCount.sum();
        }

        @Override
        public long getMissCount() {
            return this.missCount.sum();
        }

        @Override
        public long getLoadSuccessCount() {
            return this.loadSuccessCount.sum();
        }

        @Override
        public long getLoadFailureCount() {
            return this.loadFailureCount.sum();
        }

        @Override
        public long getEvictionCount() {
            return this.evictionCount.sum();
        }

        @Override
        public long getTotalLoadTime() {
            return this.totalLoadTime.sum();
        }

        @Override
        public long getOpenFileCount() {
            return this.openFileCount.sum();
        }

        @Override
        public long getOpenByteCount() {
            return this.openByteCount.sum();
        }

        @Override
        public void resetCounters() {
            this.hitCount.reset();
            this.missCount.reset();
            this.loadSuccessCount.reset();
            this.loadFailureCount.reset();
            this.totalLoadTime.reset();
            this.evictionCount.reset();
        }

        @Override
        public Map<String, Long> getOpenByteCountPerRepository() {
            return Collections.unmodifiableMap(this.openByteCountPerRepository.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> ((LongAdder)e.getValue()).sum(), (u, v) -> v)));
        }
    }

    static interface StatsRecorder {
        public void recordHits(int var1);

        public void recordMisses(int var1);

        public void recordLoadSuccess(long var1);

        public void recordLoadFailure(long var1);

        public void recordEvictions(int var1);

        public void recordOpenFiles(int var1);

        public void recordOpenBytes(Pack var1, int var2);

        @NonNull
        public WindowCacheStats getStats();
    }
}

