/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.core.internal;

import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.LongAdder;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.CyclicTimeouts;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.io.Retainable;
import org.eclipse.jetty.io.RetainableByteBuffer;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.IteratingCallback;
import org.eclipse.jetty.util.NanoTime;
import org.eclipse.jetty.util.TypeUtil;
import org.eclipse.jetty.util.thread.AutoLock;
import org.eclipse.jetty.util.thread.Scheduler;
import org.eclipse.jetty.websocket.core.CloseStatus;
import org.eclipse.jetty.websocket.core.Frame;
import org.eclipse.jetty.websocket.core.exception.WebSocketException;
import org.eclipse.jetty.websocket.core.exception.WebSocketWriteTimeoutException;
import org.eclipse.jetty.websocket.core.internal.FrameEntry;
import org.eclipse.jetty.websocket.core.internal.Generator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FrameFlusher
extends IteratingCallback {
    public static final Frame FLUSH_FRAME = new Frame(-1){

        @Override
        public boolean isControlFrame() {
            return true;
        }
    };
    private static final Logger LOG = LoggerFactory.getLogger(FrameFlusher.class);
    private final AutoLock _lock = new AutoLock();
    private final LongAdder _messagesOut = new LongAdder();
    private final LongAdder _bytesOut = new LongAdder();
    private final ByteBufferPool _bufferPool;
    private final EndPoint _endPoint;
    private final int _bufferSize;
    private final Generator _generator;
    private final int _maxGather;
    private final CyclicTimeouts<Entry> _cyclicTimeouts;
    private final Deque<Entry> _queue = new ArrayDeque<Entry>();
    private final List<Entry> _currentEntries;
    private final List<Entry> _completedEntries = new ArrayList<Entry>();
    private final List<RetainableByteBuffer> _releasableBuffers = new ArrayList<RetainableByteBuffer>();
    private RetainableByteBuffer _batchBuffer;
    private boolean _canEnqueue = true;
    private Throwable _closedCause;
    private long _frameTimeout;
    private boolean _useDirectByteBuffers;

    public FrameFlusher(ByteBufferPool bufferPool, Scheduler scheduler, Generator generator, EndPoint endPoint, int bufferSize, int maxGather) {
        this._bufferPool = bufferPool;
        this._endPoint = endPoint;
        this._bufferSize = bufferSize;
        this._generator = Objects.requireNonNull(generator);
        this._maxGather = maxGather;
        this._currentEntries = new ArrayList<Entry>(maxGather);
        this._cyclicTimeouts = new CyclicTimeouts<Entry>(scheduler){
            private boolean _expired;
            {
                this._expired = false;
            }

            @Override
            protected boolean onExpired(Entry expirable) {
                this._expired = true;
                return false;
            }

            @Override
            protected Iterator<Entry> iterator() {
                return TypeUtil.concat(FrameFlusher.this._currentEntries.iterator(), FrameFlusher.this._queue.iterator());
            }

            @Override
            protected void iterate() {
                try (AutoLock l = FrameFlusher.this._lock.lock();){
                    super.iterate();
                }
                if (this._expired) {
                    FrameFlusher.this.abort(new WebSocketWriteTimeoutException("FrameFlusher Frame Write Timeout"));
                }
            }
        };
    }

    public boolean isUseDirectByteBuffers() {
        return this._useDirectByteBuffers;
    }

    public void setUseDirectByteBuffers(boolean useDirectByteBuffers) {
        this._useDirectByteBuffers = useDirectByteBuffers;
    }

    public boolean enqueue(Frame frame, Callback callback, boolean batch) {
        Entry entry = new Entry(frame, callback, batch);
        byte opCode = frame.getOpCode();
        Throwable error = null;
        boolean abort = false;
        ArrayList<Entry> failedEntries = null;
        CloseStatus closeStatus = null;
        try (AutoLock l = this._lock.lock();){
            if (!this._canEnqueue || this._closedCause != null) {
                error = this._closedCause == null ? new ClosedChannelException() : this._closedCause;
            } else {
                switch (opCode) {
                    case 8: {
                        closeStatus = CloseStatus.getCloseStatus(frame);
                        if (closeStatus.isAbnormal()) {
                            failedEntries = new ArrayList<Entry>(this._queue);
                            this._queue.clear();
                        }
                        this._queue.offerLast(entry);
                        this._canEnqueue = false;
                        break;
                    }
                    case 9: 
                    case 10: {
                        this._queue.offerFirst(entry);
                        break;
                    }
                    default: {
                        if (entry.isExpired()) {
                            error = new WebSocketWriteTimeoutException("FrameFlusher Frame Write Timeout");
                            abort = true;
                            break;
                        }
                        this._queue.offerLast(entry);
                    }
                }
                if (!abort) {
                    this._cyclicTimeouts.schedule(entry);
                }
            }
        }
        if (failedEntries != null) {
            WebSocketException failure = new WebSocketException("Flusher received abnormal CloseFrame: " + CloseStatus.codeString(closeStatus.getCode()), closeStatus.getCause());
            for (Entry e : failedEntries) {
                this.notifyCallbackFailure(e.callback, failure);
            }
        }
        if (error != null) {
            this.notifyCallbackFailure(callback, error);
            if (abort) {
                this.abort(error);
            }
            return false;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Enqueued {} to {}", (Object)entry, (Object)this);
        }
        return true;
    }

    public void onClose(Throwable cause) {
        try (AutoLock l = this._lock.lock();){
            this._closedCause = cause == null ? new ClosedChannelException() : cause;
        }
        this.abort(this._closedCause);
    }

    @Override
    protected IteratingCallback.Action process() throws Throwable {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Flushing {}", (Object)this);
        }
        boolean flush = false;
        try (AutoLock l = this._lock.lock();){
            if (this._closedCause != null) {
                throw this._closedCause;
            }
            while (!this._queue.isEmpty() && this._currentEntries.size() <= this._maxGather) {
                Entry entry = this._queue.poll();
                this._currentEntries.add(entry);
                if (entry.frame != FLUSH_FRAME) continue;
                flush = true;
                break;
            }
        }
        boolean canBatch = true;
        ArrayList<ByteBuffer> buffers = new ArrayList<ByteBuffer>(this._maxGather * 2 + 1);
        if (this._batchBuffer != null) {
            buffers.add(this._batchBuffer.getByteBuffer());
        }
        for (Entry entry : this._currentEntries) {
            boolean batch;
            if (entry.frame == FLUSH_FRAME) continue;
            this._messagesOut.increment();
            int batchSpace = this._batchBuffer == null ? this._bufferSize : BufferUtil.space(this._batchBuffer.getByteBuffer());
            boolean bl = batch = canBatch && entry.batch && !entry.frame.isControlFrame() && entry.frame.getPayloadLength() < this._bufferSize / 4 && batchSpace - 28 >= entry.frame.getPayloadLength();
            if (batch) {
                if (this._batchBuffer == null) {
                    this._batchBuffer = this.acquireBuffer(this._bufferSize);
                    buffers.add(this._batchBuffer.getByteBuffer());
                }
                this._generator.generateWholeFrame(entry.frame, this._batchBuffer.getByteBuffer());
                continue;
            }
            if (canBatch && this._batchBuffer != null && batchSpace >= 28) {
                this._generator.generateHeader(entry.frame, this._batchBuffer.getByteBuffer());
            } else {
                RetainableByteBuffer headerBuffer = this.acquireBuffer(28);
                this._releasableBuffers.add(headerBuffer);
                this._generator.generateHeader(entry.frame, headerBuffer.getByteBuffer());
                buffers.add(headerBuffer.getByteBuffer());
            }
            ByteBuffer payload = entry.frame.getPayload();
            if (BufferUtil.hasContent(payload)) {
                if (entry.frame.isMasked()) {
                    RetainableByteBuffer masked = this.acquireBuffer(entry.frame.getPayloadLength());
                    payload = masked.getByteBuffer();
                    this._releasableBuffers.add(masked);
                    this._generator.generatePayload(entry.frame, payload);
                }
                buffers.add(payload.slice());
            }
            canBatch = false;
            flush = true;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("{} processed {} entries flush={}: {}", this, this._currentEntries.size(), flush, this._currentEntries);
        }
        if (flush) {
            if (this._batchBuffer != null) {
                this._releasableBuffers.add(this._batchBuffer);
                this._batchBuffer = null;
            }
            int i = 0;
            int bytes = 0;
            ByteBuffer[] bufferArray = new ByteBuffer[buffers.size()];
            for (ByteBuffer bb : buffers) {
                bytes += bb.limit() - bb.position();
                bufferArray[i++] = bb;
            }
            this._bytesOut.add(bytes);
            this._endPoint.write((Callback)this, bufferArray);
            buffers.clear();
        } else {
            if (this._currentEntries.isEmpty()) {
                this.releaseAggregateIfEmpty();
                return IteratingCallback.Action.IDLE;
            }
            this.succeeded();
        }
        return IteratingCallback.Action.SCHEDULED;
    }

    private RetainableByteBuffer acquireBuffer(int capacity) {
        return this._bufferPool.acquire(capacity, this.isUseDirectByteBuffers());
    }

    private int getQueueSize() {
        try (AutoLock l = this._lock.lock();){
            int n = this._queue.size();
            return n;
        }
    }

    @Override
    protected void onSuccess() {
        try (AutoLock l = this._lock.lock();){
            assert (this._completedEntries.isEmpty());
            this._completedEntries.addAll(this._currentEntries);
            this._currentEntries.clear();
        }
        for (Entry entry : this._completedEntries) {
            if (entry.frame.getOpCode() == 8) {
                this._endPoint.shutdownOutput();
            }
            this.notifyCallbackSuccess(entry.callback);
        }
        this._completedEntries.clear();
        this._releasableBuffers.forEach(Retainable::release);
        this._releasableBuffers.clear();
        super.onSuccess();
    }

    @Override
    public void onCompleteFailure(Throwable failure) {
        if (this._batchBuffer != null) {
            this._batchBuffer.clear();
        }
        this.releaseAggregateIfEmpty();
        try (AutoLock l = this._lock.lock();){
            this._canEnqueue = false;
            if (this._closedCause == null) {
                this._closedCause = failure;
            } else if (this._closedCause != failure) {
                this._closedCause.addSuppressed(failure);
            }
            assert (this._completedEntries.isEmpty());
            this._completedEntries.addAll(this._queue);
            this._completedEntries.addAll(this._currentEntries);
            this._queue.clear();
            this._currentEntries.clear();
            this._cyclicTimeouts.destroy();
        }
        for (Entry entry : this._completedEntries) {
            this.notifyCallbackFailure(entry.callback, failure);
        }
        this._completedEntries.clear();
        this._releasableBuffers.forEach(Retainable::release);
        this._releasableBuffers.clear();
        this._endPoint.close(this._closedCause);
    }

    private void releaseAggregateIfEmpty() {
        if (this._batchBuffer != null && !this._batchBuffer.hasRemaining()) {
            this._batchBuffer.release();
            this._batchBuffer = null;
        }
    }

    protected void notifyCallbackSuccess(Callback callback) {
        block3: {
            try {
                if (callback != null) {
                    callback.succeeded();
                }
            }
            catch (Throwable x) {
                if (!LOG.isDebugEnabled()) break block3;
                LOG.debug("Exception while notifying success of callback {}", (Object)callback, (Object)x);
            }
        }
    }

    protected void notifyCallbackFailure(Callback callback, Throwable failure) {
        block3: {
            try {
                if (callback != null) {
                    callback.failed(failure);
                }
            }
            catch (Throwable x) {
                if (!LOG.isDebugEnabled()) break block3;
                LOG.debug("Exception while notifying failure of callback {}", (Object)callback, (Object)x);
            }
        }
    }

    public void setFrameWriteTimeout(long idleTimeout) {
        this._frameTimeout = idleTimeout;
    }

    public long getMessagesOut() {
        return this._messagesOut.longValue();
    }

    public long getBytesOut() {
        return this._bytesOut.longValue();
    }

    @Override
    public String toString() {
        return String.format("%s[queueSize=%d,aggregate=%s]", super.toString(), this.getQueueSize(), this._batchBuffer);
    }

    private class Entry
    extends FrameEntry
    implements CyclicTimeouts.Expirable {
        private final long _expiry;

        private Entry(Frame frame, Callback callback, boolean batch) {
            super(frame, callback, batch);
            this._expiry = CyclicTimeouts.Expirable.calcExpireNanoTime(FrameFlusher.this._frameTimeout);
        }

        @Override
        public long getExpireNanoTime() {
            return this._expiry;
        }

        public boolean isExpired() {
            return this._expiry != Long.MAX_VALUE && NanoTime.until(this._expiry) < 0L;
        }

        @Override
        public String toString() {
            return String.format("%s{%s,%s,batch=%b,expire=%s}", TypeUtil.toShortName(this.getClass()), this.frame, this.callback, this.batch, NanoTime.millisUntil(this._expiry));
        }
    }
}

