/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.ee11.annotations;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;
import org.eclipse.jetty.util.ExceptionUtil;
import org.eclipse.jetty.util.FileID;
import org.eclipse.jetty.util.StringUtil;
import org.eclipse.jetty.util.resource.Resource;
import org.eclipse.jetty.util.resource.ResourceFactory;
import org.objectweb.asm.AnnotationVisitor;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.FieldVisitor;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Opcodes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AnnotationParser {
    private static final Logger LOG = LoggerFactory.getLogger(AnnotationParser.class);
    private static final int ASM_VERSION = AnnotationParser.asmVersion();
    protected Map<String, URI> _parsedClassNames = new ConcurrentHashMap<String, URI>();
    private final int _asmVersion;

    private static int asmVersion() {
        Optional<Integer> asmVersion = ((Stream)Arrays.stream(Opcodes.class.getFields()).sequential()).filter(f -> f.getName().matches("ASM[0-9]+")).map(f -> f.getName().substring(3)).map(Integer::parseInt).max(Integer::compareTo);
        if (asmVersion.isEmpty()) {
            throw new IllegalStateException("Invalid " + Opcodes.class.getName());
        }
        int asmFieldId = asmVersion.get();
        try {
            String fieldName = "ASM" + asmFieldId;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Using ASM API from {}.{}", (Object)Opcodes.class.getName(), (Object)fieldName);
            }
            return (Integer)Opcodes.class.getField(fieldName).get(null);
        }
        catch (Throwable e) {
            throw new IllegalStateException(e);
        }
    }

    public static String normalize(String name) {
        if (name == null) {
            return null;
        }
        if (name.startsWith("L") && name.endsWith(";")) {
            name = name.substring(1, name.length() - 1);
        }
        if (name.endsWith(".class")) {
            name = name.substring(0, name.length() - ".class".length());
        }
        return StringUtil.replace((String)name, (char)'/', (char)'.');
    }

    public static String[] normalize(String[] list) {
        if (list == null) {
            return null;
        }
        String[] normalList = new String[list.length];
        int i = 0;
        for (String s : list) {
            normalList[i++] = AnnotationParser.normalize(s);
        }
        return normalList;
    }

    public AnnotationParser() {
        this(ASM_VERSION);
    }

    public AnnotationParser(int asmVersion) {
        if (asmVersion == 0) {
            asmVersion = ASM_VERSION;
        }
        this._asmVersion = asmVersion;
    }

    public void parse(Set<? extends Handler> handlers, Resource r) throws Exception {
        if (r == null) {
            return;
        }
        if (!r.exists()) {
            return;
        }
        if (FileID.isJavaArchive((Path)r.getPath())) {
            this.parseJar(handlers, r);
            return;
        }
        if (r.isDirectory()) {
            this.parseDir(handlers, r);
            return;
        }
        if (FileID.isClassFile((Path)r.getPath())) {
            this.parseClass(handlers, null, r.getPath());
        }
        if (LOG.isDebugEnabled()) {
            LOG.warn("Resource not able to be scanned for classes: {}", (Object)r);
        }
    }

    protected void parseDir(Set<? extends Handler> handlers, Resource dirResource) throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Scanning dir {}", (Object)dirResource);
        }
        assert (dirResource.isDirectory());
        ExceptionUtil.MultiException multiException = new ExceptionUtil.MultiException();
        for (Resource candidate : dirResource.getAllResources()) {
            Path relative;
            if (candidate.isDirectory() || (relative = dirResource.getPathTo(candidate)) == null || FileID.isHidden((Path)relative) || FileID.isMetaInfVersions((Path)relative) || FileID.isModuleInfoClass((Path)relative) || !FileID.isClassFile((Path)relative)) continue;
            try {
                this.parseClass(handlers, dirResource, candidate.getPath());
            }
            catch (Exception ex) {
                multiException.add((Throwable)new RuntimeException("Error scanning entry " + String.valueOf(ex), ex));
            }
        }
        multiException.ifExceptionThrow();
    }

    protected void parseJar(Set<? extends Handler> handlers, Resource jarResource) throws Exception {
        if (jarResource == null) {
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Scanning jar {}", (Object)jarResource);
        }
        try (ResourceFactory.Closeable resourceFactory = ResourceFactory.closeable();){
            Resource insideJarResource = resourceFactory.newJarFileResource(jarResource.getURI());
            this.parseDir(handlers, insideJarResource);
        }
    }

    protected void parseClass(Set<? extends Handler> handlers, Resource containingResource, Path classFile) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Parse class from {}", (Object)classFile.toUri());
        }
        URI location = classFile.toUri();
        try (InputStream in = Files.newInputStream(classFile, new OpenOption[0]);){
            ClassReader reader = new ClassReader(in);
            reader.accept((ClassVisitor)new MyClassVisitor(handlers, containingResource, this._asmVersion), 7);
            String classname = AnnotationParser.normalize(reader.getClassName());
            URI existing = this._parsedClassNames.putIfAbsent(classname, location);
            if (existing != null) {
                LOG.warn("{} scanned from multiple locations: {}, {}", new Object[]{classname, existing, location});
            }
        }
        catch (IOException | IllegalArgumentException e) {
            throw new IOException("Unable to parse class: " + String.valueOf(classFile.toUri()), e);
        }
    }

    Map<String, URI> getParsedClassNames() {
        return Collections.unmodifiableMap(this._parsedClassNames);
    }

    public static class MyClassVisitor
    extends ClassVisitor {
        final int _asmVersion;
        final Resource _containingResource;
        final Set<? extends Handler> _handlers;
        ClassInfo _ci;

        public MyClassVisitor(Set<? extends Handler> handlers, Resource containingResource, int asmVersion) {
            super(asmVersion);
            this._asmVersion = asmVersion;
            this._handlers = handlers;
            this._containingResource = containingResource;
        }

        public void visit(int version, int access, String name, String signature, String superName, String[] interfaces) {
            this._ci = new ClassInfo(this._containingResource, AnnotationParser.normalize(name), version, access, signature, AnnotationParser.normalize(superName), AnnotationParser.normalize(interfaces));
            for (Handler handler : this._handlers) {
                handler.handle(this._ci);
            }
        }

        public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
            String annotationName = AnnotationParser.normalize(desc);
            for (Handler handler : this._handlers) {
                handler.handle(this._ci, annotationName);
            }
            return null;
        }

        public MethodVisitor visitMethod(int access, String name, String methodDesc, String signature, String[] exceptions) {
            return new MyMethodVisitor(this._handlers, this._ci, access, name, methodDesc, signature, exceptions, this._asmVersion);
        }

        public FieldVisitor visitField(int access, String fieldName, String fieldType, String signature, Object value) {
            return new MyFieldVisitor(this._handlers, this._ci, access, fieldName, fieldType, signature, value, this._asmVersion);
        }
    }

    public static class MyFieldVisitor
    extends FieldVisitor {
        final FieldInfo _fieldInfo;
        final Set<? extends Handler> _handlers;

        public MyFieldVisitor(Set<? extends Handler> handlers, ClassInfo classInfo, int access, String fieldName, String fieldType, String signature, Object value, int asmVersion) {
            super(asmVersion);
            this._handlers = handlers;
            this._fieldInfo = new FieldInfo(classInfo, fieldName, access, fieldType, signature, value);
        }

        public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
            String annotationName = AnnotationParser.normalize(desc);
            for (Handler handler : this._handlers) {
                handler.handle(this._fieldInfo, annotationName);
            }
            return null;
        }
    }

    public static class MyMethodVisitor
    extends MethodVisitor {
        final MethodInfo _mi;
        final Set<? extends Handler> _handlers;

        public MyMethodVisitor(Set<? extends Handler> handlers, ClassInfo classInfo, int access, String name, String methodDesc, String signature, String[] exceptions, int asmVersion) {
            super(asmVersion);
            this._handlers = handlers;
            this._mi = new MethodInfo(classInfo, name, access, methodDesc, signature, exceptions);
        }

        public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
            String annotationName = AnnotationParser.normalize(desc);
            for (Handler handler : this._handlers) {
                handler.handle(this._mi, annotationName);
            }
            return null;
        }
    }

    public static abstract class AbstractHandler
    implements Handler {
        @Override
        public void handle(ClassInfo classInfo) {
        }

        @Override
        public void handle(MethodInfo methodInfo) {
        }

        @Override
        public void handle(FieldInfo fieldInfo) {
        }

        @Override
        public void handle(ClassInfo info, String annotationName) {
        }

        @Override
        public void handle(MethodInfo info, String annotationName) {
        }

        @Override
        public void handle(FieldInfo info, String annotationName) {
        }
    }

    public static interface Handler {
        public void handle(ClassInfo var1);

        public void handle(MethodInfo var1);

        public void handle(FieldInfo var1);

        public void handle(ClassInfo var1, String var2);

        public void handle(MethodInfo var1, String var2);

        public void handle(FieldInfo var1, String var2);
    }

    public static class FieldInfo {
        final ClassInfo _classInfo;
        final String _fieldName;
        final int _access;
        final String _fieldType;
        final String _signature;
        final Object _value;

        public FieldInfo(ClassInfo classInfo, String fieldName, int access, String fieldType, String signature, Object value) {
            this._classInfo = classInfo;
            this._fieldName = fieldName;
            this._access = access;
            this._fieldType = fieldType;
            this._signature = signature;
            this._value = value;
        }

        public ClassInfo getClassInfo() {
            return this._classInfo;
        }

        public String getFieldName() {
            return this._fieldName;
        }

        public int getAccess() {
            return this._access;
        }

        public String getFieldType() {
            return this._fieldType;
        }

        public String getSignature() {
            return this._signature;
        }

        public Object getValue() {
            return this._value;
        }
    }

    public static class MethodInfo {
        final ClassInfo _classInfo;
        final String _methodName;
        final int _access;
        final String _desc;
        final String _signature;
        final String[] _exceptions;

        public MethodInfo(ClassInfo classInfo, String methodName, int access, String desc, String signature, String[] exceptions) {
            this._classInfo = classInfo;
            this._methodName = methodName;
            this._access = access;
            this._desc = desc;
            this._signature = signature;
            this._exceptions = exceptions;
        }

        public ClassInfo getClassInfo() {
            return this._classInfo;
        }

        public String getMethodName() {
            return this._methodName;
        }

        public int getAccess() {
            return this._access;
        }

        public String getDesc() {
            return this._desc;
        }

        public String getSignature() {
            return this._signature;
        }

        public String[] getExceptions() {
            return this._exceptions;
        }
    }

    public static class ClassInfo {
        final Resource _containingResource;
        final String _className;
        final int _version;
        final int _access;
        final String _signature;
        final String _superName;
        final String[] _interfaces;

        public ClassInfo(Resource resource, String className, int version, int access, String signature, String superName, String[] interfaces) {
            this._containingResource = resource;
            this._className = className;
            this._version = version;
            this._access = access;
            this._signature = signature;
            this._superName = superName;
            this._interfaces = interfaces;
        }

        public String getClassName() {
            return this._className;
        }

        public int getVersion() {
            return this._version;
        }

        public int getAccess() {
            return this._access;
        }

        public String getSignature() {
            return this._signature;
        }

        public String getSuperName() {
            return this._superName;
        }

        public String[] getInterfaces() {
            return this._interfaces;
        }

        public Resource getContainingResource() {
            return this._containingResource;
        }
    }
}

