/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.common;

import java.nio.ByteBuffer;
import java.util.List;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.websocket.api.ProtocolException;
import org.eclipse.jetty.websocket.api.WebSocketBehavior;
import org.eclipse.jetty.websocket.api.WebSocketPolicy;
import org.eclipse.jetty.websocket.api.extensions.Extension;
import org.eclipse.jetty.websocket.api.extensions.Frame;
import org.eclipse.jetty.websocket.common.CloseInfo;

public class Generator {
    private static final Logger LOG = Log.getLogger(Generator.class);
    public static final int OVERHEAD = 28;
    private final WebSocketBehavior behavior;
    private final ByteBufferPool bufferPool;
    private boolean validating;
    private boolean rsv1InUse = false;
    private boolean rsv2InUse = false;
    private boolean rsv3InUse = false;

    public Generator(WebSocketPolicy policy, ByteBufferPool bufferPool) {
        this(policy, bufferPool, true);
    }

    public Generator(WebSocketPolicy policy, ByteBufferPool bufferPool, boolean validating) {
        this.behavior = policy.getBehavior();
        this.bufferPool = bufferPool;
        this.validating = validating;
    }

    public void assertFrameValid(Frame frame) {
        if (!this.validating) {
            return;
        }
        if (!this.rsv1InUse && frame.isRsv1()) {
            throw new ProtocolException("RSV1 not allowed to be set");
        }
        if (!this.rsv2InUse && frame.isRsv2()) {
            throw new ProtocolException("RSV2 not allowed to be set");
        }
        if (!this.rsv3InUse && frame.isRsv3()) {
            throw new ProtocolException("RSV3 not allowed to be set");
        }
        if (frame.getType().isControl()) {
            ByteBuffer payload;
            if (frame.getPayloadLength() > 125) {
                throw new ProtocolException("Invalid control frame payload length");
            }
            if (!frame.isFin()) {
                throw new ProtocolException("Control Frames must be FIN=true");
            }
            if (frame.getType().getOpCode() == 8 && (payload = frame.getPayload()) != null) {
                new CloseInfo(payload, true);
            }
        }
    }

    public void configureFromExtensions(List<? extends Extension> exts) {
        this.rsv1InUse = false;
        this.rsv2InUse = false;
        this.rsv3InUse = false;
        for (Extension extension : exts) {
            if (extension.isRsv1User()) {
                this.rsv1InUse = true;
            }
            if (extension.isRsv2User()) {
                this.rsv2InUse = true;
            }
            if (!extension.isRsv3User()) continue;
            this.rsv3InUse = true;
        }
    }

    public synchronized ByteBuffer generate(Frame frame) {
        int bufferSize = frame.getPayloadLength() + 28;
        return this.generate(bufferSize, frame);
    }

    public synchronized ByteBuffer generate(int windowSize, Frame frame) {
        if (windowSize < 28) {
            throw new IllegalArgumentException("Cannot have windowSize less than 28");
        }
        LOG.debug("{} Generate: {} (windowSize {})", new Object[]{this.behavior, frame, windowSize});
        ByteBuffer buffer = this.bufferPool.acquire(windowSize, true);
        BufferUtil.clearToFill(buffer);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Acquired Buffer (windowSize={}): {}", windowSize, BufferUtil.toDetailString(buffer));
        }
        int newlimit = Math.min(buffer.position() + windowSize, buffer.limit());
        buffer.limit(newlimit);
        LOG.debug("Buffer limited: {}", buffer);
        if (frame.remaining() == frame.getPayloadLength()) {
            this.assertFrameValid(frame);
            byte b = 0;
            if (frame.isFin()) {
                b = (byte)(b | 0x80);
            }
            if (frame.isRsv1()) {
                b = (byte)(b | 0x40);
            }
            if (frame.isRsv2()) {
                b = (byte)(b | 0x20);
            }
            if (frame.isRsv3()) {
                b = (byte)(b | 0x10);
            }
            byte opcode = frame.getOpCode();
            if (frame.isContinuation()) {
                opcode = 0;
            }
            b = (byte)(b | opcode & 0xF);
            buffer.put(b);
            b = 0;
            b = (byte)(b | (frame.isMasked() ? 128 : 0));
            int payloadLength = frame.getPayloadLength();
            if (payloadLength > 65535) {
                b = (byte)(b | 0x7F);
                buffer.put(b);
                buffer.put((byte)0);
                buffer.put((byte)0);
                buffer.put((byte)0);
                buffer.put((byte)0);
                buffer.put((byte)(payloadLength >> 24 & 0xFF));
                buffer.put((byte)(payloadLength >> 16 & 0xFF));
                buffer.put((byte)(payloadLength >> 8 & 0xFF));
                buffer.put((byte)(payloadLength & 0xFF));
            } else if (payloadLength >= 126) {
                b = (byte)(b | 0x7E);
                buffer.put(b);
                buffer.put((byte)(payloadLength >> 8));
                buffer.put((byte)(payloadLength & 0xFF));
            } else {
                b = (byte)(b | payloadLength & 0x7F);
                buffer.put(b);
            }
            if (frame.isMasked()) {
                buffer.put(frame.getMask());
            }
        }
        if (frame.hasPayload()) {
            int maskingStartPosition = buffer.position();
            int payloadOffset = frame.getPayload().position();
            int payloadStart = frame.getPayloadStart();
            BufferUtil.put(frame.getPayload(), buffer);
            if (frame.isMasked()) {
                int size = buffer.position() - maskingStartPosition;
                byte[] mask = frame.getMask();
                for (int i = 0; i < size; ++i) {
                    int posBuf = i + maskingStartPosition;
                    int posFrame = i + (payloadOffset - payloadStart);
                    byte b = buffer.get(posBuf);
                    b = (byte)(b ^ mask[posFrame % 4]);
                    buffer.put(posBuf, b);
                }
            }
        }
        BufferUtil.flipToFlush(buffer, 0);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Generated Buffer: {}", BufferUtil.toDetailString(buffer));
        }
        return buffer;
    }

    public ByteBufferPool getBufferPool() {
        return this.bufferPool;
    }

    public boolean isRsv1InUse() {
        return this.rsv1InUse;
    }

    public boolean isRsv2InUse() {
        return this.rsv2InUse;
    }

    public boolean isRsv3InUse() {
        return this.rsv3InUse;
    }

    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("Generator[");
        builder.append((Object)this.behavior);
        if (this.validating) {
            builder.append(",validating");
        }
        if (this.rsv1InUse) {
            builder.append(",+rsv1");
        }
        if (this.rsv2InUse) {
            builder.append(",+rsv2");
        }
        if (this.rsv3InUse) {
            builder.append(",+rsv3");
        }
        builder.append("]");
        return builder.toString();
    }
}

