/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.commandrouter.impl;

import io.fabric8.kubernetes.api.model.ContainerStatus;
import io.fabric8.kubernetes.api.model.Pod;
import io.fabric8.kubernetes.api.model.PodList;
import io.fabric8.kubernetes.client.DefaultKubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.Watch;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.WatcherException;
import io.fabric8.kubernetes.client.dsl.NonNamespaceOperation;
import io.vertx.core.Future;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import org.eclipse.hono.commandrouter.AdapterInstanceStatusService;
import org.eclipse.hono.util.AdapterInstanceStatus;
import org.eclipse.hono.util.CommandConstants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KubernetesBasedAdapterInstanceStatusService
implements AdapterInstanceStatusService {
    private static final Logger LOG = LoggerFactory.getLogger(KubernetesBasedAdapterInstanceStatusService.class);
    private static final String ADAPTER_NAME_MATCH = "adapter";
    private static final long WATCH_RECREATION_DELAY_MILLIS = 100L;
    private static final int MAX_LRU_MAP_ENTRIES = 200;
    private final KubernetesClient client;
    private final String namespace;
    private final AtomicInteger active = new AtomicInteger();
    private final Map<String, String> containerIdToPodNameMap = new ConcurrentHashMap<String, String>();
    private final Map<String, String> podNameToContainerIdMap = new ConcurrentHashMap<String, String>();
    private final Set<String> terminatedContainerIds = Collections.newSetFromMap(Collections.synchronizedMap(new LRUMap(200)));
    private final Set<String> suspectedContainerIds = Collections.newSetFromMap(Collections.synchronizedMap(new LRUMap(200)));
    private Watch watch;

    private KubernetesBasedAdapterInstanceStatusService() throws KubernetesClientException {
        this((KubernetesClient)new DefaultKubernetesClient());
    }

    KubernetesBasedAdapterInstanceStatusService(KubernetesClient client) throws KubernetesClientException {
        this.client = Objects.requireNonNull(client);
        this.namespace = Optional.ofNullable(client.getNamespace()).orElse("default");
        this.initAdaptersListAndWatch(this.namespace);
    }

    public static KubernetesBasedAdapterInstanceStatusService create() {
        if (KubernetesBasedAdapterInstanceStatusService.runningInKubernetes()) {
            try {
                return new KubernetesBasedAdapterInstanceStatusService();
            }
            catch (Exception e) {
                LOG.error("error creating KubernetesClient or pod watch: {}", (Object)e.toString());
            }
        }
        return null;
    }

    private static boolean runningInKubernetes() {
        return System.getenv("KUBERNETES_SERVICE_HOST") != null;
    }

    private void initAdaptersListAndWatch(String namespace) throws KubernetesClientException {
        if (this.active.get() == -1) {
            return;
        }
        this.containerIdToPodNameMap.clear();
        this.podNameToContainerIdMap.clear();
        ((PodList)((NonNamespaceOperation)this.client.pods().inNamespace(namespace)).list()).getItems().forEach(pod -> {
            LOG.trace("handle pod list result entry: {}", (Object)pod.getMetadata().getName());
            this.applyPodStatus(Watcher.Action.ADDED, (Pod)pod);
        });
        Iterator<String> iter = this.suspectedContainerIds.iterator();
        while (iter.hasNext()) {
            String suspectedContainerIdEntry = iter.next();
            if (!this.containerIdToPodNameMap.containsKey(suspectedContainerIdEntry)) {
                this.terminatedContainerIds.add(suspectedContainerIdEntry);
            }
            iter.remove();
        }
        this.watch = ((NonNamespaceOperation)this.client.pods().inNamespace(namespace)).watch((Object)new Watcher<Pod>(){

            public void eventReceived(Watcher.Action watchAction, Pod pod) {
                LOG.trace("event received: {}, pod: {}", (Object)watchAction, pod != null ? pod.getMetadata().getName() : null);
                KubernetesBasedAdapterInstanceStatusService.this.applyPodStatus(watchAction, pod);
            }

            public void onClose(WatcherException e) {
                KubernetesBasedAdapterInstanceStatusService.this.onWatcherClosed(e);
            }
        });
        this.active.compareAndExchange(0, 1);
        LOG.info("initialized list of active adapter containers: {}", this.containerIdToPodNameMap.size() <= 20 ? this.containerIdToPodNameMap : this.containerIdToPodNameMap.size() + " containers");
    }

    private void applyPodStatus(Watcher.Action watchAction, Pod pod) {
        if (watchAction == Watcher.Action.DELETED) {
            String podName = pod.getMetadata().getName();
            String shortContainerId = this.podNameToContainerIdMap.remove(podName);
            if (shortContainerId != null && this.containerIdToPodNameMap.remove(shortContainerId) != null) {
                LOG.info("removed entry for deleted pod [{}], container [{}]; active adapter containers now: {}", new Object[]{podName, shortContainerId, this.containerIdToPodNameMap.size()});
                this.terminatedContainerIds.add(shortContainerId);
                this.onAdapterContainerRemoved(shortContainerId);
            }
        } else if (watchAction == Watcher.Action.ERROR) {
            LOG.error("got ERROR watch action event");
        } else {
            pod.getStatus().getContainerStatuses().forEach(containerStatus -> this.applyContainerStatus(watchAction, pod, (ContainerStatus)containerStatus));
        }
    }

    private void applyContainerStatus(Watcher.Action watchAction, Pod pod, ContainerStatus containerStatus) {
        if (!containerStatus.getName().contains(ADAPTER_NAME_MATCH) || containerStatus.getContainerID() == null) {
            return;
        }
        String podName = pod.getMetadata().getName();
        String shortContainerId = KubernetesBasedAdapterInstanceStatusService.getShortContainerId(containerStatus.getContainerID());
        if (shortContainerId == null) {
            LOG.warn("unexpected format of container id [{}] in pod [{}]", (Object)containerStatus.getContainerID(), (Object)podName);
        } else if (containerStatus.getState() != null && containerStatus.getState().getTerminated() != null) {
            this.podNameToContainerIdMap.remove(podName, shortContainerId);
            if (this.containerIdToPodNameMap.remove(shortContainerId) != null) {
                LOG.info("removed entry for pod [{}] and terminated container [{}] (reason: '{}'); active adapter containers now: {}", new Object[]{podName, shortContainerId, containerStatus.getState().getTerminated().getReason(), this.containerIdToPodNameMap.size()});
                this.terminatedContainerIds.add(shortContainerId);
                this.onAdapterContainerRemoved(shortContainerId);
            }
        } else {
            String oldContainerId;
            if (watchAction == Watcher.Action.MODIFIED && (oldContainerId = this.podNameToContainerIdMap.get(podName)) != null && !oldContainerId.equals(shortContainerId)) {
                this.podNameToContainerIdMap.remove(podName, oldContainerId);
                if (this.containerIdToPodNameMap.remove(oldContainerId) != null) {
                    LOG.info("removed obsolete entry for pod [{}], container [{}]; active adapter containers now: {}", new Object[]{podName, oldContainerId, this.containerIdToPodNameMap.size()});
                    this.terminatedContainerIds.add(oldContainerId);
                    this.onAdapterContainerRemoved(oldContainerId);
                }
            }
            this.podNameToContainerIdMap.put(podName, shortContainerId);
            if (this.containerIdToPodNameMap.put(shortContainerId, podName) == null) {
                LOG.info("added entry for pod [{}], container [{}]; active adapter containers now: {}", new Object[]{podName, shortContainerId, this.containerIdToPodNameMap.size()});
                this.suspectedContainerIds.remove(shortContainerId);
                this.onAdapterContainerAdded(shortContainerId);
            }
        }
    }

    protected void onAdapterContainerAdded(String containerId) {
    }

    protected void onAdapterContainerRemoved(String containerId) {
    }

    private void onWatcherClosed(WatcherException cause) {
        if (this.active.compareAndExchange(1, 0) != 1) {
            return;
        }
        this.containerIdToPodNameMap.clear();
        this.podNameToContainerIdMap.clear();
        LOG.error("Watcher closed with error", (Throwable)cause);
        while (this.active.get() == 0) {
            try {
                Thread.sleep(100L);
                LOG.info("Recreating watch");
                this.initAdaptersListAndWatch(this.namespace);
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
            }
            catch (Exception ex) {
                LOG.error("error re-initializing adapter list and pod watch", (Throwable)cause);
            }
        }
    }

    public Future<Void> start() {
        return Future.succeededFuture();
    }

    public Future<Void> stop() {
        LOG.trace("stopping status service");
        if (this.active.getAndSet(-1) == -1) {
            return Future.succeededFuture();
        }
        Watch w = this.watch;
        if (w != null) {
            w.close();
        }
        this.client.close();
        return Future.succeededFuture();
    }

    public AdapterInstanceStatus getStatus(String adapterInstanceId) {
        if (this.active.get() != 1) {
            LOG.debug("no status info available for adapter instance id [{}]; service not active", (Object)adapterInstanceId);
            return AdapterInstanceStatus.UNKNOWN;
        }
        Matcher matcher = CommandConstants.KUBERNETES_ADAPTER_INSTANCE_ID_PATTERN.matcher(adapterInstanceId);
        if (!matcher.matches()) {
            return AdapterInstanceStatus.UNKNOWN;
        }
        String shortContainerId = matcher.group(1);
        String podName = this.containerIdToPodNameMap.get(shortContainerId);
        if (podName != null) {
            LOG.trace("found alive container in pod [{}] for adapter instance id [{}]", (Object)podName, (Object)adapterInstanceId);
            return AdapterInstanceStatus.ALIVE;
        }
        if (this.terminatedContainerIds.contains(shortContainerId)) {
            LOG.debug("container already terminated for adapter instance id [{}]", (Object)adapterInstanceId);
            return AdapterInstanceStatus.DEAD;
        }
        LOG.debug("no container found for adapter instance id [{}]", (Object)adapterInstanceId);
        this.suspectedContainerIds.add(shortContainerId);
        return AdapterInstanceStatus.SUSPECTED_DEAD;
    }

    Optional<Set<String>> getActiveAdapterInstanceContainerIds() {
        if (this.active.get() != 1) {
            return Optional.empty();
        }
        return Optional.of(new HashSet<String>(this.containerIdToPodNameMap.keySet()));
    }

    static String getShortContainerId(String containerId) {
        int lastSlashIndex = containerId.lastIndexOf(47);
        if (lastSlashIndex == -1) {
            return null;
        }
        String fullId = containerId.substring(lastSlashIndex + 1);
        if (fullId.length() < 12) {
            return null;
        }
        return fullId.substring(0, 12);
    }

    private static class LRUMap
    extends LinkedHashMap<String, Boolean> {
        private final int maxEntries;

        LRUMap(int maxEntries) {
            super(16, 0.75f, true);
            this.maxEntries = maxEntries;
        }

        @Override
        protected boolean removeEldestEntry(Map.Entry<String, Boolean> eldest) {
            return this.size() > this.maxEntries;
        }
    }
}

