/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.adapter.auth.device.x509;

import io.vertx.core.Future;
import io.vertx.core.Vertx;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.PKIXParameters;
import java.security.cert.PKIXRevocationChecker;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.eclipse.hono.adapter.auth.device.x509.OCSPNonceExtension;
import org.eclipse.hono.service.auth.X509CertificateChainValidator;
import org.eclipse.hono.util.RevocableTrustAnchor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DeviceCertificateValidator
implements X509CertificateChainValidator {
    private static final Logger LOG = LoggerFactory.getLogger(DeviceCertificateValidator.class);
    private final Vertx vertx;

    public DeviceCertificateValidator(Vertx vertx) {
        Objects.requireNonNull(vertx);
        this.vertx = vertx;
    }

    public Future<Void> validate(List<X509Certificate> chain, TrustAnchor trustAnchor) {
        Objects.requireNonNull(chain);
        Objects.requireNonNull(trustAnchor);
        return this.validate(chain, Set.of(trustAnchor));
    }

    public Future<Void> validate(List<X509Certificate> chain, Set<TrustAnchor> trustAnchors) {
        Objects.requireNonNull(chain);
        Objects.requireNonNull(trustAnchors);
        if (chain.isEmpty()) {
            throw new IllegalArgumentException("certificate chain must not be empty");
        }
        if (trustAnchors.isEmpty()) {
            throw new IllegalArgumentException("trust anchor list must not be empty");
        }
        return this.vertx.executeBlocking(() -> {
            this.validateAnchors(chain, trustAnchors);
            return null;
        }, false);
    }

    private void validateAnchors(List<X509Certificate> chain, Set<TrustAnchor> trustAnchors) throws CertificateException {
        GeneralSecurityException lastException = null;
        for (TrustAnchor anchor : trustAnchors) {
            try {
                this.validateSingleAnchor(chain, anchor);
                return;
            }
            catch (CertPathValidatorException e) {
                lastException = e;
                if (e.getReason() != CertPathValidatorException.BasicReason.REVOKED && e.getReason() != CertPathValidatorException.BasicReason.UNDETERMINED_REVOCATION_STATUS) continue;
                LOG.warn("Certificate [subject DN: {}] revocation check failed.", (Object)chain.get(0).getSubjectX500Principal().getName(), (Object)e);
                break;
            }
            catch (GeneralSecurityException e) {
                lastException = e;
            }
        }
        LOG.debug("validation of device certificate [subject DN: {}] failed", (Object)chain.get(0).getSubjectX500Principal().getName(), (Object)lastException);
        if (lastException instanceof CertificateException) {
            CertificateException certException = (CertificateException)lastException;
            throw certException;
        }
        throw new CertificateException("validation of device certificate failed", lastException);
    }

    private void validateSingleAnchor(List<X509Certificate> chain, TrustAnchor anchor) throws GeneralSecurityException {
        PKIXParameters params = new PKIXParameters(Set.of(anchor));
        CertificateFactory factory = CertificateFactory.getInstance("X.509");
        CertPath path = factory.generateCertPath(chain);
        CertPathValidator validator = CertPathValidator.getInstance("PKIX");
        this.configureRevocationCheck(validator, params, anchor);
        validator.validate(path, params);
        LOG.debug("validation of device certificate [subject DN: {}] succeeded", (Object)chain.get(0).getSubjectX500Principal().getName());
    }

    private void configureRevocationCheck(CertPathValidator validator, PKIXParameters parameters, TrustAnchor trustAnchor) throws CertificateException {
        RevocableTrustAnchor revocableTrustAnchor;
        if (trustAnchor instanceof RevocableTrustAnchor && (revocableTrustAnchor = (RevocableTrustAnchor)trustAnchor).isOcspEnabled()) {
            PKIXRevocationChecker revocationChecker = (PKIXRevocationChecker)validator.getRevocationChecker();
            EnumSet<PKIXRevocationChecker.Option> options = EnumSet.noneOf(PKIXRevocationChecker.Option.class);
            parameters.setRevocationEnabled(true);
            if (revocableTrustAnchor.isOcspEnabled()) {
                if (revocableTrustAnchor.getOcspResponderUri() != null) {
                    revocationChecker.setOcspResponder(revocableTrustAnchor.getOcspResponderUri());
                }
                if (revocableTrustAnchor.getOcspResponderCert() != null) {
                    revocationChecker.setOcspResponderCert(revocableTrustAnchor.getOcspResponderCert());
                }
                options.add(PKIXRevocationChecker.Option.ONLY_END_ENTITY);
                if (revocableTrustAnchor.isOcspNonceEnabled()) {
                    try {
                        revocationChecker.setOcspExtensions(List.of(new OCSPNonceExtension()));
                    }
                    catch (IOException e) {
                        throw new CertificateException("Cannot process OCSP nonce.", e);
                    }
                }
                options.add(PKIXRevocationChecker.Option.NO_FALLBACK);
            }
            revocationChecker.setOptions(options);
            parameters.addCertPathChecker(revocationChecker);
        } else {
            parameters.setRevocationEnabled(false);
        }
    }
}

