/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.adapter.auth.device;

import io.opentracing.Span;
import io.opentracing.SpanContext;
import io.opentracing.Tracer;
import io.opentracing.noop.NoopTracerFactory;
import io.opentracing.tag.Tags;
import io.vertx.core.Future;
import io.vertx.core.json.JsonObject;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.eclipse.hono.adapter.auth.device.DeviceCertificateValidator;
import org.eclipse.hono.adapter.auth.device.X509Authentication;
import org.eclipse.hono.client.ClientErrorException;
import org.eclipse.hono.client.registry.TenantClient;
import org.eclipse.hono.service.auth.X509CertificateChainValidator;
import org.eclipse.hono.tracing.TracingHelper;
import org.eclipse.hono.util.TenantObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TenantServiceBasedX509Authentication
implements X509Authentication {
    private static final Logger log = LoggerFactory.getLogger(TenantServiceBasedX509Authentication.class);
    private final Tracer tracer;
    private final TenantClient tenantClient;
    private final X509CertificateChainValidator certPathValidator;

    public TenantServiceBasedX509Authentication(TenantClient tenantClient) {
        this(tenantClient, (Tracer)NoopTracerFactory.create());
    }

    public TenantServiceBasedX509Authentication(TenantClient tenantClient, Tracer tracer) {
        this(tenantClient, tracer, new DeviceCertificateValidator());
    }

    public TenantServiceBasedX509Authentication(TenantClient tenantClient, Tracer tracer, X509CertificateChainValidator certPathValidator) {
        this.tenantClient = Objects.requireNonNull(tenantClient);
        this.tracer = Objects.requireNonNull(tracer);
        this.certPathValidator = Objects.requireNonNull(certPathValidator);
    }

    @Override
    public Future<JsonObject> validateClientCertificate(Certificate[] path, SpanContext spanContext) {
        Objects.requireNonNull(path);
        Span span = TracingHelper.buildChildSpan((Tracer)this.tracer, (SpanContext)spanContext, (String)"validate device certificate", (String)this.getClass().getSimpleName()).withTag(Tags.SPAN_KIND.getKey(), "client").start();
        return this.getX509CertificatePath(path).compose(x509chain -> {
            X509Certificate deviceCert = (X509Certificate)x509chain.get(0);
            HashMap<String, String> detail = new HashMap<String, String>(4);
            detail.put("subject DN", deviceCert.getSubjectX500Principal().getName("RFC2253"));
            detail.put("issuer DN", deviceCert.getIssuerX500Principal().getName("RFC2253"));
            detail.put("not before", deviceCert.getNotBefore().toString());
            detail.put("not after", deviceCert.getNotAfter().toString());
            span.log(detail);
            Future<TenantObject> tenantTracker = this.getTenant(deviceCert, span);
            return tenantTracker.compose(tenant -> {
                Set trustAnchors = tenant.getTrustAnchors();
                if (trustAnchors.isEmpty()) {
                    return Future.failedFuture((Throwable)new ClientErrorException(tenant.getTenantId(), 401, "no valid trust anchors defined for tenant"));
                }
                if (log.isTraceEnabled()) {
                    StringBuilder b = new StringBuilder("found tenant [tenant-id: ").append(tenant.getTenantId()).append("]").append(" for client certificate [subject-dn: ").append(deviceCert.getSubjectX500Principal().getName("RFC2253")).append(", issuer-dn: ").append(deviceCert.getIssuerX500Principal().getName("RFC2253")).append("]").append(System.lineSeparator()).append("with trust anchors:").append(System.lineSeparator());
                    trustAnchors.stream().forEach(ta -> b.append("Trust Anchor [subject-dn: ").append(ta.getCAName()).append("]"));
                    log.trace(b.toString());
                }
                List<X509Certificate> chainToValidate = List.of(deviceCert);
                return this.certPathValidator.validate(chainToValidate, trustAnchors).recover(t -> Future.failedFuture((Throwable)new ClientErrorException(tenant.getTenantId(), 401, t.getMessage(), t)));
            }).compose(ok -> this.getCredentials((List<X509Certificate>)x509chain, (TenantObject)tenantTracker.result()));
        }).map(authInfo -> {
            span.log("certificate validated successfully");
            span.finish();
            return authInfo;
        }).recover(t -> {
            log.debug("validation of client certificate failed", t);
            TracingHelper.logError((Span)span, (Throwable)t);
            span.finish();
            return Future.failedFuture((Throwable)t);
        });
    }

    private Future<TenantObject> getTenant(X509Certificate clientCert, Span span) {
        return this.tenantClient.get(clientCert.getIssuerX500Principal(), span.context());
    }

    private Future<List<X509Certificate>> getX509CertificatePath(Certificate[] clientPath) {
        LinkedList<X509Certificate> path = new LinkedList<X509Certificate>();
        for (Certificate cert : clientPath) {
            if (!(cert instanceof X509Certificate)) {
                log.info("cannot authenticate device using unsupported certificate type [{}]", (Object)cert.getClass().getName());
                return Future.failedFuture((Throwable)new ClientErrorException(401));
            }
            path.add((X509Certificate)cert);
        }
        return Future.succeededFuture(path);
    }

    protected Future<JsonObject> getCredentials(List<X509Certificate> clientCertPath, TenantObject tenant) {
        X509Certificate deviceCert = clientCertPath.get(0);
        String subjectDn = deviceCert.getSubjectX500Principal().getName("RFC2253");
        log.debug("authenticating device of tenant [{}] using X509 certificate [subject DN: {}]", (Object)tenant.getTenantId(), (Object)subjectDn);
        JsonObject authInfo = new JsonObject().put("subject-dn", subjectDn).put("tenant-id", tenant.getTenantId());
        String issuerDn = deviceCert.getIssuerX500Principal().getName("RFC2253");
        if (tenant.isAutoProvisioningEnabled(issuerDn)) {
            try {
                authInfo.put("client-certificate", deviceCert.getEncoded());
            }
            catch (CertificateEncodingException e) {
                log.error("Encoding of device certificate failed [subject DN: {}]", (Object)subjectDn, (Object)e);
                return Future.failedFuture((Throwable)e);
            }
        }
        return Future.succeededFuture((Object)authInfo);
    }
}

