/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.elements;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import org.eclipse.californium.elements.AddressEndpointContext;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.EndpointContextMatcher;
import org.eclipse.californium.elements.EndpointMismatchException;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.RawDataChannel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UDPConnector
implements Connector {
    public static final Logger LOGGER = LoggerFactory.getLogger((String)UDPConnector.class.getName());
    public static final int UNDEFINED = 0;
    static final ThreadGroup ELEMENTS_THREAD_GROUP = new ThreadGroup("Californium/Elements");
    private volatile boolean running;
    private DatagramSocket socket;
    private final InetSocketAddress localAddr;
    private List<Thread> receiverThreads;
    private List<Thread> senderThreads;
    private final BlockingQueue<RawData> outgoing;
    private volatile EndpointContextMatcher endpointContextMatcher;
    private RawDataChannel receiver;
    private int receiveBufferSize = 0;
    private int sendBufferSize = 0;
    private int senderCount = 1;
    private int receiverCount = 1;
    private int receiverPacketSize = 2048;

    public UDPConnector() {
        this(null);
    }

    public UDPConnector(InetSocketAddress address) {
        this.localAddr = address == null ? new InetSocketAddress(0) : address;
        this.running = false;
        this.outgoing = new LinkedBlockingQueue<RawData>();
    }

    @Override
    public synchronized void start() throws IOException {
        int i;
        if (this.running) {
            return;
        }
        this.socket = new DatagramSocket(this.localAddr.getPort(), this.localAddr.getAddress());
        this.running = true;
        if (this.receiveBufferSize != 0) {
            this.socket.setReceiveBufferSize(this.receiveBufferSize);
        }
        this.receiveBufferSize = this.socket.getReceiveBufferSize();
        if (this.sendBufferSize != 0) {
            this.socket.setSendBufferSize(this.sendBufferSize);
        }
        this.sendBufferSize = this.socket.getSendBufferSize();
        LOGGER.info("UDPConnector starts up {} sender threads and {} receiver threads", new Object[]{this.senderCount, this.receiverCount});
        this.receiverThreads = new LinkedList<Thread>();
        for (i = 0; i < this.receiverCount; ++i) {
            this.receiverThreads.add(new Receiver("UDP-Receiver-" + this.localAddr + "[" + i + "]"));
        }
        this.senderThreads = new LinkedList<Thread>();
        for (i = 0; i < this.senderCount; ++i) {
            this.senderThreads.add(new Sender("UDP-Sender-" + this.localAddr + "[" + i + "]"));
        }
        for (Thread t : this.receiverThreads) {
            t.start();
        }
        for (Thread t : this.senderThreads) {
            t.start();
        }
        if (LOGGER.isInfoEnabled()) {
            String startupMsg = "UDPConnector listening on " + this.socket.getLocalSocketAddress() + ", recv buf = " + this.receiveBufferSize + ", send buf = " + this.sendBufferSize + ", recv packet size = " + this.receiverPacketSize;
            LOGGER.info(startupMsg);
        }
    }

    @Override
    public synchronized void stop() {
        if (!this.running) {
            return;
        }
        this.running = false;
        if (this.senderThreads != null) {
            for (Thread t : this.senderThreads) {
                t.interrupt();
            }
            this.senderThreads.clear();
            this.senderThreads = null;
        }
        if (this.receiverThreads != null) {
            for (Thread t : this.receiverThreads) {
                t.interrupt();
            }
            this.receiverThreads.clear();
            this.receiverThreads = null;
        }
        this.outgoing.clear();
        String address = this.localAddr.toString();
        if (this.socket != null) {
            address = this.socket.getLocalSocketAddress().toString();
            this.socket.close();
            this.socket = null;
        }
        LOGGER.info("UDPConnector on [{}] has stopped.", (Object)address);
    }

    @Override
    public void destroy() {
        this.stop();
    }

    @Override
    public void send(RawData msg) {
        if (msg == null) {
            throw new NullPointerException("Message must not be null");
        }
        this.outgoing.add(msg);
    }

    @Override
    public void setRawDataReceiver(RawDataChannel receiver) {
        this.receiver = receiver;
    }

    @Override
    public void setEndpointContextMatcher(EndpointContextMatcher matcher) {
        this.endpointContextMatcher = matcher;
    }

    @Override
    public InetSocketAddress getAddress() {
        if (this.socket == null) {
            return this.localAddr;
        }
        return new InetSocketAddress(this.socket.getLocalAddress(), this.socket.getLocalPort());
    }

    public void setReceiveBufferSize(int size) {
        this.receiveBufferSize = size;
    }

    public int getReceiveBufferSize() {
        return this.receiveBufferSize;
    }

    public void setSendBufferSize(int size) {
        this.sendBufferSize = size;
    }

    public int getSendBufferSize() {
        return this.sendBufferSize;
    }

    public void setReceiverThreadCount(int count) {
        this.receiverCount = count;
    }

    public int getReceiverThreadCount() {
        return this.receiverCount;
    }

    public void setSenderThreadCount(int count) {
        this.senderCount = count;
    }

    public int getSenderThreadCount() {
        return this.senderCount;
    }

    public void setReceiverPacketSize(int size) {
        this.receiverPacketSize = size;
    }

    public int getReceiverPacketSize() {
        return this.receiverPacketSize;
    }

    @Override
    public String getProtocol() {
        return "UDP";
    }

    public String toString() {
        return this.getProtocol() + "-" + this.getAddress();
    }

    private class Sender
    extends NetworkStageThread {
        private DatagramPacket datagram;

        private Sender(String name) {
            super(name);
            this.datagram = new DatagramPacket(new byte[0], 0);
        }

        @Override
        protected void work() throws InterruptedException {
            RawData raw = (RawData)UDPConnector.this.outgoing.take();
            try {
                EndpointContextMatcher endpointMatcher = UDPConnector.this.endpointContextMatcher;
                if (endpointMatcher != null && !endpointMatcher.isToBeSent(raw.getEndpointContext(), null)) {
                    LOGGER.warn("UDPConnector ({}) drops {} bytes to {}:{}", new Object[]{UDPConnector.this.socket.getLocalSocketAddress(), this.datagram.getLength(), this.datagram.getAddress(), this.datagram.getPort()});
                    raw.onError(new EndpointMismatchException());
                    return;
                }
                this.datagram.setData(raw.getBytes());
                this.datagram.setAddress(raw.getAddress());
                this.datagram.setPort(raw.getPort());
                LOGGER.debug("UDPConnector ({}) sends {} bytes to {}:{}", new Object[]{this, this.datagram.getLength(), this.datagram.getAddress(), this.datagram.getPort()});
                UDPConnector.this.socket.send(this.datagram);
                raw.onSent();
            }
            catch (IOException ex) {
                raw.onError(ex);
            }
        }
    }

    private class Receiver
    extends NetworkStageThread {
        private DatagramPacket datagram;
        private int size;

        private Receiver(String name) {
            super(name);
            this.size = UDPConnector.this.receiverPacketSize;
            this.datagram = new DatagramPacket(new byte[this.size], this.size);
        }

        @Override
        protected void work() throws IOException {
            this.datagram.setLength(this.size);
            UDPConnector.this.socket.receive(this.datagram);
            LOGGER.debug("UDPConnector ({}) received {} bytes from {}:{}", new Object[]{UDPConnector.this.socket.getLocalSocketAddress(), this.datagram.getLength(), this.datagram.getAddress(), this.datagram.getPort()});
            byte[] bytes = Arrays.copyOfRange(this.datagram.getData(), this.datagram.getOffset(), this.datagram.getLength());
            RawData msg = RawData.inbound(bytes, new AddressEndpointContext(this.datagram.getAddress(), this.datagram.getPort()), false);
            UDPConnector.this.receiver.receiveData(msg);
        }
    }

    private abstract class NetworkStageThread
    extends Thread {
        protected NetworkStageThread(String name) {
            super(ELEMENTS_THREAD_GROUP, name);
            this.setDaemon(true);
        }

        @Override
        public void run() {
            LOGGER.debug("Starting network stage thread [{}]", (Object)this.getName());
            while (true) {
                try {
                    do {
                        this.work();
                    } while (UDPConnector.this.running);
                    LOGGER.debug("Network stage thread [{}] was stopped successfully", (Object)this.getName());
                }
                catch (Throwable t) {
                    if (UDPConnector.this.running) {
                        LOGGER.error("Exception in network stage thread [{}]:", (Object)this.getName(), (Object)t);
                        continue;
                    }
                    LOGGER.debug("Network stage thread [{}] was stopped successfully at:", (Object)this.getName(), (Object)t);
                }
                break;
            }
        }

        protected abstract void work() throws Exception;
    }
}

