/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.ApplicationMessage;
import org.eclipse.californium.scandium.dtls.ChangeCipherSpecMessage;
import org.eclipse.californium.scandium.dtls.ContentType;
import org.eclipse.californium.scandium.dtls.DTLSMessage;
import org.eclipse.californium.scandium.dtls.DTLSSession;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.ProtocolVersion;
import org.eclipse.californium.scandium.dtls.cipher.CCMBlockCipher;
import org.eclipse.californium.scandium.dtls.cipher.CipherSuite;
import org.eclipse.californium.scandium.util.ByteArrayUtils;
import org.eclipse.californium.scandium.util.DatagramReader;
import org.eclipse.californium.scandium.util.DatagramWriter;

public class Record {
    protected static final Logger LOGGER = Logger.getLogger(Record.class.getCanonicalName());
    private static final int CONTENT_TYPE_BITS = 8;
    private static final int VERSION_BITS = 8;
    private static final int EPOCH_BITS = 16;
    private static final int SEQUENCE_NUMBER_BITS = 48;
    private static final int LENGTH_BITS = 16;
    private ContentType type = null;
    private ProtocolVersion version = new ProtocolVersion();
    private int epoch = -1;
    private long sequenceNumber;
    private int length = 0;
    private DTLSMessage fragment = null;
    private byte[] fragmentBytes = null;
    private DTLSSession session;

    public Record(ContentType type, ProtocolVersion version, int epoch, long sequenceNumber, int length, byte[] fragmentBytes) {
        this.type = type;
        this.version = version;
        this.epoch = epoch;
        this.sequenceNumber = sequenceNumber;
        this.length = length;
        this.fragmentBytes = fragmentBytes;
    }

    public Record(ContentType type, int epoch, int sequenceNumber, DTLSMessage fragment, DTLSSession session) {
        this.type = type;
        this.epoch = epoch;
        this.sequenceNumber = sequenceNumber;
        this.session = session;
        this.setFragment(fragment);
    }

    public byte[] toByteArray() {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.type.getCode(), 8);
        writer.write(this.version.getMajor(), 8);
        writer.write(this.version.getMinor(), 8);
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        this.length = this.fragmentBytes.length;
        writer.write(this.length, 16);
        writer.writeBytes(this.fragmentBytes);
        return writer.toByteArray();
    }

    public static List<Record> fromByteArray(byte[] byteArray) {
        ArrayList<Record> records = new ArrayList<Record>();
        DatagramReader reader = new DatagramReader(byteArray);
        while (reader.bytesAvailable()) {
            int type = reader.read(8);
            ContentType contentType = ContentType.getTypeByValue(type);
            if (contentType == null) {
                if (!LOGGER.isLoggable(Level.WARNING)) break;
                LOGGER.warning(String.format("Received illegal record content type: %s", type));
                break;
            }
            int major = reader.read(8);
            int minor = reader.read(8);
            ProtocolVersion version = new ProtocolVersion(major, minor);
            int epoch = reader.read(16);
            long sequenceNumber = reader.readLong(48);
            int length = reader.read(16);
            byte[] fragmentBytes = reader.readBytes(length);
            records.add(new Record(contentType, version, epoch, sequenceNumber, length, fragmentBytes));
        }
        return records;
    }

    private byte[] encryptFragment(byte[] byteArray) {
        if (this.session == null) {
            return byteArray;
        }
        byte[] encryptedFragment = byteArray;
        CipherSuite cipherSuite = this.session.getWriteState().getCipherSuite();
        switch (cipherSuite.getCipherType()) {
            case NULL: {
                break;
            }
            case AEAD: {
                encryptedFragment = this.encryptAEAD(byteArray);
                break;
            }
            case BLOCK: {
                break;
            }
            case STREAM: {
                break;
            }
        }
        return encryptedFragment;
    }

    private byte[] decryptFragment(byte[] byteArray) throws HandshakeException {
        if (this.session == null) {
            return byteArray;
        }
        byte[] fragment = byteArray;
        CipherSuite cipherSuite = this.session.getReadState().getCipherSuite();
        switch (cipherSuite.getCipherType()) {
            case NULL: {
                break;
            }
            case AEAD: {
                fragment = this.decryptAEAD(byteArray);
                break;
            }
            case BLOCK: {
                break;
            }
            case STREAM: {
                break;
            }
        }
        return fragment;
    }

    protected byte[] encryptAEAD(byte[] byteArray) {
        byte[] iv = this.session.getWriteState().getIv().getIV();
        byte[] nonce = this.generateNonce(iv);
        byte[] key = this.session.getWriteState().getEncryptionKey().getEncoded();
        byte[] additionalData = this.generateAdditionalData(this.getLength());
        byte[] encryptedFragment = CCMBlockCipher.encrypt(key, nonce, additionalData, byteArray, 8);
        byte[] explicitNonce = this.generateExplicitNonce();
        encryptedFragment = ByteArrayUtils.concatenate(explicitNonce, encryptedFragment);
        return encryptedFragment;
    }

    protected byte[] decryptAEAD(byte[] byteArray) throws HandshakeException {
        byte[] explicitNonceUsed;
        byte[] iv = this.session.getReadState().getIv().getIV();
        byte[] key = this.session.getReadState().getEncryptionKey().getEncoded();
        byte[] additionalData = this.generateAdditionalData(this.getLength() - 16);
        DatagramReader reader = new DatagramReader(byteArray);
        byte[] explicitNonce = this.generateExplicitNonce();
        if (!Arrays.equals(explicitNonce, explicitNonceUsed = reader.readBytes(8)) && LOGGER.isLoggable(Level.FINE)) {
            StringBuffer b = new StringBuffer("The explicit nonce used by the sender does not match the values provided in the DTLS record");
            b.append("\nUsed    : ").append(ByteArrayUtils.toHexString(explicitNonceUsed));
            b.append("\nExpected: ").append(ByteArrayUtils.toHexString(explicitNonce));
            LOGGER.log(Level.FINE, b.toString());
        }
        byte[] nonce = this.getNonce(iv, explicitNonceUsed);
        byte[] decrypted = CCMBlockCipher.decrypt(key, nonce, additionalData, reader.readBytesLeft(), 8);
        return decrypted;
    }

    private byte[] generateNonce(byte[] iv) {
        return this.getNonce(iv, this.generateExplicitNonce());
    }

    private byte[] getNonce(byte[] implicitNonce, byte[] explicitNonce) {
        DatagramWriter writer = new DatagramWriter();
        writer.writeBytes(implicitNonce);
        writer.writeBytes(explicitNonce);
        return writer.toByteArray();
    }

    private byte[] generateExplicitNonce() {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        return writer.toByteArray();
    }

    private byte[] generateAdditionalData(int length) {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        writer.write(this.type.getCode(), 8);
        writer.write(this.version.getMajor(), 8);
        writer.write(this.version.getMinor(), 8);
        writer.write(length, 16);
        return writer.toByteArray();
    }

    public ContentType getType() {
        return this.type;
    }

    public void setType(ContentType type) {
        this.type = type;
    }

    public ProtocolVersion getVersion() {
        return this.version;
    }

    public void setVersion(ProtocolVersion version) {
        this.version = version;
    }

    public int getEpoch() {
        return this.epoch;
    }

    public void setEpoch(int epoch) {
        this.epoch = epoch;
    }

    public long getSequenceNumber() {
        return this.sequenceNumber;
    }

    public void setSequenceNumber(int sequenceNumber) {
        this.sequenceNumber = sequenceNumber;
    }

    public int getLength() {
        return this.length;
    }

    public void setLength(int length) {
        this.length = length;
    }

    public DTLSSession getSession() {
        return this.session;
    }

    public void setSession(DTLSSession session) {
        this.session = session;
    }

    public byte[] getFragmentBytes() {
        return this.fragmentBytes;
    }

    public DTLSMessage getFragment() throws HandshakeException {
        if (this.fragment == null) {
            switch (this.type) {
                case ALERT: {
                    byte[] decryptedMessage = this.decryptFragment(this.fragmentBytes);
                    if (decryptedMessage == null) break;
                    this.fragment = AlertMessage.fromByteArray(decryptedMessage);
                    break;
                }
                case APPLICATION_DATA: {
                    byte[] decryptedMessage = this.decryptFragment(this.fragmentBytes);
                    if (decryptedMessage == null) break;
                    this.fragment = ApplicationMessage.fromByteArray(decryptedMessage);
                    break;
                }
                case CHANGE_CIPHER_SPEC: {
                    byte[] decryptedMessage = this.decryptFragment(this.fragmentBytes);
                    if (decryptedMessage == null) break;
                    this.fragment = ChangeCipherSpecMessage.fromByteArray(decryptedMessage);
                    break;
                }
                case HANDSHAKE: {
                    byte[] decryptedMessage = this.decryptFragment(this.fragmentBytes);
                    CipherSuite.KeyExchangeAlgorithm keyExchangeAlgorithm = CipherSuite.KeyExchangeAlgorithm.NULL;
                    boolean receiveRawPublicKey = false;
                    if (this.session != null) {
                        keyExchangeAlgorithm = this.session.getKeyExchange();
                        receiveRawPublicKey = this.session.receiveRawPublicKey();
                    }
                    if (decryptedMessage == null) break;
                    this.fragment = HandshakeMessage.fromByteArray(decryptedMessage, keyExchangeAlgorithm, receiveRawPublicKey);
                    break;
                }
                default: {
                    LOGGER.severe("Unknown content type: " + (Object)((Object)this.type));
                }
            }
        }
        if (this.fragment == null) {
            AlertMessage alert = new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.BAD_RECORD_MAC);
            throw new HandshakeException("The decryption failed.", alert);
        }
        return this.fragment;
    }

    public void setFragment(DTLSMessage fragment) {
        if (this.fragmentBytes == null) {
            byte[] byteArray = fragment.toByteArray();
            this.length = byteArray.length;
            switch (this.type) {
                case ALERT: 
                case APPLICATION_DATA: 
                case CHANGE_CIPHER_SPEC: 
                case HANDSHAKE: {
                    byteArray = this.encryptFragment(byteArray);
                    break;
                }
                default: {
                    LOGGER.severe("Unknown content type: " + this.type.toString());
                }
            }
            this.fragmentBytes = byteArray;
        }
        this.fragment = fragment;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("==[ DTLS Message  ]============================================\n");
        sb.append("Content Type: " + this.type.toString() + "\n");
        sb.append("Version: " + this.version.getMajor() + ", " + this.version.getMinor() + "\n");
        sb.append("Epoch: " + this.epoch + "\n");
        sb.append("Sequence Number: " + this.sequenceNumber + "\n");
        sb.append("Length: " + this.length + "\n");
        sb.append(this.fragment.toString());
        sb.append("===============================================================");
        return sb.toString();
    }
}

