/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.security.cert.Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.californium.elements.ConnectorBase;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.scandium.DTLSConnectorConfig;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.ApplicationMessage;
import org.eclipse.californium.scandium.dtls.ClientHandshaker;
import org.eclipse.californium.scandium.dtls.ClientHello;
import org.eclipse.californium.scandium.dtls.ContentType;
import org.eclipse.californium.scandium.dtls.DTLSFlight;
import org.eclipse.californium.scandium.dtls.DTLSSession;
import org.eclipse.californium.scandium.dtls.FragmentedHandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.Handshaker;
import org.eclipse.californium.scandium.dtls.Record;
import org.eclipse.californium.scandium.dtls.ResumingClientHandshaker;
import org.eclipse.californium.scandium.dtls.ResumingServerHandshaker;
import org.eclipse.californium.scandium.dtls.ServerHandshaker;
import org.eclipse.californium.scandium.util.ByteArrayUtils;

public class DTLSConnector
extends ConnectorBase {
    private static final Logger LOGGER = Logger.getLogger(DTLSConnector.class.getCanonicalName());
    private final DTLSConnectorConfig config = new DTLSConnectorConfig(this);
    private final InetSocketAddress address;
    private DatagramSocket socket;
    private Timer timer = new Timer(true);
    private Map<String, DTLSSession> dtlsSessions = new ConcurrentHashMap<String, DTLSSession>();
    private Map<String, Handshaker> handshakers = new ConcurrentHashMap<String, Handshaker>();
    private Map<String, DTLSFlight> flights = new ConcurrentHashMap<String, DTLSFlight>();
    private final Certificate[] rootCerts;

    public DTLSConnector(InetSocketAddress address, Certificate[] rootCertificates) {
        super(address);
        this.address = address;
        this.rootCerts = rootCertificates;
    }

    public void close() {
        for (DTLSSession session : this.dtlsSessions.values()) {
            this.close(session.getPeer());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close(InetSocketAddress peerAddress) {
        String addrKey = this.addressToKey(peerAddress);
        try {
            DTLSSession session = this.dtlsSessions.get(addrKey);
            if (session != null) {
                AlertMessage closeNotify = new AlertMessage(AlertMessage.AlertLevel.WARNING, AlertMessage.AlertDescription.CLOSE_NOTIFY);
                DTLSFlight flight = new DTLSFlight();
                flight.addMessage(new Record(ContentType.ALERT, session.getWriteEpoch(), session.getSequenceNumber(), closeNotify, session));
                flight.setRetransmissionNeeded(false);
                this.cancelPreviousFlight(peerAddress);
                flight.setPeerAddress(peerAddress);
                flight.setSession(session);
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Sending CLOSE_NOTIFY to " + peerAddress.toString());
                }
                this.sendFlight(flight);
            } else if (LOGGER.isLoggable(Level.WARNING)) {
                LOGGER.warning("Session to close not found: " + peerAddress.toString());
            }
        }
        finally {
            this.dtlsSessions.remove(addrKey);
            this.handshakers.remove(addrKey);
            this.flights.remove(addrKey);
        }
    }

    @Override
    public synchronized void start() throws IOException {
        this.socket = new DatagramSocket(this.address.getPort(), this.address.getAddress());
        super.start();
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.info("DLTS connector listening on " + this.address);
        }
    }

    @Override
    public synchronized void stop() {
        this.close();
        this.socket.close();
        super.stop();
    }

    @Override
    protected RawData receiveNext() throws Exception {
        byte[] buffer = new byte[this.config.getMaxPayloadSize()];
        DatagramPacket packet = new DatagramPacket(buffer, buffer.length);
        this.socket.receive(packet);
        if (packet.getLength() == 0) {
            return null;
        }
        InetSocketAddress peerAddress = new InetSocketAddress(packet.getAddress(), packet.getPort());
        if (LOGGER.isLoggable(Level.FINEST)) {
            LOGGER.finest(" => find handshaker for key " + peerAddress.toString());
        }
        DTLSSession session = this.dtlsSessions.get(this.addressToKey(peerAddress));
        Handshaker handshaker = this.handshakers.get(this.addressToKey(peerAddress));
        byte[] data = Arrays.copyOfRange(packet.getData(), packet.getOffset(), packet.getLength());
        try {
            List<Record> records = Record.fromByteArray(data);
            for (Record record : records) {
                record.setSession(session);
                RawData raw = null;
                ContentType contentType = record.getType();
                LOGGER.finest(" => contentType: " + (Object)((Object)contentType));
                DTLSFlight flight = null;
                block1 : switch (contentType) {
                    case APPLICATION_DATA: {
                        if (session == null) {
                            if (LOGGER.isLoggable(Level.INFO)) {
                                LOGGER.info("Discarded unexpected application data message from " + peerAddress.toString());
                            }
                            return null;
                        }
                        this.handshakers.remove(this.addressToKey(peerAddress));
                        ApplicationMessage applicationData = (ApplicationMessage)record.getFragment();
                        raw = new RawData(applicationData.getData());
                        break;
                    }
                    case ALERT: {
                        AlertMessage alert = (AlertMessage)record.getFragment();
                        switch (alert.getDescription()) {
                            case CLOSE_NOTIFY: {
                                session.setActive(false);
                                if (LOGGER.isLoggable(Level.FINE)) {
                                    LOGGER.fine("Received CLOSE_NOTIFY from " + peerAddress.toString());
                                }
                                AlertMessage closeNotify = new AlertMessage(AlertMessage.AlertLevel.WARNING, AlertMessage.AlertDescription.CLOSE_NOTIFY);
                                flight = new DTLSFlight();
                                flight.addMessage(new Record(ContentType.ALERT, session.getWriteEpoch(), session.getSequenceNumber(), closeNotify, session));
                                flight.setRetransmissionNeeded(false);
                                if (this.dtlsSessions.remove(this.addressToKey(peerAddress)) != null) {
                                    if (!LOGGER.isLoggable(Level.INFO)) break block1;
                                    LOGGER.info("Closed session with peer: " + peerAddress.toString());
                                    break;
                                }
                                if (!LOGGER.isLoggable(Level.WARNING)) break block1;
                                LOGGER.warning("Session to close not found: " + peerAddress.toString());
                                break;
                            }
                            default: {
                                if (LOGGER.isLoggable(Level.WARNING)) {
                                    LOGGER.warning((Object)((Object)alert.getDescription()) + " with " + peerAddress.toString());
                                }
                                this.cancelPreviousFlight(peerAddress);
                                this.dtlsSessions.remove(this.addressToKey(peerAddress));
                                this.handshakers.remove(this.addressToKey(peerAddress));
                                break;
                            }
                        }
                        break;
                    }
                    case CHANGE_CIPHER_SPEC: 
                    case HANDSHAKE: {
                        if (LOGGER.isLoggable(Level.FINEST)) {
                            LOGGER.finest(" => handshaker: " + handshaker);
                        }
                        if (handshaker == null) {
                            HandshakeMessage handshake = (HandshakeMessage)record.getFragment();
                            switch (handshake.getMessageType()) {
                                case HELLO_REQUEST: {
                                    if (session == null) {
                                        session = new DTLSSession(peerAddress, true);
                                        this.dtlsSessions.put(this.addressToKey(peerAddress), session);
                                        if (LOGGER.isLoggable(Level.INFO)) {
                                            LOGGER.info("Created new session as client with peer: " + peerAddress.toString());
                                        }
                                    }
                                    handshaker = new ClientHandshaker(peerAddress, null, session, this.rootCerts, this.config);
                                    handshaker.setMaxFragmentLength(this.config.getMaxFragmentLength());
                                    this.handshakers.put(this.addressToKey(peerAddress), handshaker);
                                    if (!LOGGER.isLoggable(Level.FINEST)) break;
                                    LOGGER.finest("Stored re-handshaker: " + handshaker.toString() + " for " + peerAddress.toString());
                                    break;
                                }
                                case CLIENT_HELLO: {
                                    if (!(handshake instanceof FragmentedHandshakeMessage)) {
                                        ClientHello clientHello = (ClientHello)handshake;
                                        session = this.getSessionByIdentifier(clientHello.getSessionId().getSessionId());
                                    }
                                    if (session == null) {
                                        session = new DTLSSession(peerAddress, false);
                                        this.dtlsSessions.put(this.addressToKey(peerAddress), session);
                                        if (LOGGER.isLoggable(Level.INFO)) {
                                            LOGGER.info("Created new session as server with peer: " + peerAddress.toString());
                                        }
                                        handshaker = new ServerHandshaker(peerAddress, session, this.rootCerts, this.config);
                                        handshaker.setMaxFragmentLength(this.config.getMaxFragmentLength());
                                    } else {
                                        handshaker = new ResumingServerHandshaker(peerAddress, session, this.rootCerts, this.config);
                                        handshaker.setMaxFragmentLength(this.config.getMaxFragmentLength());
                                    }
                                    this.handshakers.put(this.addressToKey(peerAddress), handshaker);
                                    if (!LOGGER.isLoggable(Level.FINEST)) break;
                                    LOGGER.finest("Stored handshaker: " + handshaker.toString() + " for " + peerAddress.toString());
                                    break;
                                }
                                default: {
                                    LOGGER.severe("Received unexpected first handshake message (type=" + (Object)((Object)handshake.getMessageType()) + ") from " + peerAddress.toString() + ":\n" + handshake.toString());
                                }
                            }
                        }
                        flight = handshaker.processMessage(record);
                        break;
                    }
                    default: {
                        LOGGER.severe("Received unknown DTLS record from " + peerAddress.toString() + ":\n" + ByteArrayUtils.toHexString(data));
                    }
                }
                if (flight != null) {
                    this.cancelPreviousFlight(peerAddress);
                    flight.setPeerAddress(peerAddress);
                    flight.setSession(session);
                    if (flight.isRetransmissionNeeded()) {
                        this.flights.put(this.addressToKey(peerAddress), flight);
                        this.scheduleRetransmission(flight);
                    }
                    this.sendFlight(flight);
                }
                if (raw == null) continue;
                raw.setAddress(packet.getAddress());
                raw.setPort(packet.getPort());
                return raw;
            }
        }
        catch (Exception e) {
            AlertMessage alert;
            DTLSFlight flight = new DTLSFlight();
            flight.setRetransmissionNeeded(false);
            flight.setPeerAddress(peerAddress);
            flight.setSession(session);
            if (e instanceof HandshakeException) {
                alert = ((HandshakeException)e).getAlert();
                LOGGER.severe("Handshake Exception (" + peerAddress.toString() + "): " + e.getMessage() + " we close the session");
                this.close(session.getPeer());
            } else {
                alert = new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.HANDSHAKE_FAILURE);
                LOGGER.log(Level.SEVERE, "Unknown Exception (" + peerAddress + ").", e);
            }
            LOGGER.log(Level.SEVERE, "Datagram which lead to exception (" + peerAddress + "): " + ByteArrayUtils.toHexString(data), e);
            if (session == null) {
                session = new DTLSSession(peerAddress, false);
            }
            this.cancelPreviousFlight(peerAddress);
            flight.addMessage(new Record(ContentType.ALERT, session.getWriteEpoch(), session.getSequenceNumber(), alert, session));
            this.sendFlight(flight);
        }
        return null;
    }

    @Override
    protected void sendNext(RawData message) throws Exception {
        InetSocketAddress peerAddress = message.getInetSocketAddress();
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine("Sending message to " + peerAddress);
        }
        DTLSSession session = this.dtlsSessions.get(this.addressToKey(peerAddress));
        Record encryptedMessage = null;
        ClientHandshaker handshaker = null;
        if (session == null) {
            session = new DTLSSession(peerAddress, true);
            this.dtlsSessions.put(this.addressToKey(peerAddress), session);
            handshaker = new ClientHandshaker(peerAddress, message, session, this.rootCerts, this.config);
            handshaker.setMaxFragmentLength(this.config.getMaxFragmentLength());
        } else if (session.isActive()) {
            ApplicationMessage fragment = new ApplicationMessage(message.getBytes());
            encryptedMessage = new Record(ContentType.APPLICATION_DATA, session.getWriteEpoch(), session.getSequenceNumber(), fragment, session);
        } else {
            handshaker = new ResumingClientHandshaker(peerAddress, message, session, this.rootCerts, this.config);
            handshaker.setMaxFragmentLength(this.config.getMaxFragmentLength());
        }
        DTLSFlight flight = new DTLSFlight();
        if (handshaker != null) {
            this.handshakers.put(this.addressToKey(peerAddress), handshaker);
            if (LOGGER.isLoggable(Level.FINEST)) {
                LOGGER.finest("Stored handshaker on send: " + handshaker.toString() + " for " + peerAddress.toString());
            }
            flight = ((Handshaker)handshaker).getStartHandshakeMessage();
            this.flights.put(this.addressToKey(peerAddress), flight);
            this.scheduleRetransmission(flight);
        }
        if (encryptedMessage != null) {
            flight.addMessage(encryptedMessage);
        }
        flight.setPeerAddress(peerAddress);
        flight.setSession(session);
        this.sendFlight(flight);
    }

    public DTLSSession getSessionByAddress(InetSocketAddress address) {
        if (address == null) {
            return null;
        }
        return this.dtlsSessions.get(this.addressToKey(address));
    }

    private DTLSSession getSessionByIdentifier(byte[] sessionID) {
        byte[] id;
        if (sessionID == null) {
            return null;
        }
        for (Map.Entry<String, DTLSSession> entry : this.dtlsSessions.entrySet()) {
            try {
                id = entry.getValue().getSessionIdentifier().getSessionId();
                if (!Arrays.equals(sessionID, id)) continue;
                return entry.getValue();
            }
            catch (Exception e) {
            }
        }
        for (DTLSSession session : this.dtlsSessions.values()) {
            try {
                id = session.getSessionIdentifier().getSessionId();
                if (!Arrays.equals(sessionID, id)) continue;
                return session;
            }
            catch (Exception e) {
            }
        }
        return null;
    }

    private void sendFlight(DTLSFlight flight) {
        byte[] payload = new byte[]{};
        ArrayList<DatagramPacket> datagrams = new ArrayList<DatagramPacket>();
        for (Record record : flight.getMessages()) {
            byte[] recordBytes;
            if (flight.getTries() > 0) {
                int epoch = record.getEpoch();
                record.setSequenceNumber(flight.getSession().getSequenceNumber(epoch));
            }
            if (payload.length + (recordBytes = record.toByteArray()).length > this.config.getMaxPayloadSize()) {
                DatagramPacket datagram = new DatagramPacket(payload, payload.length, flight.getPeerAddress().getAddress(), flight.getPeerAddress().getPort());
                datagrams.add(datagram);
                payload = new byte[]{};
            }
            payload = ByteArrayUtils.concatenate(payload, recordBytes);
        }
        DatagramPacket datagram = new DatagramPacket(payload, payload.length, flight.getPeerAddress().getAddress(), flight.getPeerAddress().getPort());
        datagrams.add(datagram);
        try {
            for (DatagramPacket datagramPacket : datagrams) {
                this.socket.send(datagramPacket);
            }
        }
        catch (IOException iOException) {
            LOGGER.log(Level.SEVERE, "Could not send the datagram", iOException);
        }
    }

    private void handleTimeout(DTLSFlight flight) {
        int max = this.config.getMaxRetransmit();
        if (flight.getTries() < max) {
            flight.incrementTries();
            this.sendFlight(flight);
            this.scheduleRetransmission(flight);
        } else {
            LOGGER.fine("Maximum retransmissions reached.");
        }
    }

    private void scheduleRetransmission(DTLSFlight flight) {
        if (flight.getRetransmitTask() != null) {
            flight.getRetransmitTask().cancel();
        }
        if (flight.isRetransmissionNeeded()) {
            flight.setRetransmitTask(new RetransmitTask(flight));
            if (flight.getTimeout() == 0) {
                flight.setTimeout(this.config.getRetransmissionTimeout());
            } else {
                flight.incrementTimeout();
            }
            this.timer.schedule(flight.getRetransmitTask(), flight.getTimeout());
        }
    }

    private void cancelPreviousFlight(InetSocketAddress peerAddress) {
        DTLSFlight previousFlight = this.flights.get(this.addressToKey(peerAddress));
        if (previousFlight != null) {
            previousFlight.getRetransmitTask().cancel();
            previousFlight.setRetransmitTask(null);
            this.flights.remove(this.addressToKey(peerAddress));
        }
    }

    @Override
    public String getName() {
        return "DTLS";
    }

    @Override
    public InetSocketAddress getAddress() {
        if (this.socket == null) {
            return this.getLocalAddr();
        }
        return new InetSocketAddress(this.socket.getLocalAddress(), this.socket.getLocalPort());
    }

    private String addressToKey(InetSocketAddress address) {
        return address.toString().split("/")[1];
    }

    public DTLSConnectorConfig getConfig() {
        return this.config;
    }

    private class RetransmitTask
    extends TimerTask {
        private DTLSFlight flight;

        RetransmitTask(DTLSFlight flight) {
            this.flight = flight;
        }

        @Override
        public void run() {
            DTLSConnector.this.handleTimeout(this.flight);
        }
    }
}

