/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Endpoint;
import org.eclipse.californium.core.network.EndpointManager;
import org.eclipse.californium.core.network.EndpointObserver;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.Matcher;
import org.eclipse.californium.core.network.Outbox;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.interceptors.MessageInterceptor;
import org.eclipse.californium.core.network.serialization.DataParser;
import org.eclipse.californium.core.network.serialization.Serializer;
import org.eclipse.californium.core.network.stack.CoapStack;
import org.eclipse.californium.core.server.MessageDeliverer;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.RawDataChannel;
import org.eclipse.californium.elements.UDPConnector;

public class CoAPEndpoint
implements Endpoint {
    private static final Logger LOGGER = Logger.getLogger(CoAPEndpoint.class.getCanonicalName());
    private final CoapStack coapstack;
    private final Connector connector;
    private final NetworkConfig config;
    private ScheduledExecutorService executor;
    private boolean started;
    private List<EndpointObserver> observers = new ArrayList<EndpointObserver>(0);
    private List<MessageInterceptor> interceptors = new ArrayList<MessageInterceptor>(0);
    private Matcher matcher;
    private Serializer serializer;

    public CoAPEndpoint() {
        this(0);
    }

    public CoAPEndpoint(int port) {
        this(new InetSocketAddress(port));
    }

    public CoAPEndpoint(InetSocketAddress address) {
        this(address, NetworkConfig.getStandard());
    }

    public CoAPEndpoint(NetworkConfig config) {
        this(new InetSocketAddress(0), config);
    }

    public CoAPEndpoint(InetSocketAddress address, NetworkConfig config) {
        this(CoAPEndpoint.createUDPConnector(address, config), config);
    }

    public CoAPEndpoint(Connector connector, NetworkConfig config) {
        this.config = config;
        this.connector = connector;
        this.serializer = new Serializer();
        this.matcher = new Matcher(config);
        this.coapstack = new CoapStack(config, new OutboxImpl());
        connector.setRawDataReceiver(new InboxImpl());
    }

    private static Connector createUDPConnector(InetSocketAddress address, NetworkConfig config) {
        UDPConnector c = new UDPConnector(address);
        c.setReceiverThreadCount(config.getInt("UDP_CONNECTOR_RECEIVER_THREAD_COUNT"));
        c.setSenderThreadCount(config.getInt("UDP_CONNECTOR_SENDER_THREAD_COUNT"));
        c.setReceiveBufferSize(config.getInt("UDP_CONNECTOR_RECEIVE_BUFFER"));
        c.setSendBufferSize(config.getInt("UDP_CONNECTOR_SEND_BUFFER"));
        c.setLogPackets(config.getBoolean("UDP_CONNECTOR_LOG_PACKETS"));
        c.setReceiverPacketSize(config.getInt("UDP_CONNECTOR_DATAGRAM_SIZE"));
        return c;
    }

    @Override
    public synchronized void start() throws IOException {
        if (this.started) {
            LOGGER.log(Level.FINE, "Endpoint bound to " + this.getAddress().toString() + " is already started");
            return;
        }
        if (this.executor == null) {
            LOGGER.fine("Endpoint " + this.toString() + " requires an executor to start. Using default single-threaded daemon executor.");
            this.setExecutor(Executors.newSingleThreadScheduledExecutor(new EndpointManager.DaemonThreadFactory()));
        }
        try {
            LOGGER.log(Level.INFO, "Starting Endpoint bound to " + this.getAddress());
            this.started = true;
            this.matcher.start();
            this.connector.start();
            for (EndpointObserver obs : this.observers) {
                obs.started(this);
            }
            this.startExecutor();
        }
        catch (IOException e) {
            LOGGER.log(Level.SEVERE, "Cannot start Endpoint at " + this.getAddress(), e);
            this.stop();
            throw e;
        }
    }

    private void startExecutor() {
        this.executeTask(new Runnable(){

            @Override
            public void run() {
            }
        });
    }

    @Override
    public synchronized void stop() {
        if (!this.started) {
            LOGGER.log(Level.INFO, "Endpoint at address " + this.getAddress() + " is already stopped");
        } else {
            LOGGER.log(Level.INFO, "Stopping endpoint at address " + this.getAddress());
            this.started = false;
            this.connector.stop();
            this.matcher.stop();
            for (EndpointObserver obs : this.observers) {
                obs.stopped(this);
            }
            this.matcher.clear();
        }
    }

    @Override
    public synchronized void destroy() {
        LOGGER.log(Level.INFO, "Destroying endpoint at address " + this.getAddress());
        if (this.started) {
            this.stop();
        }
        this.connector.destroy();
        for (EndpointObserver obs : this.observers) {
            obs.destroyed(this);
        }
    }

    @Override
    public void clear() {
        this.matcher.clear();
    }

    @Override
    public boolean isStarted() {
        return this.started;
    }

    @Override
    public synchronized void setExecutor(ScheduledExecutorService executor) {
        this.executor = executor;
        this.coapstack.setExecutor(executor);
        this.matcher.setExecutor(executor);
    }

    @Override
    public void addObserver(EndpointObserver obs) {
        this.observers.add(obs);
    }

    @Override
    public void removeObserver(EndpointObserver obs) {
        this.observers.remove(obs);
    }

    @Override
    public void addInterceptor(MessageInterceptor interceptor) {
        this.interceptors.add(interceptor);
    }

    @Override
    public void removeInterceptor(MessageInterceptor interceptor) {
        this.interceptors.remove(interceptor);
    }

    @Override
    public List<MessageInterceptor> getInterceptors() {
        return new ArrayList<MessageInterceptor>(this.interceptors);
    }

    @Override
    public void sendRequest(final Request request) {
        this.executor.execute(new Runnable(){

            @Override
            public void run() {
                try {
                    CoAPEndpoint.this.coapstack.sendRequest(request);
                }
                catch (Throwable t) {
                    t.printStackTrace();
                }
            }
        });
    }

    @Override
    public void sendResponse(Exchange exchange, Response response) {
        this.coapstack.sendResponse(exchange, response);
    }

    @Override
    public void sendEmptyMessage(final Exchange exchange, final EmptyMessage message) {
        this.executor.execute(new Runnable(){

            @Override
            public void run() {
                try {
                    CoAPEndpoint.this.coapstack.sendEmptyMessage(exchange, message);
                }
                catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }

    @Override
    public void setMessageDeliverer(MessageDeliverer deliverer) {
        this.coapstack.setDeliverer(deliverer);
    }

    @Override
    public InetSocketAddress getAddress() {
        return this.connector.getAddress();
    }

    @Override
    public NetworkConfig getConfig() {
        return this.config;
    }

    private void executeTask(final Runnable task) {
        this.executor.execute(new Runnable(){

            @Override
            public void run() {
                try {
                    task.run();
                }
                catch (Throwable t) {
                    t.printStackTrace();
                }
            }
        });
    }

    private class InboxImpl
    implements RawDataChannel {
        private InboxImpl() {
        }

        @Override
        public void receiveData(final RawData raw) {
            if (raw.getAddress() == null) {
                throw new NullPointerException();
            }
            if (raw.getPort() == 0) {
                throw new NullPointerException();
            }
            Runnable task = new Runnable(){

                @Override
                public void run() {
                    InboxImpl.this.receiveMessage(raw);
                }
            };
            CoAPEndpoint.this.executeTask(task);
        }

        private void receiveMessage(RawData raw) {
            DataParser parser = new DataParser(raw.getBytes());
            if (parser.isRequest()) {
                Exchange exchange;
                Request request;
                try {
                    request = parser.parseRequest();
                }
                catch (IllegalStateException e) {
                    String log = "message format error caused by " + raw.getInetSocketAddress();
                    if (!parser.isReply()) {
                        EmptyMessage rst = new EmptyMessage(CoAP.Type.RST);
                        rst.setDestination(raw.getAddress());
                        rst.setDestinationPort(raw.getPort());
                        rst.setMID(parser.getMID());
                        for (MessageInterceptor interceptor : CoAPEndpoint.this.interceptors) {
                            interceptor.sendEmptyMessage(rst);
                        }
                        CoAPEndpoint.this.connector.send(CoAPEndpoint.this.serializer.serialize(rst));
                        log = log + " and reseted";
                    }
                    LOGGER.info(log);
                    return;
                }
                request.setSource(raw.getAddress());
                request.setSourcePort(raw.getPort());
                for (MessageInterceptor interceptor : CoAPEndpoint.this.interceptors) {
                    interceptor.receiveRequest(request);
                }
                if (!request.isCanceled() && (exchange = CoAPEndpoint.this.matcher.receiveRequest(request)) != null) {
                    exchange.setEndpoint(CoAPEndpoint.this);
                    CoAPEndpoint.this.coapstack.receiveRequest(exchange, request);
                }
            } else if (parser.isResponse()) {
                Response response = parser.parseResponse();
                response.setSource(raw.getAddress());
                response.setSourcePort(raw.getPort());
                Object exchange = CoAPEndpoint.this.interceptors.iterator();
                while (exchange.hasNext()) {
                    MessageInterceptor interceptor = (MessageInterceptor)exchange.next();
                    interceptor.receiveResponse(response);
                }
                if (!response.isCanceled() && (exchange = CoAPEndpoint.this.matcher.receiveResponse(response)) != null) {
                    ((Exchange)exchange).setEndpoint(CoAPEndpoint.this);
                    response.setRTT(System.currentTimeMillis() - ((Exchange)exchange).getTimestamp());
                    CoAPEndpoint.this.coapstack.receiveResponse((Exchange)exchange, response);
                }
            } else if (parser.isEmpty()) {
                EmptyMessage message = parser.parseEmptyMessage();
                message.setSource(raw.getAddress());
                message.setSourcePort(raw.getPort());
                for (MessageInterceptor interceptor : CoAPEndpoint.this.interceptors) {
                    interceptor.receiveEmptyMessage(message);
                }
                if (!message.isCanceled()) {
                    if (message.getType() == CoAP.Type.CON || message.getType() == CoAP.Type.NON) {
                        EmptyMessage rst = EmptyMessage.newRST(message);
                        LOGGER.info("Responding to ping by " + raw.getInetSocketAddress());
                        for (MessageInterceptor interceptor : CoAPEndpoint.this.interceptors) {
                            interceptor.sendEmptyMessage(rst);
                        }
                        CoAPEndpoint.this.connector.send(CoAPEndpoint.this.serializer.serialize(rst));
                    } else {
                        Exchange exchange = CoAPEndpoint.this.matcher.receiveEmptyMessage(message);
                        if (exchange != null) {
                            exchange.setEndpoint(CoAPEndpoint.this);
                            CoAPEndpoint.this.coapstack.receiveEmptyMessage(exchange, message);
                        }
                    }
                }
            } else {
                LOGGER.finest("Silently ignoring non-CoAP message from " + raw.getInetSocketAddress());
            }
        }
    }

    private class OutboxImpl
    implements Outbox {
        private OutboxImpl() {
        }

        @Override
        public void sendRequest(Exchange exchange, Request request) {
            CoAPEndpoint.this.matcher.sendRequest(exchange, request);
            for (MessageInterceptor interceptor : CoAPEndpoint.this.interceptors) {
                interceptor.sendRequest(request);
            }
            if (!request.isCanceled()) {
                CoAPEndpoint.this.connector.send(CoAPEndpoint.this.serializer.serialize(request));
            }
        }

        @Override
        public void sendResponse(Exchange exchange, Response response) {
            CoAPEndpoint.this.matcher.sendResponse(exchange, response);
            for (MessageInterceptor interceptor : CoAPEndpoint.this.interceptors) {
                interceptor.sendResponse(response);
            }
            if (!response.isCanceled()) {
                CoAPEndpoint.this.connector.send(CoAPEndpoint.this.serializer.serialize(response));
            }
        }

        @Override
        public void sendEmptyMessage(Exchange exchange, EmptyMessage message) {
            CoAPEndpoint.this.matcher.sendEmptyMessage(exchange, message);
            for (MessageInterceptor interceptor : CoAPEndpoint.this.interceptors) {
                interceptor.sendEmptyMessage(message);
            }
            if (!message.isCanceled()) {
                CoAPEndpoint.this.connector.send(CoAPEndpoint.this.serializer.serialize(message));
            }
        }
    }
}

