/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.elements;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.LinkedBlockingQueue;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.EndpointContext;
import org.eclipse.californium.elements.EndpointContextMatcher;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.RawDataChannel;
import org.eclipse.californium.elements.UdpEndpointContext;
import org.eclipse.californium.elements.UdpMulticastConnector;
import org.eclipse.californium.elements.config.Configuration;
import org.eclipse.californium.elements.config.UdpConfig;
import org.eclipse.californium.elements.exception.EndpointMismatchException;
import org.eclipse.californium.elements.util.Bytes;
import org.eclipse.californium.elements.util.ClockUtil;
import org.eclipse.californium.elements.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UDPConnector
implements Connector {
    @Deprecated
    public static final Logger LOGGER = LoggerFactory.getLogger(UDPConnector.class);
    static final ThreadGroup ELEMENTS_THREAD_GROUP = new ThreadGroup("Californium/Elements");
    protected final InetSocketAddress localAddr;
    private final List<Thread> receiverThreads = new LinkedList<Thread>();
    private final List<Thread> senderThreads = new LinkedList<Thread>();
    private final BlockingQueue<RawData> outgoing;
    private final List<UdpMulticastConnector> multicastReceivers = new CopyOnWriteArrayList<UdpMulticastConnector>();
    private final int senderCount;
    private final int receiverCount;
    private final int receiverPacketSize;
    private final Integer configReceiveBufferSize;
    private final Integer configSendBufferSize;
    protected volatile boolean running;
    private volatile DatagramSocket socket;
    protected volatile InetSocketAddress effectiveAddr;
    private volatile EndpointContextMatcher endpointContextMatcher;
    private volatile RawDataChannel receiver;
    private Integer receiveBufferSize;
    private Integer sendBufferSize;
    private boolean reuseAddress;
    protected boolean multicast;

    public UDPConnector(InetSocketAddress address, Configuration configuration) {
        this.localAddr = address == null ? new InetSocketAddress(0) : address;
        this.running = false;
        this.effectiveAddr = this.localAddr;
        this.outgoing = new LinkedBlockingQueue<RawData>(configuration.get(UdpConfig.UDP_CONNECTOR_OUT_CAPACITY));
        this.receiverCount = configuration.get(UdpConfig.UDP_RECEIVER_THREAD_COUNT);
        this.senderCount = configuration.get(UdpConfig.UDP_SENDER_THREAD_COUNT);
        this.receiverPacketSize = configuration.get(UdpConfig.UDP_DATAGRAM_SIZE);
        this.configReceiveBufferSize = configuration.get(UdpConfig.UDP_RECEIVE_BUFFER_SIZE);
        this.configSendBufferSize = configuration.get(UdpConfig.UDP_SEND_BUFFER_SIZE);
        this.receiveBufferSize = this.configReceiveBufferSize;
        this.sendBufferSize = this.configSendBufferSize;
    }

    @Override
    public boolean isRunning() {
        return this.running;
    }

    @Override
    public synchronized void start() throws IOException {
        if (this.running) {
            return;
        }
        for (UdpMulticastConnector multicastReceiver : this.multicastReceivers) {
            multicastReceiver.start();
        }
        DatagramSocket socket = new DatagramSocket(null);
        socket.setReuseAddress(this.reuseAddress);
        socket.bind(this.localAddr);
        this.init(socket);
    }

    protected void init(DatagramSocket socket) throws IOException {
        int i;
        this.socket = socket;
        this.effectiveAddr = (InetSocketAddress)socket.getLocalSocketAddress();
        if (this.configReceiveBufferSize != null) {
            socket.setReceiveBufferSize(this.configReceiveBufferSize);
        }
        this.receiveBufferSize = socket.getReceiveBufferSize();
        if (this.configSendBufferSize != null) {
            socket.setSendBufferSize(this.configSendBufferSize);
        }
        this.sendBufferSize = socket.getSendBufferSize();
        this.running = true;
        LOGGER.info("UDPConnector starts up {} sender threads and {} receiver threads", (Object)this.senderCount, (Object)this.receiverCount);
        for (i = 0; i < this.receiverCount; ++i) {
            this.receiverThreads.add(new Receiver("UDP-Receiver-" + this.localAddr + "[" + i + "]"));
        }
        if (!this.multicast) {
            for (i = 0; i < this.senderCount; ++i) {
                this.senderThreads.add(new Sender("UDP-Sender-" + this.localAddr + "[" + i + "]"));
            }
        }
        for (Thread t : this.receiverThreads) {
            t.start();
        }
        for (Thread t : this.senderThreads) {
            t.start();
        }
        LOGGER.info("UDPConnector listening on {}, recv buf = {}, send buf = {}, recv packet size = {}", this.effectiveAddr, this.receiveBufferSize, this.sendBufferSize, this.receiverPacketSize);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void stop() {
        ArrayList pending = new ArrayList(this.outgoing.size());
        UDPConnector uDPConnector = this;
        synchronized (uDPConnector) {
            if (!this.running) {
                return;
            }
            this.running = false;
            LOGGER.debug("UDPConnector on [{}] stopping ...", (Object)this.effectiveAddr);
            for (Connector connector : this.multicastReceivers) {
                connector.stop();
            }
            for (Thread thread : this.senderThreads) {
                thread.interrupt();
            }
            for (Thread thread : this.receiverThreads) {
                thread.interrupt();
            }
            this.outgoing.drainTo(pending);
            if (this.socket != null) {
                this.socket.close();
                this.socket = null;
            }
            for (Thread thread : this.senderThreads) {
                thread.interrupt();
                try {
                    thread.join(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
            this.senderThreads.clear();
            for (Thread thread : this.receiverThreads) {
                thread.interrupt();
                try {
                    thread.join(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
            this.receiverThreads.clear();
            LOGGER.debug("UDPConnector on [{}] has stopped.", (Object)this.effectiveAddr);
        }
        for (RawData data : pending) {
            this.notifyMsgAsInterrupted(data);
        }
    }

    @Override
    public void destroy() {
        this.stop();
        for (Connector connector : this.multicastReceivers) {
            connector.destroy();
        }
        this.receiver = null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void send(RawData msg) {
        boolean running;
        if (msg == null) {
            throw new NullPointerException("Message must not be null");
        }
        if (this.multicast) {
            throw new IllegalStateException("Connector is a multicast receiver!");
        }
        if (msg.getInetSocketAddress().getPort() == 0) {
            String destination = StringUtil.toString(msg.getInetSocketAddress());
            LOGGER.trace("Discarding message with {} bytes to [{}] without destination-port", (Object)msg.getSize(), (Object)destination);
            msg.onError(new IOException("CoAP message to " + destination + " dropped, destination port 0!"));
            return;
        }
        boolean added = false;
        UDPConnector uDPConnector = this;
        synchronized (uDPConnector) {
            running = this.running;
            if (running) {
                added = this.outgoing.offer(msg);
            }
        }
        if (!running) {
            this.notifyMsgAsInterrupted(msg);
        } else if (!added) {
            msg.onError(new InterruptedIOException("Connector overloaded."));
        }
    }

    @Override
    public void setRawDataReceiver(RawDataChannel receiver) {
        this.receiver = receiver;
        for (UdpMulticastConnector multicastReceiver : this.multicastReceivers) {
            multicastReceiver.setRawDataReceiver(receiver);
        }
    }

    @Override
    public void setEndpointContextMatcher(EndpointContextMatcher matcher) {
        this.endpointContextMatcher = matcher;
        for (UdpMulticastConnector multicastReceiver : this.multicastReceivers) {
            multicastReceiver.setEndpointContextMatcher(matcher);
        }
    }

    public void addMulticastReceiver(UdpMulticastConnector multicastReceiver) {
        if (multicastReceiver == null) {
            throw new NullPointerException("Connector must not be null!");
        }
        if (!multicastReceiver.isMutlicastReceiver()) {
            throw new IllegalArgumentException("Connector is no valid multicast receiver!");
        }
        if (this.multicast) {
            throw new IllegalStateException("Connector itself is a multicast receiver!");
        }
        this.multicastReceivers.add(multicastReceiver);
        multicastReceiver.setRawDataReceiver(this.receiver);
    }

    public void removeMulticastReceiver(UdpMulticastConnector multicastReceiver) {
        if (this.multicastReceivers.remove(multicastReceiver)) {
            multicastReceiver.setRawDataReceiver(null);
        }
    }

    @Override
    public InetSocketAddress getAddress() {
        return this.effectiveAddr;
    }

    private void notifyMsgAsInterrupted(RawData msg) {
        msg.onError(new InterruptedIOException("Connector is not running."));
    }

    @Override
    public void processDatagram(DatagramPacket datagram) {
        InetSocketAddress connector = this.effectiveAddr;
        RawDataChannel dataReceiver = this.receiver;
        if (datagram.getPort() == 0) {
            LOGGER.trace("Discarding message with {} bytes from [{}] without source-port", (Object)datagram.getLength(), StringUtil.toLog(datagram.getSocketAddress()));
            return;
        }
        if (datagram.getLength() > this.receiverPacketSize) {
            LOGGER.debug("UDPConnector ({}) received truncated UDP datagram from {}. Maximum size allowed {}. Discarding ...", connector, StringUtil.toLog(datagram.getSocketAddress()), this.receiverPacketSize);
        } else if (dataReceiver == null) {
            LOGGER.debug("UDPConnector ({}) received UDP datagram from {} without receiver. Discarding ...", (Object)connector, StringUtil.toLog(datagram.getSocketAddress()));
        } else {
            long timestamp = ClockUtil.nanoRealtime();
            String local = StringUtil.toString(connector);
            if (this.multicast) {
                local = "mc/" + local;
            }
            LOGGER.debug("UDPConnector ({}) received {} bytes from {}", local, datagram.getLength(), StringUtil.toLog(datagram.getSocketAddress()));
            byte[] bytes = Arrays.copyOfRange(datagram.getData(), datagram.getOffset(), datagram.getLength());
            RawData msg = RawData.inbound(bytes, new UdpEndpointContext(new InetSocketAddress(datagram.getAddress(), datagram.getPort())), this.multicast, timestamp, connector);
            dataReceiver.receiveData(msg);
        }
    }

    public boolean getReuseAddress() {
        return this.reuseAddress;
    }

    public void setReuseAddress(boolean enable) {
        this.reuseAddress = enable;
    }

    public Integer getReceiveBufferSize() {
        return this.receiveBufferSize;
    }

    public Integer getSendBufferSize() {
        return this.sendBufferSize;
    }

    public int getReceiverThreadCount() {
        return this.receiverCount;
    }

    public int getSenderThreadCount() {
        return this.senderCount;
    }

    public int getReceiverPacketSize() {
        return this.receiverPacketSize;
    }

    @Override
    public String getProtocol() {
        return "UDP";
    }

    public String toString() {
        return this.getProtocol() + "-" + StringUtil.toString(this.getAddress());
    }

    static {
        ELEMENTS_THREAD_GROUP.setDaemon(false);
    }

    private class Receiver
    extends NetworkStageThread {
        private final DatagramPacket datagram;
        private final int size;

        private Receiver(String name) {
            super(name);
            this.size = UDPConnector.this.receiverPacketSize + 1;
            this.datagram = new DatagramPacket(new byte[this.size], this.size);
        }

        @Override
        protected void work() throws IOException {
            this.datagram.setLength(this.size);
            DatagramSocket currentSocket = UDPConnector.this.socket;
            if (currentSocket != null) {
                currentSocket.receive(this.datagram);
                UDPConnector.this.processDatagram(this.datagram);
            }
        }
    }

    private class Sender
    extends NetworkStageThread {
        private final DatagramPacket datagram;

        private Sender(String name) {
            super(name);
            this.datagram = new DatagramPacket(Bytes.EMPTY, 0);
        }

        @Override
        protected void work() throws InterruptedException {
            RawData raw = (RawData)UDPConnector.this.outgoing.take();
            EndpointContext destination = raw.getEndpointContext();
            InetSocketAddress destinationAddress = destination.getPeerAddress();
            UdpEndpointContext connectionContext = new UdpEndpointContext(destinationAddress);
            EndpointContextMatcher endpointMatcher = UDPConnector.this.endpointContextMatcher;
            if (endpointMatcher != null && !endpointMatcher.isToBeSent(destination, connectionContext)) {
                LOGGER.warn("UDPConnector ({}) drops {} bytes to {}", UDPConnector.this.effectiveAddr, this.datagram.getLength(), StringUtil.toLog(destinationAddress));
                raw.onError(new EndpointMismatchException("UDP sending"));
                return;
            }
            this.datagram.setData(raw.getBytes());
            this.datagram.setSocketAddress(destinationAddress);
            DatagramSocket currentSocket = UDPConnector.this.socket;
            if (currentSocket != null) {
                try {
                    raw.onContextEstablished(connectionContext);
                    currentSocket.send(this.datagram);
                    raw.onSent();
                }
                catch (IOException ex) {
                    raw.onError(ex);
                }
                LOGGER.debug("UDPConnector ({}) sent {} bytes to {}", this, this.datagram.getLength(), StringUtil.toLog(destinationAddress));
            } else {
                raw.onError(new IOException("socket already closed!"));
            }
        }
    }

    private abstract class NetworkStageThread
    extends Thread {
        protected NetworkStageThread(String name) {
            super(ELEMENTS_THREAD_GROUP, name);
            this.setDaemon(true);
        }

        @Override
        public void run() {
            LOGGER.debug("Starting network stage thread [{}]", (Object)this.getName());
            while (UDPConnector.this.running) {
                try {
                    this.work();
                    if (UDPConnector.this.running) continue;
                    LOGGER.debug("Network stage thread [{}] was stopped successfully", (Object)this.getName());
                    break;
                }
                catch (InterruptedIOException t) {
                    LOGGER.trace("Network stage thread [{}] was stopped successfully at:", (Object)this.getName(), (Object)t);
                }
                catch (InterruptedException t) {
                    LOGGER.trace("Network stage thread [{}] was stopped successfully at:", (Object)this.getName(), (Object)t);
                }
                catch (IOException t) {
                    if (UDPConnector.this.running) {
                        LOGGER.error("Exception in network stage thread [{}]:", (Object)this.getName(), (Object)t);
                        continue;
                    }
                    LOGGER.trace("Network stage thread [{}] was stopped successfully at:", (Object)this.getName(), (Object)t);
                }
                catch (Throwable t) {
                    LOGGER.error("Exception in network stage thread [{}]:", (Object)this.getName(), (Object)t);
                }
            }
        }

        protected abstract void work() throws Exception;
    }
}

