/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.util.nat;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.SocketException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Queue;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NioNatUtil
implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(NioNatUtil.class);
    private static final int DATAGRAM_SIZE = 2048;
    private static final int NAT_TIMEOUT_MS = 30000;
    private static final int LB_TIMEOUT_MS = 10000;
    private static final int MESSAGE_DROPPING_LOG_INTERVAL_MS = 10000;
    private static final ThreadGroup NAT_THREAD_GROUP = new ThreadGroup("NAT");
    private static final AtomicInteger NAT_THREAD_COUNTER = new AtomicInteger();
    private final Thread proxyThread;
    private final Queue<Runnable> jobs = new ConcurrentLinkedQueue<Runnable>();
    private final String proxyName;
    private final List<NatAddress> destinations;
    private final List<NatAddress> staleDestinations;
    private final ByteBuffer proxyBuffer;
    private final DatagramChannel proxyChannel;
    private final ConcurrentMap<InetSocketAddress, NatEntry> nats = new ConcurrentHashMap<InetSocketAddress, NatEntry>();
    private final Selector selector = Selector.open();
    private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(2, new ThreadFactory(){

        @Override
        public Thread newThread(Runnable runnable) {
            Thread ret = new Thread(NAT_THREAD_GROUP, runnable, "NAT-DELAY-" + NAT_THREAD_COUNTER.getAndIncrement(), 0L);
            ret.setDaemon(true);
            return ret;
        }
    });
    private volatile boolean running = true;
    private final Random random = new Random(System.nanoTime());
    private AtomicLong messageDroppingLogTime = new AtomicLong();
    private AtomicLong forwardCounter = new AtomicLong();
    private AtomicLong backwardCounter = new AtomicLong();
    private AtomicLong wrongRoutedCounter = new AtomicLong();
    private long lastWrongRoutedCounter;
    private AtomicLong timedoutEntriesCounter = new AtomicLong();
    private long lastTimedoutEntriesCounter;
    private AtomicInteger natTimeoutMillis = new AtomicInteger(30000);
    private AtomicInteger loadBalancerTimeoutMillis = new AtomicInteger(10000);
    private AtomicBoolean reverseNatUpdate = new AtomicBoolean();
    private volatile MessageDropping forward;
    private volatile MessageDropping backward;
    private volatile MessageSizeLimit forwardSizeLimit;
    private volatile MessageSizeLimit backwardSizeLimit;
    private volatile MessageReordering reorder;

    public NioNatUtil(InetSocketAddress bindAddress, InetSocketAddress destination) throws IOException {
        this.destinations = new ArrayList<NatAddress>();
        this.staleDestinations = new ArrayList<NatAddress>();
        this.addDestination(destination);
        this.proxyBuffer = ByteBuffer.allocateDirect(2048);
        this.proxyChannel = DatagramChannel.open();
        this.proxyChannel.configureBlocking(false);
        this.proxyChannel.bind(bindAddress);
        this.proxyChannel.register(this.selector, 1);
        InetSocketAddress proxy = (InetSocketAddress)this.proxyChannel.getLocalAddress();
        this.proxyName = proxy.getHostString() + ":" + proxy.getPort();
        this.proxyThread = new Thread(NAT_THREAD_GROUP, this, "NAT-" + proxy.getPort());
        this.proxyThread.start();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean addDestination(InetSocketAddress destination) {
        if (destination != null) {
            NatAddress dest = new NatAddress(destination);
            List<NatAddress> list = this.destinations;
            synchronized (list) {
                if (!this.destinations.contains(dest)) {
                    this.destinations.add(dest);
                    return true;
                }
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean removeDestination(InetSocketAddress destination) {
        if (destination != null) {
            List<NatAddress> list = this.destinations;
            synchronized (list) {
                for (NatAddress address : this.destinations) {
                    if (!address.address.equals(destination)) continue;
                    this.destinations.remove(address);
                    address.expired = true;
                    return true;
                }
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean addStaleDestinations() {
        boolean added = false;
        List<NatAddress> list = this.destinations;
        synchronized (list) {
            for (NatAddress address : this.staleDestinations) {
                address.reset();
                if (!this.destinations.add(address)) continue;
                added = true;
            }
            this.staleDestinations.clear();
        }
        return added;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        this.messageDroppingLogTime.set(System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(10000L));
        LOGGER.info("starting NAT {}.", (Object)this.proxyName);
        long lastTimeoutCheck = System.nanoTime();
        long lastLoadBalancerCheck = System.nanoTime();
        while (this.running) {
            try {
                long timeoutCheckMillis;
                Runnable job;
                if (this.messageDroppingLogTime.get() - System.nanoTime() < 0L) {
                    this.dumpMessageDroppingStatistic();
                }
                while ((job = this.jobs.poll()) != null) {
                    job.run();
                }
                long timeout = this.natTimeoutMillis.get();
                long socketTimeout = timeout > 0L ? timeout / 2L : 1000L;
                LOGGER.debug("Select {}ms, {} channels {} ready.", socketTimeout, this.selector.keys().size(), this.selector.selectedKeys().size());
                int num = this.selector.select(socketTimeout);
                if (num > 0) {
                    Set<SelectionKey> keys = this.selector.selectedKeys();
                    LOGGER.debug("Selected {} channels {} ready.", (Object)this.selector.keys().size(), (Object)keys.size());
                    for (SelectionKey key : keys) {
                        NatEntry entry = (NatEntry)key.attachment();
                        ((Buffer)this.proxyBuffer).clear();
                        if (entry != null) {
                            if (entry.receive(this.proxyBuffer) <= 0) continue;
                            entry.backward(this.proxyBuffer);
                            continue;
                        }
                        if (this.destinations.isEmpty()) continue;
                        InetSocketAddress source = (InetSocketAddress)this.proxyChannel.receive(this.proxyBuffer);
                        NatEntry newEntry = this.getNatEntry(source);
                        MessageReordering before = this.reorder;
                        if (before != null) {
                            before.forward(source, newEntry, this.proxyBuffer);
                            continue;
                        }
                        newEntry.forward(this.proxyBuffer);
                    }
                    keys.clear();
                }
                long now = System.nanoTime();
                long balancerTimeout = this.loadBalancerTimeoutMillis.get();
                if (balancerTimeout > 0L && (timeoutCheckMillis = TimeUnit.NANOSECONDS.toMillis(now - lastLoadBalancerCheck)) > balancerTimeout / 4L) {
                    lastLoadBalancerCheck = now;
                    long expireNanos = now - TimeUnit.MILLISECONDS.toNanos(balancerTimeout);
                    List<NatAddress> list = this.destinations;
                    synchronized (list) {
                        if (this.destinations.size() > 1) {
                            Iterator<NatAddress> iterator = this.destinations.iterator();
                            while (iterator.hasNext()) {
                                NatAddress dest = iterator.next();
                                if (!dest.expires(expireNanos)) continue;
                                iterator.remove();
                                this.staleDestinations.add(dest);
                                LOGGER.warn("expires {}", (Object)dest.name);
                                if (this.destinations.size() >= 2) continue;
                                break;
                            }
                        }
                    }
                }
                if (timeout <= 0L || (timeoutCheckMillis = TimeUnit.NANOSECONDS.toMillis(now - lastTimeoutCheck)) <= timeout / 4L) continue;
                lastTimeoutCheck = now;
                long expireNanos = now - TimeUnit.MILLISECONDS.toNanos(timeout);
                Iterator iterator = this.nats.values().iterator();
                while (iterator.hasNext()) {
                    NatEntry entry = (NatEntry)iterator.next();
                    if (!entry.expires(expireNanos)) continue;
                    iterator.remove();
                    this.timedoutEntriesCounter.incrementAndGet();
                }
            }
            catch (SocketException e) {
                if (!this.running) continue;
                LOGGER.error("NAT {} to {} socket error", this.proxyName, this.getDestinationForLogging(), e);
            }
            catch (InterruptedIOException e) {
                if (!this.running) continue;
                LOGGER.error("NAT {} to {} interrupted", this.proxyName, this.getDestinationForLogging(), e);
            }
            catch (Exception e) {
                LOGGER.error("NAT {} to {} error", this.proxyName, this.getDestinationForLogging(), e);
            }
        }
    }

    private NatEntry getNatEntry(InetSocketAddress source) throws IOException {
        NatEntry previousEntry;
        NatEntry entry = (NatEntry)this.nats.get(source);
        if (entry == null && (previousEntry = this.nats.putIfAbsent(source, entry = new NatEntry(source, this.selector))) != null) {
            entry.stop();
            entry = previousEntry;
        }
        return entry;
    }

    private void runTask(Runnable run) {
        this.jobs.add(run);
        this.selector.wakeup();
    }

    public void stop() {
        if (this.reorder != null) {
            this.reorder.stop();
        }
        this.running = false;
        try {
            this.proxyChannel.close();
        }
        catch (IOException e) {
            LOGGER.error("io-error on close!", e);
        }
        this.proxyThread.interrupt();
        this.stopAllNatEntries();
        this.scheduler.shutdownNow();
        try {
            this.proxyThread.join(1000L);
            this.scheduler.awaitTermination(1000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException ex) {
            LOGGER.error("shutdown failed!", ex);
        }
        try {
            this.selector.close();
        }
        catch (IOException e) {
            LOGGER.error("io-error on close!", e);
        }
        LOGGER.warn("NAT stopped. {} forwarded messages, {} backwarded", (Object)this.forwardCounter, (Object)this.backwardCounter);
    }

    public int stopAllNatEntries() {
        return this.stopNatEntries(this.nats.size());
    }

    public int stopNatEntries(int num) {
        int counter = 0;
        Iterator iterator = this.nats.values().iterator();
        while (num > 0 && iterator.hasNext()) {
            NatEntry entry = (NatEntry)iterator.next();
            iterator.remove();
            entry.stop();
            --num;
            ++counter;
        }
        return counter;
    }

    public void setNatTimeoutMillis(int natTimeoutMillis) {
        this.natTimeoutMillis.set(natTimeoutMillis);
    }

    public void setLoadBalancerTimeoutMillis(int loadBalancerTimeoutMillis) {
        this.loadBalancerTimeoutMillis.set(loadBalancerTimeoutMillis);
    }

    public int getNumberOfEntries() {
        return this.nats.size();
    }

    public int getNumberOfDestinations() {
        return this.destinations.size();
    }

    public int getNumberOfStaleDestinations() {
        return this.staleDestinations.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<NatAddress> getDestinations() {
        ArrayList<NatAddress> result = new ArrayList<NatAddress>();
        List<NatAddress> list = this.destinations;
        synchronized (list) {
            for (NatAddress address : this.destinations) {
                result.add(address);
            }
        }
        return result;
    }

    public int reassignDestinationAddresses() {
        int count = 0;
        if (this.destinations.size() > 1) {
            for (NatEntry entry : this.nats.values()) {
                if (!entry.setDestination(this.getRandomDestination())) continue;
                ++count;
            }
        }
        return count;
    }

    public void reassignNewLocalAddresses() {
        if (Thread.currentThread() != this.proxyThread) {
            final CountDownLatch ready = new CountDownLatch(1);
            this.runTask(new Runnable(){

                @Override
                public void run() {
                    NioNatUtil.this.reassignNewLocalAddresses();
                    ready.countDown();
                }
            });
            try {
                ready.await();
            }
            catch (InterruptedException interruptedException) {}
        } else {
            ArrayList<NatEntry> olds = new ArrayList<NatEntry>(this.nats.size());
            HashSet keys = new HashSet(this.nats.keySet());
            for (InetSocketAddress incoming : keys) {
                try {
                    NatEntry entry = new NatEntry(incoming, this.selector);
                    NatEntry old = this.nats.put(incoming, entry);
                    if (null != old) {
                        old.setIncoming(null);
                        olds.add(old);
                        LOGGER.info("changed NAT for {} from {} to {}.", incoming, old.getPort(), entry.getPort());
                        continue;
                    }
                    LOGGER.info("add NAT for {} to {}.", (Object)incoming, (Object)entry.getPort());
                }
                catch (IOException e) {
                    LOGGER.error("Failed to reassing NAT entry for {}.", (Object)incoming, (Object)e);
                }
            }
            for (NatEntry old : olds) {
                old.stop();
            }
        }
    }

    public int assignLocalAddress(final InetSocketAddress incoming) throws IOException {
        if (Thread.currentThread() != this.proxyThread) {
            final AtomicInteger port = new AtomicInteger();
            final AtomicReference error = new AtomicReference();
            final CountDownLatch ready = new CountDownLatch(1);
            this.runTask(new Runnable(){

                @Override
                public void run() {
                    try {
                        int p = NioNatUtil.this.assignLocalAddress(incoming);
                        port.set(p);
                    }
                    catch (IOException e) {
                        error.set(e);
                    }
                    ready.countDown();
                }
            });
            try {
                ready.await();
                if (error.get() != null) {
                    throw (IOException)error.get();
                }
                return port.get();
            }
            catch (InterruptedException e) {
                return 0;
            }
        }
        NatEntry entry = new NatEntry(incoming, this.selector);
        NatEntry old = this.nats.put(incoming, entry);
        if (null != old) {
            LOGGER.info("changed NAT for {} from {} to {}.", incoming, old.getPort(), entry.getPort());
            old.stop();
        } else {
            LOGGER.info("add NAT for {} to {}.", (Object)incoming, (Object)entry.getPort());
        }
        return entry.getPort();
    }

    public void mixLocalAddresses() {
        Random random = new Random();
        ArrayList<NatEntry> destinations = new ArrayList<NatEntry>();
        HashSet keys = new HashSet(this.nats.keySet());
        for (InetSocketAddress incoming : keys) {
            NatEntry entry = (NatEntry)this.nats.remove(incoming);
            entry.setIncoming(null);
            destinations.add(entry);
        }
        for (InetSocketAddress incoming : keys) {
            int index = random.nextInt(destinations.size());
            NatEntry entry = (NatEntry)destinations.remove(index);
            entry.setIncoming(incoming);
            NatEntry temp = this.nats.put(incoming, entry);
            if (temp == null || temp == entry) continue;
            temp.stop();
        }
    }

    public boolean removeLocalAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.remove(incoming);
        if (null != entry) {
            entry.stop();
        } else {
            LOGGER.warn("no mapping found for {}!", (Object)incoming);
        }
        return null != entry;
    }

    public int getLocalPortForAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.get(incoming);
        if (null != entry) {
            return entry.getPort();
        }
        LOGGER.warn("no mapping found for {}!", (Object)incoming);
        return -1;
    }

    public InetSocketAddress getLocalAddressForAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.get(incoming);
        if (null != entry) {
            return entry.getSocketAddres();
        }
        LOGGER.warn("no mapping found for {}!", (Object)incoming);
        return null;
    }

    public InetSocketAddress getProxySocketAddress() throws IOException {
        return (InetSocketAddress)this.proxyChannel.getLocalAddress();
    }

    public void setMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forward != null || this.backward != null) {
                this.forward = null;
                this.backward = null;
                LOGGER.info("NAT stops message dropping.");
            }
        } else {
            this.forward = new MessageDropping("request", percent);
            this.backward = new MessageDropping("responses", percent);
            LOGGER.info("NAT message dropping {}%.", (Object)percent);
        }
    }

    public void setForwardMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forward != null) {
                this.forward = null;
                LOGGER.info("NAT stops forward message dropping.");
            }
        } else {
            this.forward = new MessageDropping("request", percent);
            LOGGER.info("NAT forward message dropping {}%.", (Object)percent);
        }
    }

    public void setBackwardMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.backward != null) {
                this.backward = null;
                LOGGER.info("NAT stops backward message dropping.");
            }
        } else {
            this.backward = new MessageDropping("response", percent);
            LOGGER.info("NAT backward message dropping {}%.", (Object)percent);
        }
    }

    public void setMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forwardSizeLimit != null || this.backwardSizeLimit != null) {
                this.forwardSizeLimit = null;
                this.backwardSizeLimit = null;
                LOGGER.info("NAT stops message size limit.");
            }
        } else {
            this.forwardSizeLimit = new MessageSizeLimit("request", percent, sizeLimit, drop);
            this.backwardSizeLimit = new MessageSizeLimit("responses", percent, sizeLimit, drop);
            LOGGER.info("NAT message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setForwardMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forwardSizeLimit != null) {
                this.forwardSizeLimit = null;
                LOGGER.info("NAT stops forward message size limit.");
            }
        } else {
            this.forwardSizeLimit = new MessageSizeLimit("request", percent, sizeLimit, drop);
            LOGGER.info("NAT forward message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setBackwardMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.backwardSizeLimit != null) {
                this.backwardSizeLimit = null;
                LOGGER.info("NAT stops backward message size limit.");
            }
        } else {
            this.backwardSizeLimit = new MessageSizeLimit("response", percent, sizeLimit, drop);
            LOGGER.info("NAT backward message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setMessageReordering(int percent, int delayMillis, int randomDelayMillis) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message reordering " + percent + "% out of range [0...100]!");
        }
        if (this.reorder != null) {
            this.reorder.stop();
        }
        if (percent == 0) {
            if (this.reorder != null) {
                this.reorder = null;
                LOGGER.info("NAT stops message reordering.");
            }
        } else {
            this.reorder = new MessageReordering("reordering", percent, delayMillis, randomDelayMillis);
            LOGGER.info("NAT message reordering {}%.", (Object)percent);
        }
    }

    public void setReverseNatUpdate(boolean reverseUpdate) {
        this.reverseNatUpdate.set(reverseUpdate);
    }

    public boolean useReverseNatUpdate() {
        return this.reverseNatUpdate.get();
    }

    public void dumpMessageDroppingStatistic() {
        long current;
        this.messageDroppingLogTime.set(System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(10000L));
        TransmissionManipulation drops = this.forward;
        if (drops != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.backward) != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.forwardSizeLimit) != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.backwardSizeLimit) != null) {
            drops.dumpStatistic();
        }
        if (this.lastWrongRoutedCounter < (current = this.wrongRoutedCounter.get())) {
            LOGGER.warn("wrong routed messages {} (overall {}).", (Object)(current - this.lastWrongRoutedCounter), (Object)this.lastWrongRoutedCounter);
            this.lastWrongRoutedCounter = current;
        }
        if (this.lastTimedoutEntriesCounter < (current = this.timedoutEntriesCounter.get())) {
            LOGGER.warn("timed out NAT entries {} (overall {}).", (Object)(current - this.lastTimedoutEntriesCounter), (Object)this.lastTimedoutEntriesCounter);
            this.lastTimedoutEntriesCounter = current;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NatAddress getRandomDestination() {
        if (this.destinations.isEmpty()) {
            return null;
        }
        List<NatAddress> list = this.destinations;
        synchronized (list) {
            int size = this.destinations.size();
            if (size == 1) {
                return this.destinations.get(0);
            }
            int index = this.random.nextInt(size);
            return this.destinations.get(index);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NatAddress getDestination(InetSocketAddress destination) {
        if (destination != null) {
            List<NatAddress> list = this.destinations;
            synchronized (list) {
                for (NatAddress address : this.destinations) {
                    if (!address.address.equals(destination)) continue;
                    return address;
                }
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String getDestinationForLogging() {
        if (this.destinations.isEmpty()) {
            return "---";
        }
        List<NatAddress> list = this.destinations;
        synchronized (list) {
            int size = this.destinations.size();
            if (size == 1) {
                return this.destinations.get((int)0).name;
            }
            return this.destinations.get((int)0).name + "-" + this.destinations.get((int)(size - 1)).name;
        }
    }

    static {
        NAT_THREAD_GROUP.setDaemon(false);
    }

    private class NatEntry {
        private final DatagramChannel outgoing;
        private final String natName;
        private final InetSocketAddress local;
        private NatAddress incoming;
        private NatAddress destination;

        public NatEntry(InetSocketAddress incoming, Selector selector) throws IOException {
            this.setDestination(NioNatUtil.this.getRandomDestination());
            this.outgoing = DatagramChannel.open();
            this.outgoing.configureBlocking(false);
            this.outgoing.bind(null);
            this.local = (InetSocketAddress)this.outgoing.getLocalAddress();
            this.natName = Integer.toString(this.local.getPort());
            this.setIncoming(incoming);
            this.outgoing.register(selector, 1, this);
        }

        public synchronized boolean setDestination(NatAddress destination) {
            if (this.destination == destination) {
                return false;
            }
            if (this.destination != null) {
                if (this.destination.equals(destination)) {
                    return false;
                }
                this.destination.usageCounter.decrementAndGet();
            }
            this.destination = destination;
            if (this.destination != null) {
                this.destination.usageCounter.incrementAndGet();
            }
            return true;
        }

        public synchronized void setIncoming(InetSocketAddress incoming) {
            this.incoming = incoming != null ? new NatAddress(incoming) : null;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public boolean expires(long expireNanos) {
            NatAddress incoming;
            NatEntry natEntry = this;
            synchronized (natEntry) {
                incoming = this.incoming;
            }
            if (incoming == null) {
                return true;
            }
            if (incoming.expires(expireNanos)) {
                this.stop();
                LOGGER.info("expired listen on {} for incoming {}", (Object)this.natName, (Object)incoming.name);
                return true;
            }
            return false;
        }

        public void stop() {
            try {
                if (this.destination != null) {
                    this.destination.usageCounter.decrementAndGet();
                }
                this.outgoing.close();
            }
            catch (IOException e) {
                LOGGER.error("IO-error on closing", e);
            }
        }

        public InetSocketAddress getSocketAddres() {
            return this.local;
        }

        public int getPort() {
            return this.local.getPort();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public int receive(ByteBuffer packet) throws IOException {
            NatAddress destination;
            NatEntry natEntry = this;
            synchronized (natEntry) {
                destination = this.destination;
            }
            try {
                SocketAddress source = this.outgoing.receive(packet);
                if (destination.address.equals(source)) {
                    destination.updateReceive();
                } else {
                    NioNatUtil.this.wrongRoutedCounter.incrementAndGet();
                    if (NioNatUtil.this.reverseNatUpdate.get()) {
                        NatAddress newDestination = NioNatUtil.this.getDestination((InetSocketAddress)source);
                        this.setDestination(newDestination);
                    } else {
                        ((Buffer)packet).clear();
                    }
                }
                return packet.position();
            }
            catch (IOException ex) {
                return -1;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void backward(ByteBuffer packet) throws IOException {
            NatAddress destination;
            NatAddress incoming;
            NatEntry natEntry = this;
            synchronized (natEntry) {
                incoming = this.incoming;
                destination = this.destination;
            }
            if (incoming == null) {
                return;
            }
            incoming.updateUsage();
            MessageDropping dropping = NioNatUtil.this.backward;
            if (dropping != null && dropping.dropMessage()) {
                LOGGER.debug("backward drops {} bytes from {} to {} via {}", packet.position(), destination.name, incoming.name, this.natName);
            } else {
                MessageSizeLimit limit = NioNatUtil.this.backwardSizeLimit;
                MessageSizeLimit.Manipulation manipulation = limit != null ? limit.limitMessageSize(packet) : MessageSizeLimit.Manipulation.NONE;
                switch (manipulation) {
                    case NONE: {
                        LOGGER.debug("backward {} bytes from {} to {} via {}", packet.position(), destination.name, incoming.name, this.natName);
                        break;
                    }
                    case DROP: {
                        LOGGER.debug("backward drops {} bytes from {} to {} via {}", packet.position(), destination.name, incoming.name, this.natName);
                        break;
                    }
                    case LIMIT: {
                        LOGGER.debug("backward limited {} bytes from {} to {} via {}", packet.position(), destination.name, incoming.name, this.natName);
                    }
                }
                if (manipulation != MessageSizeLimit.Manipulation.DROP) {
                    ((Buffer)packet).flip();
                    if (NioNatUtil.this.proxyChannel.send(packet, incoming.address) == 0) {
                        LOGGER.debug("backward overloaded {} bytes from {} to {} via {}", packet.position(), destination.name, incoming.name, this.natName);
                    } else {
                        NioNatUtil.this.backwardCounter.incrementAndGet();
                    }
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public boolean forward(ByteBuffer packet) throws IOException {
            MessageDropping dropping;
            NatAddress destination;
            NatAddress incoming;
            NatEntry natEntry = this;
            synchronized (natEntry) {
                incoming = this.incoming;
                destination = this.destination;
            }
            if (incoming == null) {
                return false;
            }
            incoming.updateUsage();
            if (destination.expired) {
                destination.usageCounter.decrementAndGet();
                destination = NioNatUtil.this.getRandomDestination();
                destination.usageCounter.incrementAndGet();
                natEntry = this;
                synchronized (natEntry) {
                    this.destination = destination;
                }
            }
            if ((dropping = NioNatUtil.this.forward) != null && dropping.dropMessage()) {
                LOGGER.debug("forward drops {} bytes from {} to {} via {}", packet.position(), incoming.name, destination.name, this.natName);
            } else {
                MessageSizeLimit limit = NioNatUtil.this.forwardSizeLimit;
                MessageSizeLimit.Manipulation manipulation = limit != null ? limit.limitMessageSize(packet) : MessageSizeLimit.Manipulation.NONE;
                switch (manipulation) {
                    case NONE: {
                        LOGGER.debug("forward {} bytes from {} to {} via {}", packet.position(), incoming.name, destination.name, this.natName);
                        break;
                    }
                    case DROP: {
                        LOGGER.debug("forward drops {} bytes from {} to {} via {}", packet.position(), incoming.name, destination.name, this.natName);
                        break;
                    }
                    case LIMIT: {
                        LOGGER.debug("forward limited {} bytes from {} to {} via {}", packet.position(), incoming.name, destination.name, this.natName);
                    }
                }
                if (manipulation != MessageSizeLimit.Manipulation.DROP) {
                    ((Buffer)packet).flip();
                    if (this.outgoing.send(packet, destination.address) == 0) {
                        LOGGER.info("forward overloaded {} bytes from {} to {} via {}", packet.limit(), incoming.name, destination.name, this.natName);
                        return false;
                    }
                    destination.updateSend();
                    NioNatUtil.this.forwardCounter.incrementAndGet();
                }
            }
            return true;
        }
    }

    private class MessageReordering
    extends TransmissionManipulation {
        private final int delayMillis;
        private final int randomDelayMillis;
        private boolean reordering;

        public MessageReordering(String title, int threshold, int delayMillis, int randomDelayMillis) {
            super(title + " reorders", threshold);
            this.reordering = true;
            this.delayMillis = delayMillis;
            this.randomDelayMillis = randomDelayMillis;
        }

        public void forward(final InetSocketAddress source, NatEntry entry, ByteBuffer data) throws IOException {
            if (!this.isRunning()) {
                return;
            }
            if (this.manipulateMessage()) {
                ((Buffer)data).flip();
                final ByteBuffer clone = ByteBuffer.allocate(data.limit());
                clone.put(data);
                final long delay = this.delayMillis + this.random.nextInt(this.randomDelayMillis);
                NioNatUtil.this.scheduler.schedule(new Runnable(){

                    @Override
                    public void run() {
                        if (MessageReordering.this.isRunning()) {
                            try {
                                LOGGER.info("deliver message {} bytes, delayed {}ms for {}", clone.position(), delay, source);
                                NatEntry entry = (NatEntry)NioNatUtil.this.nats.get(source);
                                if (entry != null) {
                                    entry.forward(clone);
                                }
                            }
                            catch (IOException ex) {
                                LOGGER.info("delayed forward failed!", ex);
                            }
                        }
                    }
                }, delay, TimeUnit.MILLISECONDS);
            } else {
                entry.forward(data);
            }
        }

        public synchronized void stop() {
            this.reordering = false;
        }

        private synchronized boolean isRunning() {
            return this.reordering;
        }
    }

    private static class MessageDropping
    extends TransmissionManipulation {
        public MessageDropping(String title, int threshold) {
            super(title + " drops", threshold);
        }

        public boolean dropMessage() {
            return this.manipulateMessage();
        }
    }

    private static class MessageSizeLimit
    extends TransmissionManipulation {
        private final boolean drop;
        private final int sizeLimit;

        public MessageSizeLimit(String title, int threshold, int sizeLimit, boolean drop) {
            super(title + " size limit", threshold);
            this.sizeLimit = sizeLimit;
            this.drop = drop;
        }

        public Manipulation limitMessageSize(ByteBuffer packet) {
            if (packet.position() > this.sizeLimit && this.manipulateMessage()) {
                if (this.drop) {
                    return Manipulation.DROP;
                }
                packet.position(this.sizeLimit);
                return Manipulation.LIMIT;
            }
            return Manipulation.NONE;
        }

        private static enum Manipulation {
            NONE,
            DROP,
            LIMIT;

        }
    }

    private static class TransmissionManipulation {
        private final String title;
        protected final Random random = new Random();
        private final int threshold;
        private final AtomicInteger sentMessages = new AtomicInteger();
        private final AtomicInteger manipulatedMessages = new AtomicInteger();

        public TransmissionManipulation(String title, int threshold) {
            this.title = title;
            this.threshold = threshold;
            this.random.setSeed(threshold);
        }

        public boolean manipulateMessage() {
            if (this.threshold == 0) {
                return false;
            }
            if (this.threshold == 100) {
                return true;
            }
            if (this.threshold > this.random.nextInt(100)) {
                this.manipulatedMessages.incrementAndGet();
                return true;
            }
            this.sentMessages.incrementAndGet();
            return false;
        }

        public void dumpStatistic() {
            int sent = this.sentMessages.get();
            int manipulated = this.manipulatedMessages.get();
            if (sent > 0) {
                LOGGER.warn("manipulated {} {}/{}%, sent {} {}.", this.title, manipulated, manipulated * 100 / (manipulated + sent), this.title, sent);
            } else if (manipulated > 0) {
                LOGGER.warn("manipulated {} {}/100%, no {} sent!.", this.title, manipulated, this.title);
            }
        }
    }

    public static class NatAddress {
        public final InetSocketAddress address;
        public final String name;
        private final AtomicInteger usageCounter;
        private long lastNanos;
        private boolean expired;

        private NatAddress(InetSocketAddress address) {
            this.address = address;
            this.name = address.getHostString() + ":" + address.getPort();
            this.usageCounter = new AtomicInteger();
            this.updateReceive();
        }

        private void reset() {
            this.expired = false;
            this.updateReceive();
        }

        private synchronized void updateUsage() {
            if (!this.expired) {
                this.lastNanos = System.nanoTime();
            }
        }

        private synchronized void updateSend() {
            if (!this.expired && this.lastNanos < 0L) {
                this.lastNanos = System.nanoTime();
            }
        }

        private synchronized void updateReceive() {
            this.lastNanos = -1L;
        }

        private synchronized boolean expires(long expireNanos) {
            if (!this.expired) {
                this.expired = this.lastNanos > 0L && expireNanos - this.lastNanos > 0L;
            }
            return this.expired;
        }

        public int hashCode() {
            return this.address.hashCode();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            NatAddress other = (NatAddress)obj;
            return this.address.equals(other.address);
        }

        public int usageCounter() {
            return this.usageCounter.get();
        }
    }
}

