/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.util.nat;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NatUtil
implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(NatUtil.class);
    private static final int DATAGRAM_SIZE = 2048;
    private static final int NAT_TIMEOUT_MS = 60000;
    private static final int MESSAGE_DROPPING_LOG_INTERVAL_MS = 10000;
    private static final ThreadGroup NAT_THREAD_GROUP = new ThreadGroup("NAT");
    private static final AtomicInteger NAT_THREAD_COUNTER = new AtomicInteger();
    private final Thread proxyThread;
    private final String proxyName;
    private final InetSocketAddress[] destinations;
    private final String[] destinationNames;
    private final DatagramSocket proxySocket;
    private final DatagramPacket proxyPacket;
    private final ConcurrentMap<InetSocketAddress, NatEntry> nats = new ConcurrentHashMap<InetSocketAddress, NatEntry>();
    private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(2, new ThreadFactory(){

        @Override
        public Thread newThread(Runnable runnable) {
            Thread ret = new Thread(NAT_THREAD_GROUP, runnable, "NAT-" + NAT_THREAD_COUNTER.getAndIncrement(), 0L);
            ret.setDaemon(true);
            return ret;
        }
    });
    private volatile boolean running = true;
    private final Random random = new Random(System.nanoTime());
    private AtomicLong messageDroppingLogTime = new AtomicLong();
    private AtomicLong forwardCounter = new AtomicLong();
    private AtomicLong backwardCounter = new AtomicLong();
    private AtomicInteger natTimeoutMillis = new AtomicInteger(60000);
    private volatile MessageDropping forward;
    private volatile MessageDropping backward;
    private volatile MessageSizeLimit forwardSizeLimit;
    private volatile MessageSizeLimit backwardSizeLimit;
    private volatile MessageReordering reorder;

    public NatUtil(InetSocketAddress bindAddress, InetSocketAddress destination) throws Exception {
        this(bindAddress, new InetSocketAddress[]{destination});
    }

    public NatUtil(InetSocketAddress bindAddress, List<InetSocketAddress> destinations) throws Exception {
        this(bindAddress, destinations.toArray(new InetSocketAddress[0]));
    }

    public NatUtil(InetSocketAddress bindAddress, InetSocketAddress ... destinations) throws Exception {
        this.destinations = destinations;
        this.proxySocket = bindAddress == null ? new DatagramSocket() : new DatagramSocket(bindAddress);
        InetSocketAddress proxy = (InetSocketAddress)this.proxySocket.getLocalSocketAddress();
        this.proxyName = proxy.getHostString() + ":" + proxy.getPort();
        this.destinationNames = new String[destinations.length];
        for (int index = 0; index < destinations.length; ++index) {
            this.destinationNames[index] = destinations[index].getHostString() + ":" + destinations[index].getPort();
        }
        this.proxyPacket = new DatagramPacket(new byte[2048], 2048);
        this.proxyThread = new Thread(NAT_THREAD_GROUP, this, "NAT-" + proxy.getPort());
        this.proxyThread.start();
    }

    @Override
    public void run() {
        this.messageDroppingLogTime.set(System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(10000L));
        if (this.destinations.length == 1) {
            LOGGER.info("starting NAT {} to {}.", (Object)this.proxyName, (Object)this.destinationNames[0]);
        } else {
            LOGGER.info("starting NAT-LB {} to {}-{}.", this.proxyName, this.destinationNames[0], this.destinationNames[this.destinationNames.length - 1]);
        }
        while (this.running) {
            try {
                if (this.messageDroppingLogTime.get() - System.nanoTime() < 0L) {
                    this.dumpMessageDroppingStatistic();
                }
                this.proxyPacket.setLength(2048);
                this.proxySocket.setSoTimeout(this.getSocketTimeout());
                this.proxySocket.receive(this.proxyPacket);
                MessageReordering before = this.reorder;
                if (before != null) {
                    before.forward(this.proxyPacket);
                    continue;
                }
                this.deliver(this.proxyPacket);
            }
            catch (SocketTimeoutException e) {
                if (!this.running) continue;
                if (this.destinations.length == 1) {
                    LOGGER.debug("listen NAT {} to {}.", (Object)this.proxyName, (Object)this.destinationNames[0]);
                    continue;
                }
                LOGGER.debug("listen NAT-LB {} to {}-{}.", this.proxyName, this.destinationNames[0], this.destinationNames[this.destinationNames.length - 1]);
            }
            catch (SocketException e) {
                if (!this.running) continue;
                if (this.destinations.length == 1) {
                    LOGGER.error("NAT {} to {} socket error", this.proxyName, this.destinationNames[0], e);
                    continue;
                }
                LOGGER.error("NAT-LB {} to {}-{} socket error", this.proxyName, this.destinationNames[0], this.destinationNames[this.destinationNames.length - 1], e);
            }
            catch (InterruptedIOException e) {
                if (!this.running) continue;
                if (this.destinations.length == 1) {
                    LOGGER.error("NAT {} to {} interrupted", this.proxyName, this.destinationNames[0], e);
                    continue;
                }
                LOGGER.error("NAT-LB {} to {}-{} interrupted", this.proxyName, this.destinationNames[0], this.destinationNames[this.destinationNames.length - 1], e);
            }
            catch (Exception e) {
                if (this.destinations.length == 1) {
                    LOGGER.error("NAT {} to {} error", this.proxyName, this.destinationNames[0], e);
                    continue;
                }
                LOGGER.error("NAT-LB {} to {}-{} error", this.proxyName, this.destinationNames[0], this.destinationNames[this.destinationNames.length - 1], e);
            }
        }
    }

    public void deliver(DatagramPacket packet) throws IOException {
        if (this.running) {
            NatEntry previousEntry;
            InetSocketAddress incoming = (InetSocketAddress)packet.getSocketAddress();
            NatEntry entry = (NatEntry)this.nats.get(incoming);
            if (null == entry && (previousEntry = this.nats.putIfAbsent(incoming, entry = new NatEntry(incoming))) != null) {
                entry.stop();
                entry = previousEntry;
            }
            entry.forward(packet);
        }
    }

    public void stop() {
        this.running = false;
        this.proxySocket.close();
        this.proxyThread.interrupt();
        this.stopAllNatEntries();
        this.scheduler.shutdownNow();
        try {
            this.proxyThread.join(1000L);
            this.scheduler.awaitTermination(1000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException ex) {
            LOGGER.error("shutdown failed!", ex);
        }
        LOGGER.warn("NAT stopped. {} forwarded messages, {} backwarded", (Object)this.forwardCounter, (Object)this.backwardCounter);
    }

    public void stopAllNatEntries() {
        for (NatEntry entry : this.nats.values()) {
            entry.stop();
        }
        this.nats.clear();
    }

    public void setNatTimeoutMillis(int natTimeoutMillis) {
        this.natTimeoutMillis.set(natTimeoutMillis);
    }

    public int getNumberOfEntries() {
        return this.nats.size();
    }

    public int reassignDestinationAddresses() {
        int count = 0;
        if (this.destinations.length > 1) {
            for (NatEntry entry : this.nats.values()) {
                if (!entry.setDestination(this.getRandomDestination())) continue;
                ++count;
            }
        }
        return count;
    }

    public void reassignNewLocalAddresses() {
        HashSet keys = new HashSet(this.nats.keySet());
        for (InetSocketAddress incoming : keys) {
            try {
                this.assignLocalAddress(incoming);
            }
            catch (SocketException e) {
                LOGGER.error("Failed to reassing NAT entry for {}.", (Object)incoming, (Object)e);
            }
        }
    }

    public int assignLocalAddress(InetSocketAddress incoming) throws SocketException {
        NatEntry entry = new NatEntry(incoming);
        NatEntry old = this.nats.put(incoming, entry);
        if (null != old) {
            LOGGER.info("changed NAT for {} from {} to {}.", incoming, old.getPort(), entry.getPort());
            old.stop();
        } else {
            LOGGER.info("add NAT for {} to {}.", (Object)incoming, (Object)entry.getPort());
        }
        return entry.getPort();
    }

    public void mixLocalAddresses() {
        Random random = new Random();
        ArrayList<NatEntry> destinations = new ArrayList<NatEntry>();
        HashSet keys = new HashSet(this.nats.keySet());
        for (InetSocketAddress incoming : keys) {
            NatEntry entry = (NatEntry)this.nats.remove(incoming);
            destinations.add(entry);
        }
        for (InetSocketAddress incoming : keys) {
            int index = random.nextInt(destinations.size());
            NatEntry entry = (NatEntry)destinations.remove(index);
            entry.setIncoming(incoming);
            this.nats.put(incoming, entry);
        }
    }

    public boolean removeLocalAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.remove(incoming);
        if (null != entry) {
            entry.stop();
        } else {
            LOGGER.warn("no mapping found for {}!", (Object)incoming);
        }
        return null != entry;
    }

    public int getLocalPortForAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.get(incoming);
        if (null != entry) {
            return entry.getPort();
        }
        LOGGER.warn("no mapping found for {}!", (Object)incoming);
        return -1;
    }

    public InetSocketAddress getLocalAddressForAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.get(incoming);
        if (null != entry) {
            return entry.getSocketAddress();
        }
        LOGGER.warn("no mapping found for {}!", (Object)incoming);
        return null;
    }

    public InetSocketAddress getProxySocketAddress() {
        return (InetSocketAddress)this.proxySocket.getLocalSocketAddress();
    }

    public void setMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forward != null || this.backward != null) {
                this.forward = null;
                this.backward = null;
                LOGGER.info("NAT stops message dropping.");
            }
        } else {
            this.forward = new MessageDropping("request", percent);
            this.backward = new MessageDropping("responses", percent);
            LOGGER.info("NAT message dropping {}%.", (Object)percent);
        }
    }

    public void setForwardMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forward != null) {
                this.forward = null;
                LOGGER.info("NAT stops forward message dropping.");
            }
        } else {
            this.forward = new MessageDropping("request", percent);
            LOGGER.info("NAT forward message dropping {}%.", (Object)percent);
        }
    }

    public void setBackwardMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.backward != null) {
                this.backward = null;
                LOGGER.info("NAT stops backward message dropping.");
            }
        } else {
            this.backward = new MessageDropping("response", percent);
            LOGGER.info("NAT backward message dropping {}%.", (Object)percent);
        }
    }

    public void setMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forwardSizeLimit != null || this.backwardSizeLimit != null) {
                this.forwardSizeLimit = null;
                this.backwardSizeLimit = null;
                LOGGER.info("NAT stops message size limit.");
            }
        } else {
            this.forwardSizeLimit = new MessageSizeLimit("request", percent, sizeLimit, drop);
            this.backwardSizeLimit = new MessageSizeLimit("responses", percent, sizeLimit, drop);
            LOGGER.info("NAT message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setForwardMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forwardSizeLimit != null) {
                this.forwardSizeLimit = null;
                LOGGER.info("NAT stops forward message size limit.");
            }
        } else {
            this.forwardSizeLimit = new MessageSizeLimit("request", percent, sizeLimit, drop);
            LOGGER.info("NAT forward message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setBackwardMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.backwardSizeLimit != null) {
                this.backwardSizeLimit = null;
                LOGGER.info("NAT stops backward message size limit.");
            }
        } else {
            this.backwardSizeLimit = new MessageSizeLimit("response", percent, sizeLimit, drop);
            LOGGER.info("NAT backward message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setMessageReordering(int percent, int delayMillis, int randomDelayMillis) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message reordering " + percent + "% out of range [0...100]!");
        }
        if (this.reorder != null) {
            this.reorder.stop();
        }
        if (percent == 0) {
            if (this.reorder != null) {
                this.reorder = null;
                LOGGER.info("NAT stops message reordering.");
            }
        } else {
            this.reorder = new MessageReordering("reordering", percent, delayMillis, randomDelayMillis);
            LOGGER.info("NAT message reordering {}%.", (Object)percent);
        }
    }

    public void dumpMessageDroppingStatistic() {
        this.messageDroppingLogTime.set(System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(10000L));
        TransmissionManipulation drops = this.forward;
        if (drops != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.backward) != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.forwardSizeLimit) != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.backwardSizeLimit) != null) {
            drops.dumpStatistic();
        }
    }

    public InetSocketAddress getRandomDestination() {
        if (this.destinations.length == 1) {
            return this.destinations[0];
        }
        int index = this.random.nextInt(this.destinations.length);
        return this.destinations[index];
    }

    private int getSocketTimeout() {
        return this.natTimeoutMillis.get() / 2;
    }

    static {
        NAT_THREAD_GROUP.setDaemon(false);
    }

    private class NatEntry
    implements Runnable {
        private final DatagramSocket outgoingSocket;
        private final DatagramPacket packet;
        private final String natName;
        private final Thread thread;
        private String incomingName;
        private InetSocketAddress incoming;
        private String destinationName;
        private InetSocketAddress destination;
        private volatile boolean running = true;
        private final AtomicLong lastUsage = new AtomicLong(System.nanoTime());

        public NatEntry(InetSocketAddress incoming) throws SocketException {
            this.setIncoming(incoming);
            this.setDestination(NatUtil.this.getRandomDestination());
            this.outgoingSocket = new DatagramSocket(0);
            this.packet = new DatagramPacket(new byte[2048], 2048);
            this.natName = Integer.toString(this.outgoingSocket.getLocalPort());
            this.thread = new Thread(NAT_THREAD_GROUP, this, "NAT-ENTRY-" + incoming.getPort());
            this.thread.start();
        }

        public synchronized boolean setDestination(InetSocketAddress destination) {
            if (this.destination == null || !this.destination.equals(destination)) {
                this.destination = destination;
                this.destinationName = destination.getHostString() + ":" + destination.getPort();
                return true;
            }
            return false;
        }

        public synchronized void setIncoming(InetSocketAddress incoming) {
            this.incoming = incoming;
            this.incomingName = incoming.getHostString() + ":" + incoming.getPort();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            LOGGER.info("start listening on {} for incoming {}", (Object)this.natName, (Object)this.incomingName);
            try {
                boolean timeout = false;
                while (this.running && !timeout) {
                    Object destinationName;
                    String incomingName;
                    try {
                        InetSocketAddress incoming;
                        this.packet.setLength(2048);
                        this.outgoingSocket.setSoTimeout(NatUtil.this.getSocketTimeout());
                        this.outgoingSocket.receive(this.packet);
                        this.lastUsage.set(System.nanoTime());
                        NatEntry natEntry = this;
                        synchronized (natEntry) {
                            incoming = this.incoming;
                            incomingName = this.incomingName;
                            destinationName = this.destinationName;
                        }
                        this.packet.setSocketAddress(incoming);
                        MessageDropping dropping = NatUtil.this.backward;
                        if (dropping != null && dropping.dropMessage()) {
                            LOGGER.debug("backward drops {} bytes from {} to {} via {}", this.packet.getLength(), destinationName, incomingName, this.natName);
                            continue;
                        }
                        MessageSizeLimit limit = NatUtil.this.backwardSizeLimit;
                        MessageSizeLimit.Manipulation manipulation = limit != null ? limit.limitMessageSize(this.packet) : MessageSizeLimit.Manipulation.NONE;
                        switch (manipulation) {
                            case NONE: {
                                LOGGER.debug("backward {} bytes from {} to {} via {}", this.packet.getLength(), destinationName, incomingName, this.natName);
                                break;
                            }
                            case DROP: {
                                LOGGER.debug("backward drops {} bytes from {} to {} via {}", this.packet.getLength(), destinationName, incomingName, this.natName);
                                break;
                            }
                            case LIMIT: {
                                LOGGER.debug("backward limited {} bytes from {} to {} via {}", this.packet.getLength(), destinationName, incomingName, this.natName);
                            }
                        }
                        if (manipulation == MessageSizeLimit.Manipulation.DROP) continue;
                        NatUtil.this.proxySocket.send(this.packet);
                        NatUtil.this.backwardCounter.incrementAndGet();
                    }
                    catch (SocketTimeoutException e) {
                        if (!this.running) continue;
                        destinationName = this;
                        synchronized (destinationName) {
                            incomingName = this.incomingName;
                        }
                        long quietPeriodMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - this.lastUsage.get());
                        if (quietPeriodMillis > (long)NatUtil.this.natTimeoutMillis.get()) {
                            timeout = true;
                            LOGGER.info("expired listen on {} for incoming {}", (Object)this.natName, (Object)incomingName);
                            continue;
                        }
                        LOGGER.trace("listen on {} for incoming {}", (Object)this.natName, (Object)incomingName);
                    }
                    catch (IOException e) {
                        if (!this.running) continue;
                        NatEntry natEntry = this;
                        synchronized (natEntry) {
                            incomingName = this.incomingName;
                        }
                        LOGGER.info("error occured on {} for incoming {}", this.natName, incomingName, e);
                    }
                }
            }
            finally {
                String incomingName;
                InetSocketAddress incoming;
                NatEntry natEntry = this;
                synchronized (natEntry) {
                    incoming = this.incoming;
                    incomingName = this.incomingName;
                }
                LOGGER.info("stop listen on {} for incoming {}", (Object)this.natName, (Object)incomingName);
                this.outgoingSocket.close();
                if (this.running) {
                    NatUtil.this.nats.remove(incoming, this);
                }
            }
        }

        public void stop() {
            this.running = false;
            this.outgoingSocket.close();
            this.thread.interrupt();
            try {
                this.thread.join(2000L);
            }
            catch (InterruptedException e) {
                LOGGER.error("shutdown failed!", e);
            }
        }

        public InetSocketAddress getSocketAddress() {
            return (InetSocketAddress)this.outgoingSocket.getLocalSocketAddress();
        }

        public int getPort() {
            return this.outgoingSocket.getLocalPort();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void forward(DatagramPacket packet) throws IOException {
            InetSocketAddress destination;
            String destinationName;
            String incomingName;
            NatEntry natEntry = this;
            synchronized (natEntry) {
                incomingName = this.incomingName;
                destinationName = this.destinationName;
                destination = this.destination;
            }
            MessageDropping dropping = NatUtil.this.forward;
            if (dropping != null && dropping.dropMessage()) {
                LOGGER.debug("forward drops {} bytes from {} to {} via {}", packet.getLength(), incomingName, destinationName, this.natName);
            } else {
                MessageSizeLimit limit = NatUtil.this.forwardSizeLimit;
                MessageSizeLimit.Manipulation manipulation = limit != null ? limit.limitMessageSize(packet) : MessageSizeLimit.Manipulation.NONE;
                switch (manipulation) {
                    case NONE: {
                        LOGGER.debug("forward {} bytes from {} to {} via {}", packet.getLength(), incomingName, destinationName, this.natName);
                        break;
                    }
                    case DROP: {
                        LOGGER.debug("forward drops {} bytes from {} to {} via {}", packet.getLength(), incomingName, destinationName, this.natName);
                        break;
                    }
                    case LIMIT: {
                        LOGGER.debug("forward limited {} bytes from {} to {} via {}", packet.getLength(), incomingName, destinationName, this.natName);
                    }
                }
                if (manipulation != MessageSizeLimit.Manipulation.DROP) {
                    packet.setSocketAddress(destination);
                    this.lastUsage.set(System.nanoTime());
                    this.outgoingSocket.send(packet);
                    NatUtil.this.forwardCounter.incrementAndGet();
                }
            }
        }
    }

    private class MessageReordering
    extends TransmissionManipulation {
        private final NatEntry entry;
        private final int delayMillis;
        private final int randomDelayMillis;
        private boolean reordering;

        public MessageReordering(String title, int threshold, int delayMillis, int randomDelayMillis) {
            super(title + " reorders", threshold);
            this.reordering = true;
            this.delayMillis = delayMillis;
            this.randomDelayMillis = randomDelayMillis;
            this.entry = null;
        }

        public MessageReordering(String title, NatEntry entry, int threshold, int delayMillis, int randomDelayMillis) {
            super(title + " reorders", threshold);
            this.reordering = true;
            this.delayMillis = delayMillis;
            this.randomDelayMillis = randomDelayMillis;
            this.entry = entry;
        }

        public void forward(DatagramPacket packet) throws IOException {
            if (this.manipulateMessage()) {
                final long delay = this.delayMillis + this.random.nextInt(this.randomDelayMillis);
                byte[] data = Arrays.copyOfRange(packet.getData(), packet.getOffset(), packet.getOffset() + packet.getLength());
                final DatagramPacket clone = new DatagramPacket(data, data.length, packet.getSocketAddress());
                NatUtil.this.scheduler.schedule(new Runnable(){

                    @Override
                    public void run() {
                        if (MessageReordering.this.isRunning()) {
                            try {
                                if (MessageReordering.this.entry != null) {
                                    LOGGER.info("send message {} bytes, delayed {}ms to {}", clone.getLength(), delay, clone.getSocketAddress());
                                    MessageReordering.this.entry.forward(clone);
                                } else {
                                    LOGGER.info("deliver message {} bytes, delayed {}ms to {}", clone.getLength(), delay, clone.getSocketAddress());
                                    NatUtil.this.deliver(clone);
                                }
                            }
                            catch (IOException ex) {
                                LOGGER.info("delayed forward failed!", ex);
                            }
                        }
                    }
                }, delay, TimeUnit.MILLISECONDS);
            } else {
                NatUtil.this.deliver(packet);
            }
        }

        public synchronized void stop() {
            this.reordering = false;
        }

        private synchronized boolean isRunning() {
            return this.reordering;
        }
    }

    private static class MessageDropping
    extends TransmissionManipulation {
        public MessageDropping(String title, int threshold) {
            super(title + " drops", threshold);
        }

        public boolean dropMessage() {
            return this.manipulateMessage();
        }
    }

    private static class MessageSizeLimit
    extends TransmissionManipulation {
        private final boolean drop;
        private final int sizeLimit;

        public MessageSizeLimit(String title, int threshold, int sizeLimit, boolean drop) {
            super(title + " size limit", threshold);
            this.sizeLimit = sizeLimit;
            this.drop = drop;
        }

        public Manipulation limitMessageSize(DatagramPacket packet) {
            if (packet.getLength() > this.sizeLimit && this.manipulateMessage()) {
                if (this.drop) {
                    return Manipulation.DROP;
                }
                packet.setLength(this.sizeLimit);
                return Manipulation.LIMIT;
            }
            return Manipulation.NONE;
        }

        private static enum Manipulation {
            NONE,
            DROP,
            LIMIT;

        }
    }

    private static class TransmissionManipulation {
        private final String title;
        protected final Random random = new Random();
        private final int threshold;
        private final AtomicInteger sentMessages = new AtomicInteger();
        private final AtomicInteger manipulatedMessages = new AtomicInteger();

        public TransmissionManipulation(String title, int threshold) {
            this.title = title;
            this.threshold = threshold;
            this.random.setSeed(threshold);
        }

        public boolean manipulateMessage() {
            if (this.threshold == 0) {
                return false;
            }
            if (this.threshold == 100) {
                return true;
            }
            if (this.threshold > this.random.nextInt(100)) {
                this.manipulatedMessages.incrementAndGet();
                return true;
            }
            this.sentMessages.incrementAndGet();
            return false;
        }

        public void dumpStatistic() {
            int sent = this.sentMessages.get();
            int manipulated = this.manipulatedMessages.get();
            if (sent > 0) {
                LOGGER.warn("manipulated {} {}/{}%, sent {} {}.", this.title, manipulated, manipulated * 100 / (manipulated + sent), this.title, sent);
            } else if (manipulated > 0) {
                LOGGER.warn("manipulated {} {}/100%, no {} sent!.", this.title, manipulated, this.title);
            }
        }
    }
}

