/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.util.nat;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.SocketException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NioNatUtil
implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(NioNatUtil.class);
    private static final int DATAGRAM_SIZE = 2048;
    private static final int NAT_TIMEOUT_MS = 60000;
    private static final int MESSAGE_DROPPING_LOG_INTERVAL_MS = 10000;
    private static final ThreadGroup NAT_THREAD_GROUP = new ThreadGroup("NAT");
    private static final AtomicInteger NAT_THREAD_COUNTER = new AtomicInteger();
    private final Thread proxyThread;
    private final String proxyName;
    private final InetSocketAddress[] destinations;
    private final String[] destinationNames;
    private final ByteBuffer proxyBuffer;
    private final DatagramChannel proxyChannel;
    private final ConcurrentMap<InetSocketAddress, NatEntry> nats = new ConcurrentHashMap<InetSocketAddress, NatEntry>();
    private final Selector selector = Selector.open();
    private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(2, new ThreadFactory(){

        @Override
        public Thread newThread(Runnable runnable) {
            Thread ret = new Thread(NAT_THREAD_GROUP, runnable, "NAT-DELAY-" + NAT_THREAD_COUNTER.getAndIncrement(), 0L);
            ret.setDaemon(true);
            return ret;
        }
    });
    private volatile boolean running = true;
    private final Random random = new Random(System.nanoTime());
    private AtomicLong messageDroppingLogTime = new AtomicLong();
    private AtomicLong forwardCounter = new AtomicLong();
    private AtomicLong backwardCounter = new AtomicLong();
    private AtomicInteger natTimeoutMillis = new AtomicInteger(60000);
    private volatile MessageDropping forward;
    private volatile MessageDropping backward;
    private volatile MessageSizeLimit forwardSizeLimit;
    private volatile MessageSizeLimit backwardSizeLimit;
    private volatile MessageReordering reorder;

    public NioNatUtil(InetSocketAddress bindAddress, InetSocketAddress destination) throws Exception {
        this(bindAddress, new InetSocketAddress[]{destination});
    }

    public NioNatUtil(InetSocketAddress bindAddress, List<InetSocketAddress> destinations) throws Exception {
        this(bindAddress, destinations.toArray(new InetSocketAddress[0]));
    }

    public NioNatUtil(InetSocketAddress bindAddress, InetSocketAddress ... destinations) throws Exception {
        this.destinations = destinations;
        this.proxyBuffer = ByteBuffer.allocateDirect(2048);
        this.proxyChannel = DatagramChannel.open();
        this.proxyChannel.configureBlocking(false);
        this.proxyChannel.bind(bindAddress);
        this.proxyChannel.register(this.selector, 1);
        InetSocketAddress proxy = (InetSocketAddress)this.proxyChannel.getLocalAddress();
        this.proxyName = proxy.getHostString() + ":" + proxy.getPort();
        this.destinationNames = new String[destinations.length];
        for (int index = 0; index < destinations.length; ++index) {
            this.destinationNames[index] = destinations[index].getHostString() + ":" + destinations[index].getPort();
        }
        this.proxyThread = new Thread(NAT_THREAD_GROUP, this, "NAT-" + proxy.getPort());
        this.proxyThread.start();
    }

    @Override
    public void run() {
        this.messageDroppingLogTime.set(System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(10000L));
        if (this.destinations.length == 1) {
            LOGGER.info("starting NAT {} to {}.", (Object)this.proxyName, (Object)this.destinationNames[0]);
        } else {
            LOGGER.info("starting NAT-LB {} to {}-{}.", this.proxyName, this.destinationNames[0], this.destinationNames[this.destinationNames.length - 1]);
        }
        long lastTimeoutCheck = System.nanoTime();
        while (this.running) {
            try {
                long now;
                long timeoutCheckMillis;
                if (this.messageDroppingLogTime.get() - System.nanoTime() < 0L) {
                    this.dumpMessageDroppingStatistic();
                }
                long timeout = this.getSocketTimeout();
                LOGGER.debug("Select {}ms, {} channels {} ready.", timeout, this.selector.keys().size(), this.selector.selectedKeys().size());
                int num = this.selector.select(timeout);
                if (num > 0) {
                    Set<SelectionKey> keys = this.selector.selectedKeys();
                    LOGGER.debug("Selected {} channels {} ready.", (Object)this.selector.keys().size(), (Object)keys.size());
                    for (SelectionKey key : keys) {
                        NatEntry entry = (NatEntry)key.attachment();
                        if (entry != null) {
                            if (entry.receive(this.proxyBuffer) <= 0) continue;
                            entry.backward(this.proxyBuffer);
                            continue;
                        }
                        ((Buffer)this.proxyBuffer).clear();
                        SocketAddress source = this.proxyChannel.receive(this.proxyBuffer);
                        NatEntry newEntry = this.getNatEntry(source);
                        MessageReordering before = this.reorder;
                        if (before != null) {
                            before.forward(source, newEntry, this.proxyBuffer);
                            continue;
                        }
                        newEntry.forward(this.proxyBuffer);
                    }
                    keys.clear();
                }
                if ((timeoutCheckMillis = TimeUnit.NANOSECONDS.toMillis((now = System.nanoTime()) - lastTimeoutCheck)) <= (long)(this.natTimeoutMillis.get() / 4)) continue;
                lastTimeoutCheck = now;
                for (NatEntry entry : this.nats.values()) {
                    entry.timeout(now);
                }
            }
            catch (SocketException e) {
                if (!this.running) continue;
                if (this.destinations.length == 1) {
                    LOGGER.error("NAT {} to {} socket error", this.proxyName, this.destinationNames[0], e);
                    continue;
                }
                LOGGER.error("NAT-LB {} to {}-{} socket error", this.proxyName, this.destinationNames[0], this.destinationNames[this.destinationNames.length - 1], e);
            }
            catch (InterruptedIOException e) {
                if (!this.running) continue;
                if (this.destinations.length == 1) {
                    LOGGER.error("NAT {} to {} interrupted", this.proxyName, this.destinationNames[0], e);
                    continue;
                }
                LOGGER.error("NAT-LB {} to {}-{} interrupted", this.proxyName, this.destinationNames[0], this.destinationNames[this.destinationNames.length - 1], e);
            }
            catch (Exception e) {
                if (this.destinations.length == 1) {
                    LOGGER.error("NAT {} to {} error", this.proxyName, this.destinationNames[0], e);
                    continue;
                }
                LOGGER.error("NAT-LB {} to {}-{} error", this.proxyName, this.destinationNames[0], this.destinationNames[this.destinationNames.length - 1], e);
            }
        }
    }

    public NatEntry getNatEntry(SocketAddress source) throws IOException {
        NatEntry previousEntry;
        InetSocketAddress incoming = (InetSocketAddress)source;
        NatEntry entry = (NatEntry)this.nats.get(incoming);
        if (entry == null && (previousEntry = this.nats.putIfAbsent(incoming, entry = new NatEntry(incoming, this.selector))) != null) {
            entry.stop();
            entry = previousEntry;
        }
        return entry;
    }

    public void stop() {
        this.running = false;
        try {
            this.proxyChannel.close();
        }
        catch (IOException e) {
            LOGGER.error("io-error on close!", e);
        }
        this.proxyThread.interrupt();
        this.stopAllNatEntries();
        this.scheduler.shutdownNow();
        try {
            this.proxyThread.join(1000L);
            this.scheduler.awaitTermination(1000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException ex) {
            LOGGER.error("shutdown failed!", ex);
        }
        try {
            this.selector.close();
        }
        catch (IOException e) {
            LOGGER.error("io-error on close!", e);
        }
        LOGGER.warn("NAT stopped. {} forwarded messages, {} backwarded", (Object)this.forwardCounter, (Object)this.backwardCounter);
    }

    public void stopAllNatEntries() {
        for (NatEntry entry : this.nats.values()) {
            entry.stop();
        }
        this.nats.clear();
    }

    public void setNatTimeoutMillis(int natTimeoutMillis) {
        this.natTimeoutMillis.set(natTimeoutMillis);
    }

    public int getNumberOfEntries() {
        return this.nats.size();
    }

    public int reassignDestinationAddresses() {
        int count = 0;
        if (this.destinations.length > 1) {
            for (NatEntry entry : this.nats.values()) {
                if (!entry.setDestination(this.getRandomDestination())) continue;
                ++count;
            }
        }
        return count;
    }

    public void reassignNewLocalAddresses() {
        HashSet keys = new HashSet(this.nats.keySet());
        for (InetSocketAddress incoming : keys) {
            try {
                this.assignLocalAddress(incoming);
            }
            catch (IOException e) {
                LOGGER.error("Failed to reassing NAT entry for {}.", (Object)incoming, (Object)e);
            }
        }
    }

    public int assignLocalAddress(InetSocketAddress incoming) throws IOException {
        NatEntry entry = new NatEntry(incoming, this.selector);
        NatEntry old = this.nats.put(incoming, entry);
        if (null != old) {
            LOGGER.info("changed NAT for {} from {} to {}.", incoming, old.getPort(), entry.getPort());
            old.stop();
        } else {
            LOGGER.info("add NAT for {} to {}.", (Object)incoming, (Object)entry.getPort());
        }
        return entry.getPort();
    }

    public void mixLocalAddresses() {
        Random random = new Random();
        ArrayList<NatEntry> destinations = new ArrayList<NatEntry>();
        HashSet keys = new HashSet(this.nats.keySet());
        for (InetSocketAddress incoming : keys) {
            NatEntry entry = (NatEntry)this.nats.remove(incoming);
            destinations.add(entry);
        }
        for (InetSocketAddress incoming : keys) {
            int index = random.nextInt(destinations.size());
            NatEntry entry = (NatEntry)destinations.remove(index);
            entry.setIncoming(incoming);
            this.nats.put(incoming, entry);
        }
    }

    public boolean removeLocalAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.remove(incoming);
        if (null != entry) {
            entry.stop();
        } else {
            LOGGER.warn("no mapping found for {}!", (Object)incoming);
        }
        return null != entry;
    }

    public int getLocalPortForAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.get(incoming);
        if (null != entry) {
            return entry.getPort();
        }
        LOGGER.warn("no mapping found for {}!", (Object)incoming);
        return -1;
    }

    public InetSocketAddress getLocalAddressForAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.get(incoming);
        if (null != entry) {
            return entry.getSocketAddres();
        }
        LOGGER.warn("no mapping found for {}!", (Object)incoming);
        return null;
    }

    public InetSocketAddress getProxySocketAddress() throws IOException {
        return (InetSocketAddress)this.proxyChannel.getLocalAddress();
    }

    public void setMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forward != null || this.backward != null) {
                this.forward = null;
                this.backward = null;
                LOGGER.info("NAT stops message dropping.");
            }
        } else {
            this.forward = new MessageDropping("request", percent);
            this.backward = new MessageDropping("responses", percent);
            LOGGER.info("NAT message dropping {}%.", (Object)percent);
        }
    }

    public void setForwardMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forward != null) {
                this.forward = null;
                LOGGER.info("NAT stops forward message dropping.");
            }
        } else {
            this.forward = new MessageDropping("request", percent);
            LOGGER.info("NAT forward message dropping {}%.", (Object)percent);
        }
    }

    public void setBackwardMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.backward != null) {
                this.backward = null;
                LOGGER.info("NAT stops backward message dropping.");
            }
        } else {
            this.backward = new MessageDropping("response", percent);
            LOGGER.info("NAT backward message dropping {}%.", (Object)percent);
        }
    }

    public void setMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forwardSizeLimit != null || this.backwardSizeLimit != null) {
                this.forwardSizeLimit = null;
                this.backwardSizeLimit = null;
                LOGGER.info("NAT stops message size limit.");
            }
        } else {
            this.forwardSizeLimit = new MessageSizeLimit("request", percent, sizeLimit, drop);
            this.backwardSizeLimit = new MessageSizeLimit("responses", percent, sizeLimit, drop);
            LOGGER.info("NAT message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setForwardMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forwardSizeLimit != null) {
                this.forwardSizeLimit = null;
                LOGGER.info("NAT stops forward message size limit.");
            }
        } else {
            this.forwardSizeLimit = new MessageSizeLimit("request", percent, sizeLimit, drop);
            LOGGER.info("NAT forward message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setBackwardMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.backwardSizeLimit != null) {
                this.backwardSizeLimit = null;
                LOGGER.info("NAT stops backward message size limit.");
            }
        } else {
            this.backwardSizeLimit = new MessageSizeLimit("response", percent, sizeLimit, drop);
            LOGGER.info("NAT backward message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setMessageReordering(int percent, int delayMillis, int randomDelayMillis) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message reordering " + percent + "% out of range [0...100]!");
        }
        if (this.reorder != null) {
            this.reorder.stop();
        }
        if (percent == 0) {
            if (this.reorder != null) {
                this.reorder = null;
                LOGGER.info("NAT stops message reordering.");
            }
        } else {
            this.reorder = new MessageReordering("reordering", percent, delayMillis, randomDelayMillis);
            LOGGER.info("NAT message reordering {}%.", (Object)percent);
        }
    }

    public void dumpMessageDroppingStatistic() {
        this.messageDroppingLogTime.set(System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(10000L));
        TransmissionManipulation drops = this.forward;
        if (drops != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.backward) != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.forwardSizeLimit) != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.backwardSizeLimit) != null) {
            drops.dumpStatistic();
        }
    }

    public InetSocketAddress getRandomDestination() {
        if (this.destinations.length == 1) {
            return this.destinations[0];
        }
        int index = this.random.nextInt(this.destinations.length);
        return this.destinations[index];
    }

    private int getSocketTimeout() {
        return this.natTimeoutMillis.get() / 2;
    }

    static {
        NAT_THREAD_GROUP.setDaemon(false);
    }

    private class NatEntry {
        private final DatagramChannel outgoing;
        private final String natName;
        private final AtomicLong lastUsage = new AtomicLong(System.nanoTime());
        private final InetSocketAddress local;
        private String incomingName;
        private InetSocketAddress incoming;
        private String destinationName;
        private InetSocketAddress destination;

        public NatEntry(InetSocketAddress incoming, Selector selector) throws IOException {
            this.setIncoming(incoming);
            this.setDestination(NioNatUtil.this.getRandomDestination());
            this.outgoing = DatagramChannel.open();
            this.outgoing.configureBlocking(false);
            this.outgoing.bind(null);
            this.local = (InetSocketAddress)this.outgoing.getLocalAddress();
            this.outgoing.register(selector, 1, this);
            this.natName = Integer.toString(this.local.getPort());
        }

        public synchronized boolean setDestination(InetSocketAddress destination) {
            if (this.destination == null || !this.destination.equals(destination)) {
                this.destination = destination;
                this.destinationName = destination.getHostString() + ":" + destination.getPort();
                return true;
            }
            return false;
        }

        public synchronized void setIncoming(InetSocketAddress incoming) {
            this.incoming = incoming;
            this.incomingName = incoming.getHostString() + ":" + incoming.getPort();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void timeout(long now) {
            long quietPeriodMillis = TimeUnit.NANOSECONDS.toMillis(now - this.lastUsage.get());
            if (quietPeriodMillis > (long)NioNatUtil.this.natTimeoutMillis.get()) {
                String incomingName;
                NatEntry natEntry = this;
                synchronized (natEntry) {
                    incomingName = this.incomingName;
                }
                LOGGER.info("expired listen on {} for incoming {}", (Object)this.natName, (Object)incomingName);
                this.stop();
                NioNatUtil.this.nats.remove(this.incoming, this);
            }
        }

        public void stop() {
            try {
                this.outgoing.close();
            }
            catch (IOException e) {
                LOGGER.error("IO-error on closing", e);
            }
        }

        public InetSocketAddress getSocketAddres() {
            return this.local;
        }

        public int getPort() {
            return this.local.getPort();
        }

        public int receive(ByteBuffer packet) throws IOException {
            packet.clear();
            this.outgoing.receive(packet);
            return packet.position();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void backward(ByteBuffer packet) throws IOException {
            String destinationName;
            String incomingName;
            InetSocketAddress incoming;
            this.lastUsage.set(System.nanoTime());
            NatEntry natEntry = this;
            synchronized (natEntry) {
                incoming = this.incoming;
                incomingName = this.incomingName;
                destinationName = this.destinationName;
            }
            MessageDropping dropping = NioNatUtil.this.backward;
            if (dropping != null && dropping.dropMessage()) {
                LOGGER.debug("backward drops {} bytes from {} to {} via {}", packet.position(), destinationName, incomingName, this.natName);
            } else {
                MessageSizeLimit limit = NioNatUtil.this.backwardSizeLimit;
                MessageSizeLimit.Manipulation manipulation = limit != null ? limit.limitMessageSize(packet) : MessageSizeLimit.Manipulation.NONE;
                switch (manipulation) {
                    case NONE: {
                        LOGGER.debug("backward {} bytes from {} to {} via {}", packet.position(), destinationName, incomingName, this.natName);
                        break;
                    }
                    case DROP: {
                        LOGGER.debug("backward drops {} bytes from {} to {} via {}", packet.position(), destinationName, incomingName, this.natName);
                        break;
                    }
                    case LIMIT: {
                        LOGGER.debug("backward limited {} bytes from {} to {} via {}", packet.position(), destinationName, incomingName, this.natName);
                    }
                }
                if (manipulation != MessageSizeLimit.Manipulation.DROP) {
                    ((Buffer)packet).flip();
                    if (NioNatUtil.this.proxyChannel.send(packet, incoming) == 0) {
                        LOGGER.debug("backward overloaded {} bytes from {} to {} via {}", packet.position(), destinationName, incomingName, this.natName);
                    } else {
                        NioNatUtil.this.backwardCounter.incrementAndGet();
                    }
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void forward(ByteBuffer packet) throws IOException {
            InetSocketAddress destination;
            String destinationName;
            String incomingName;
            this.lastUsage.set(System.nanoTime());
            NatEntry natEntry = this;
            synchronized (natEntry) {
                incomingName = this.incomingName;
                destinationName = this.destinationName;
                destination = this.destination;
            }
            MessageDropping dropping = NioNatUtil.this.forward;
            if (dropping != null && dropping.dropMessage()) {
                LOGGER.debug("forward drops {} bytes from {} to {} via {}", packet.position(), incomingName, destinationName, this.natName);
            } else {
                MessageSizeLimit limit = NioNatUtil.this.forwardSizeLimit;
                MessageSizeLimit.Manipulation manipulation = limit != null ? limit.limitMessageSize(packet) : MessageSizeLimit.Manipulation.NONE;
                switch (manipulation) {
                    case NONE: {
                        LOGGER.debug("forward {} bytes from {} to {} via {}", packet.position(), incomingName, destinationName, this.natName);
                        break;
                    }
                    case DROP: {
                        LOGGER.debug("forward drops {} bytes from {} to {} via {}", packet.position(), incomingName, destinationName, this.natName);
                        break;
                    }
                    case LIMIT: {
                        LOGGER.debug("forward limited {} bytes from {} to {} via {}", packet.position(), incomingName, destinationName, this.natName);
                    }
                }
                if (manipulation != MessageSizeLimit.Manipulation.DROP) {
                    ((Buffer)packet).flip();
                    if (this.outgoing.send(packet, destination) == 0) {
                        LOGGER.warn("forward overloaded {} bytes from {} to {} via {}", packet.position(), incomingName, destinationName, this.natName);
                    } else {
                        NioNatUtil.this.forwardCounter.incrementAndGet();
                    }
                }
            }
        }
    }

    private class MessageReordering
    extends TransmissionManipulation {
        private final int delayMillis;
        private final int randomDelayMillis;
        private boolean reordering;

        public MessageReordering(String title, int threshold, int delayMillis, int randomDelayMillis) {
            super(title + " reorders", threshold);
            this.reordering = true;
            this.delayMillis = delayMillis;
            this.randomDelayMillis = randomDelayMillis;
        }

        public void forward(final SocketAddress source, final NatEntry entry, ByteBuffer data) throws IOException {
            if (this.manipulateMessage()) {
                final ByteBuffer clone = data.duplicate();
                final long delay = this.delayMillis + this.random.nextInt(this.randomDelayMillis);
                NioNatUtil.this.scheduler.schedule(new Runnable(){

                    @Override
                    public void run() {
                        if (MessageReordering.this.isRunning()) {
                            try {
                                LOGGER.info("deliver message {} bytes, delayed {}ms for {}", clone.position(), delay, source);
                                entry.forward(clone);
                            }
                            catch (IOException ex) {
                                LOGGER.info("delayed forward failed!", ex);
                            }
                        }
                    }
                }, delay, TimeUnit.MILLISECONDS);
            } else {
                entry.forward(data);
            }
        }

        public synchronized void stop() {
            this.reordering = false;
        }

        private synchronized boolean isRunning() {
            return this.reordering;
        }
    }

    private static class MessageDropping
    extends TransmissionManipulation {
        public MessageDropping(String title, int threshold) {
            super(title + " drops", threshold);
        }

        public boolean dropMessage() {
            return this.manipulateMessage();
        }
    }

    private static class MessageSizeLimit
    extends TransmissionManipulation {
        private final boolean drop;
        private final int sizeLimit;

        public MessageSizeLimit(String title, int threshold, int sizeLimit, boolean drop) {
            super(title + " size limit", threshold);
            this.sizeLimit = sizeLimit;
            this.drop = drop;
        }

        public Manipulation limitMessageSize(ByteBuffer packet) {
            if (packet.position() > this.sizeLimit && this.manipulateMessage()) {
                if (this.drop) {
                    return Manipulation.DROP;
                }
                packet.position(this.sizeLimit);
                return Manipulation.LIMIT;
            }
            return Manipulation.NONE;
        }

        private static enum Manipulation {
            NONE,
            DROP,
            LIMIT;

        }
    }

    private static class TransmissionManipulation {
        private final String title;
        protected final Random random = new Random();
        private final int threshold;
        private final AtomicInteger sentMessages = new AtomicInteger();
        private final AtomicInteger manipulatedMessages = new AtomicInteger();

        public TransmissionManipulation(String title, int threshold) {
            this.title = title;
            this.threshold = threshold;
            this.random.setSeed(threshold);
        }

        public boolean manipulateMessage() {
            if (this.threshold == 0) {
                return false;
            }
            if (this.threshold == 100) {
                return true;
            }
            if (this.threshold > this.random.nextInt(100)) {
                this.manipulatedMessages.incrementAndGet();
                return true;
            }
            this.sentMessages.incrementAndGet();
            return false;
        }

        public void dumpStatistic() {
            int sent = this.sentMessages.get();
            int manipulated = this.manipulatedMessages.get();
            if (sent > 0) {
                LOGGER.warn("manipulated {} {}/{}%, sent {} {}.", this.title, manipulated, manipulated * 100 / (manipulated + sent), this.title, sent);
            } else if (manipulated > 0) {
                LOGGER.warn("manipulated {} {}/100%, no {} sent!.", this.title, manipulated, this.title);
            }
        }
    }
}

