/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network.stack;

import java.util.logging.Logger;
import org.eclipse.californium.core.coap.BlockOption;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.MessageObserverAdapter;
import org.eclipse.californium.core.coap.OptionSet;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.config.NetworkConfigObserverAdapter;
import org.eclipse.californium.core.network.stack.AbstractLayer;
import org.eclipse.californium.core.network.stack.BlockwiseStatus;

public class BlockwiseLayer
extends AbstractLayer {
    protected static final Logger LOGGER = Logger.getLogger(BlockwiseLayer.class.getCanonicalName());
    private int maxMsgSize;
    private int defaultBlockSize;

    public BlockwiseLayer(NetworkConfig config) {
        this.maxMsgSize = config.getInt("MAX_MESSAGE_SIZE");
        this.defaultBlockSize = config.getInt("DEFAULT_BLOCK_SIZE");
        LOGGER.config("BlockwiseLayer uses MAX_MESSAGE_SIZE: " + this.maxMsgSize + " and DEFAULT_BLOCK_SIZE:" + this.defaultBlockSize);
        config.addConfigObserver(new NetworkConfigObserverAdapter(){

            @Override
            public void changed(String key, int value) {
                if ("MAX_MESSAGE_SIZE".equals(key)) {
                    BlockwiseLayer.this.maxMsgSize = value;
                }
                if ("DEFAULT_BLOCK_SIZE".equals(key)) {
                    BlockwiseLayer.this.defaultBlockSize = value;
                }
            }
        });
    }

    @Override
    public void sendRequest(Exchange exchange, Request request) {
        if (request.getOptions().hasBlock2() && request.getOptions().getBlock2().getNum() > 0) {
            LOGGER.fine("Request carries explicit defined block2 option: create random access blockwise status");
            BlockwiseStatus status = new BlockwiseStatus(request.getOptions().getContentFormat());
            BlockOption block2 = request.getOptions().getBlock2();
            status.setCurrentSzx(block2.getSzx());
            status.setCurrentNum(block2.getNum());
            status.setRandomAccess(true);
            exchange.setResponseBlockStatus(status);
            super.sendRequest(exchange, request);
        } else if (this.requiresBlockwise(request)) {
            LOGGER.fine("Request payload " + request.getPayloadSize() + "/" + this.maxMsgSize + " requires Blockwise");
            BlockwiseStatus status = this.findRequestBlockStatus(exchange, request);
            Request block = this.getNextRequestBlock(request, status);
            exchange.setRequestBlockStatus(status);
            exchange.setCurrentRequest(block);
            super.sendRequest(exchange, block);
        } else {
            exchange.setCurrentRequest(request);
            super.sendRequest(exchange, request);
        }
    }

    @Override
    public void receiveRequest(Exchange exchange, Request request) {
        if (request.getOptions().hasBlock1()) {
            BlockOption block1 = request.getOptions().getBlock1();
            LOGGER.fine("Request contains block1 option " + block1);
            BlockwiseStatus status = this.findRequestBlockStatus(exchange, request);
            if (block1.getNum() == 0 && status.getCurrentNum() > 0) {
                LOGGER.finer("Block1 num is 0, the client has restarted the blockwise transfer. Reset status.");
                status = new BlockwiseStatus(request.getOptions().getContentFormat());
                exchange.setRequestBlockStatus(status);
            }
            if (block1.getNum() == status.getCurrentNum()) {
                if (request.getOptions().getContentFormat() != status.getContentFormat()) {
                    Response error = Response.createPiggybackedResponse(request, CoAP.ResponseCode.REQUEST_ENTITY_INCOMPLETE);
                    error.getOptions().setBlock1(block1.getSzx(), block1.isM(), block1.getNum());
                    error.setPayload("Changed Content-Format");
                    request.setAcknowledged(true);
                    exchange.setCurrentResponse(error);
                    super.sendResponse(exchange, error);
                    return;
                }
                status.addBlock(request.getPayload());
                status.setCurrentNum(status.getCurrentNum() + 1);
                if (block1.isM()) {
                    LOGGER.finest("There are more blocks to come. Acknowledge this block.");
                    if (request.isConfirmable()) {
                        Response piggybacked = Response.createPiggybackedResponse(request, CoAP.ResponseCode.CONTINUE);
                        piggybacked.getOptions().setBlock1(block1.getSzx(), true, block1.getNum());
                        piggybacked.setLast(false);
                        request.setAcknowledged(true);
                        exchange.setCurrentResponse(piggybacked);
                        super.sendResponse(exchange, piggybacked);
                    }
                } else {
                    LOGGER.finer("This was the last block. Deliver request");
                    exchange.setBlock1ToAck(block1);
                    this.earlyBlock2Negotiation(exchange, request);
                    Request assembled = new Request(request.getCode());
                    this.assembleMessage(status, assembled, request);
                    exchange.setRequest(assembled);
                    super.receiveRequest(exchange, assembled);
                }
            } else {
                LOGGER.warning("Wrong block number. Expected " + status.getCurrentNum() + " but received " + block1.getNum() + ". Respond with 4.08 (Request Entity Incomplete)");
                Response error = Response.createPiggybackedResponse(request, CoAP.ResponseCode.REQUEST_ENTITY_INCOMPLETE);
                error.getOptions().setBlock1(block1.getSzx(), block1.isM(), block1.getNum());
                error.setPayload("Wrong block number");
                request.setAcknowledged(true);
                exchange.setCurrentResponse(error);
                super.sendResponse(exchange, error);
            }
        } else if (exchange.getResponse() != null && request.getOptions().hasBlock2()) {
            BlockOption block2 = request.getOptions().getBlock2();
            Response response = exchange.getResponse();
            BlockwiseStatus status = this.findResponseBlockStatus(exchange, response);
            status.setCurrentNum(block2.getNum());
            status.setCurrentSzx(block2.getSzx());
            Response block = this.getNextResponseBlock(response, status);
            block.setToken(request.getToken());
            block.getOptions().removeObserve();
            if (status.isComplete()) {
                LOGGER.fine("Ongoing is complete " + status);
                exchange.setResponseBlockStatus(null);
            } else {
                LOGGER.fine("Ongoing is continuing " + status);
            }
            exchange.setCurrentResponse(block);
            super.sendResponse(exchange, block);
        } else {
            this.earlyBlock2Negotiation(exchange, request);
            exchange.setRequest(request);
            super.receiveRequest(exchange, request);
        }
    }

    @Override
    public void sendResponse(Exchange exchange, Response response) {
        BlockOption block1 = exchange.getBlock1ToAck();
        if (block1 != null) {
            exchange.setBlock1ToAck(null);
        }
        if (this.requireBlockwise(exchange, response)) {
            LOGGER.fine("Response payload " + response.getPayloadSize() + "/" + this.maxMsgSize + " requires Blockwise");
            BlockwiseStatus status = this.findResponseBlockStatus(exchange, response);
            Response block = this.getNextResponseBlock(response, status);
            block.setType(response.getType());
            if (block1 != null) {
                block.getOptions().setBlock1(block1);
            }
            if (block.getToken() == null) {
                block.setToken(exchange.getRequest().getToken());
            }
            if (response.getOptions().hasObserve()) {
                exchange.setCurrentResponse(response);
            } else {
                exchange.setCurrentResponse(block);
            }
            super.sendResponse(exchange, block);
        } else {
            if (block1 != null) {
                response.getOptions().setBlock1(block1);
            }
            exchange.setCurrentResponse(response);
            super.sendResponse(exchange, response);
        }
    }

    @Override
    public void receiveResponse(Exchange exchange, Response response) {
        BlockwiseStatus status;
        if (!response.getOptions().hasBlock1() && !response.getOptions().hasBlock2()) {
            exchange.setResponse(response);
            super.receiveResponse(exchange, response);
            return;
        }
        if (response.getOptions().hasBlock1()) {
            BlockOption block1 = response.getOptions().getBlock1();
            LOGGER.finer("Response acknowledges block " + block1);
            status = exchange.getRequestBlockStatus();
            if (!status.isComplete()) {
                int currentSize = 1 << 4 + status.getCurrentSzx();
                int nextNum = status.getCurrentNum() + currentSize / block1.getSize();
                LOGGER.finer("Send next block num = " + nextNum);
                status.setCurrentNum(nextNum);
                status.setCurrentSzx(block1.getSzx());
                Request nextBlock = this.getNextRequestBlock(exchange.getRequest(), status);
                if (nextBlock.getToken() == null) {
                    nextBlock.setToken(response.getToken());
                }
                exchange.setCurrentRequest(nextBlock);
                super.sendRequest(exchange, nextBlock);
            } else if (!response.getOptions().hasBlock2()) {
                super.receiveResponse(exchange, response);
            } else {
                LOGGER.fine("Response has Block2 option and is therefore sent blockwise");
            }
        }
        if (response.getOptions().hasBlock2()) {
            BlockOption block2 = response.getOptions().getBlock2();
            status = this.findResponseBlockStatus(exchange, response);
            if (block2.getNum() == status.getCurrentNum()) {
                status.addBlock(response.getPayload());
                if (response.getOptions().hasObserve()) {
                    status.setObserve(response.getOptions().getObserve());
                }
                if (status.isRandomAccess()) {
                    exchange.setResponse(response);
                    super.receiveResponse(exchange, response);
                } else if (block2.isM()) {
                    LOGGER.finer("Request the next response block");
                    Request request = exchange.getRequest();
                    int num = block2.getNum() + 1;
                    int szx = block2.getSzx();
                    boolean m = false;
                    Request block = new Request(request.getCode());
                    block.setToken(response.getToken());
                    block.setOptions(new OptionSet(request.getOptions()));
                    block.setDestination(request.getDestination());
                    block.setDestinationPort(request.getDestinationPort());
                    block.setType(request.getType());
                    block.getOptions().setBlock2(szx, m, num);
                    status.setCurrentNum(num);
                    block.getOptions().removeObserve();
                    exchange.setCurrentRequest(block);
                    super.sendRequest(exchange, block);
                } else {
                    LOGGER.finer("We have received all " + status.getBlockCount() + " blocks of the response. Assemble and deliver");
                    Response assembled = new Response(response.getCode());
                    this.assembleMessage(status, assembled, response);
                    assembled.setType(response.getType());
                    int observe = status.getObserve();
                    if (observe != -1) {
                        assembled.getOptions().setObserve(observe);
                        exchange.setResponseBlockStatus(null);
                    }
                    LOGGER.fine("Assembled response: " + assembled);
                    exchange.setResponse(assembled);
                    super.receiveResponse(exchange, assembled);
                }
            } else {
                LOGGER.warning("Wrong block number. Expected " + status.getCurrentNum() + " but received " + block2.getNum() + ". Reject response; exchange has failed.");
                if (response.getType() == CoAP.Type.CON) {
                    EmptyMessage rst = EmptyMessage.newRST(response);
                    super.sendEmptyMessage(exchange, rst);
                }
                exchange.getRequest().cancel();
            }
        }
    }

    @Override
    public void sendEmptyMessage(Exchange exchange, EmptyMessage message) {
        super.sendEmptyMessage(exchange, message);
    }

    @Override
    public void receiveEmptyMessage(Exchange exchange, EmptyMessage message) {
        super.receiveEmptyMessage(exchange, message);
    }

    private void earlyBlock2Negotiation(Exchange exchange, Request request) {
        if (request.getOptions().hasBlock2()) {
            BlockOption block2 = request.getOptions().getBlock2();
            LOGGER.fine("Request demands blockwise transfer of response with option " + block2 + ". Create and set new block2 status");
            BlockwiseStatus status2 = new BlockwiseStatus(request.getOptions().getContentFormat(), block2.getNum(), block2.getSzx());
            exchange.setResponseBlockStatus(status2);
        }
    }

    private BlockwiseStatus findRequestBlockStatus(Exchange exchange, Request request) {
        BlockwiseStatus status = exchange.getRequestBlockStatus();
        if (status == null) {
            status = new BlockwiseStatus(request.getOptions().getContentFormat());
            status.setCurrentSzx(this.computeSZX(this.defaultBlockSize));
            exchange.setRequestBlockStatus(status);
            LOGGER.finer("There is no assembler status yet. Create and set new block1 status: " + status);
        }
        return status;
    }

    private BlockwiseStatus findResponseBlockStatus(Exchange exchange, Response response) {
        BlockwiseStatus status = exchange.getResponseBlockStatus();
        if (status == null) {
            status = new BlockwiseStatus(response.getOptions().getContentFormat());
            status.setCurrentSzx(this.computeSZX(this.defaultBlockSize));
            exchange.setResponseBlockStatus(status);
            LOGGER.finer("There is no blockwise status yet. Create and set new block2 status: " + status);
        } else {
            LOGGER.finer("Current blockwise status: " + status);
        }
        return status;
    }

    private Request getNextRequestBlock(Request request, BlockwiseStatus status) {
        int num = status.getCurrentNum();
        int szx = status.getCurrentSzx();
        Request block = new Request(request.getCode());
        block.setOptions(new OptionSet(request.getOptions()));
        block.setDestination(request.getDestination());
        block.setDestinationPort(request.getDestinationPort());
        block.setToken(request.getToken());
        block.setType(CoAP.Type.CON);
        int currentSize = 1 << 4 + szx;
        int from = num * currentSize;
        int to = Math.min((num + 1) * currentSize, request.getPayloadSize());
        int length = to - from;
        byte[] blockPayload = new byte[length];
        System.arraycopy(request.getPayload(), from, blockPayload, 0, length);
        block.setPayload(blockPayload);
        boolean m = to < request.getPayloadSize();
        block.getOptions().setBlock1(szx, m, num);
        status.setComplete(!m);
        return block;
    }

    private Response getNextResponseBlock(Response response, BlockwiseStatus status) {
        int szx = status.getCurrentSzx();
        int num = status.getCurrentNum();
        Response block = new Response(response.getCode());
        block.setDestination(response.getDestination());
        block.setDestinationPort(response.getDestinationPort());
        block.setToken(response.getToken());
        block.setOptions(new OptionSet(response.getOptions()));
        block.addMessageObserver(new TimeoutForwarder(response));
        int payloadsize = response.getPayloadSize();
        int currentSize = 1 << 4 + szx;
        int from = num * currentSize;
        if (0 < payloadsize && from < payloadsize) {
            int to = Math.min((num + 1) * currentSize, response.getPayloadSize());
            int length = to - from;
            byte[] blockPayload = new byte[length];
            System.arraycopy(response.getPayload(), from, blockPayload, 0, length);
            block.setPayload(blockPayload);
            boolean m = to < response.getPayloadSize();
            block.getOptions().setBlock2(szx, m, num);
            block.setLast(!m);
            status.setComplete(!m);
        } else {
            block.getOptions().setBlock2(szx, false, num);
            block.setLast(true);
            status.setComplete(true);
        }
        return block;
    }

    private void assembleMessage(BlockwiseStatus status, Message message, Message last) {
        message.setMID(last.getMID());
        message.setSource(last.getSource());
        message.setSourcePort(last.getSourcePort());
        message.setToken(last.getToken());
        message.setType(last.getType());
        message.setOptions(new OptionSet(last.getOptions()));
        int length = 0;
        for (byte[] block : status.getBlocks()) {
            length += block.length;
        }
        byte[] payload = new byte[length];
        int offset = 0;
        for (byte[] block : status.getBlocks()) {
            System.arraycopy(block, 0, payload, offset, block.length);
            offset += block.length;
        }
        message.setPayload(payload);
    }

    private boolean requiresBlockwise(Request request) {
        if (request.getCode() == CoAP.Code.PUT || request.getCode() == CoAP.Code.POST) {
            return request.getPayloadSize() > this.maxMsgSize;
        }
        return false;
    }

    private boolean requireBlockwise(Exchange exchange, Response response) {
        return response.getPayloadSize() > this.maxMsgSize || exchange.getResponseBlockStatus() != null;
    }

    private int computeSZX(int blockSize) {
        return (int)(Math.log(blockSize) / Math.log(2.0)) - 4;
    }

    public static class TimeoutForwarder
    extends MessageObserverAdapter {
        private Message message;

        public TimeoutForwarder(Message message) {
            this.message = message;
        }

        @Override
        public void onTimeout() {
            this.message.setTimedOut(true);
        }
    }
}

