/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.cloud.util;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.function.Consumer;
import javax.crypto.SecretKey;
import javax.security.auth.DestroyFailedException;
import javax.security.auth.Destroyable;
import org.eclipse.californium.elements.util.Bytes;
import org.eclipse.californium.elements.util.EncryptedStreamUtil;
import org.eclipse.californium.elements.util.PemReader;
import org.eclipse.californium.elements.util.SslContextUtil;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.elements.util.SystemResourceMonitors;
import org.eclipse.californium.scandium.dtls.x509.CertificateConfigurationHelper;
import org.eclipse.californium.scandium.util.SecretUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CredentialsStore
implements Destroyable {
    private static final Logger LOGGER = LoggerFactory.getLogger(CredentialsStore.class);
    private final EncryptedStreamUtil encryptionUtility = new EncryptedStreamUtil();
    private volatile Consumer<SslContextUtil.Credentials> observer;
    private volatile SystemResourceMonitors.SystemResourceMonitor monitor;
    private volatile String tag = "";
    private volatile SslContextUtil.Credentials currentCredentials = new SslContextUtil.Credentials(null);
    private volatile boolean destroyed;
    private byte[] seed;

    public String getReadCipher() {
        return this.encryptionUtility.getReadCipher();
    }

    public CredentialsStore setTag(String tag) {
        if (tag == null) {
            throw new NullPointerException("tag must not be null!");
        }
        this.tag = tag;
        return this;
    }

    public String getTag() {
        return this.tag;
    }

    public boolean isEncrypted() {
        return this.seed != null;
    }

    public CredentialsStore setObserver(Consumer<SslContextUtil.Credentials> observer) {
        this.observer = observer;
        return this;
    }

    public SystemResourceMonitors.SystemResourceMonitor createMonitor(final SecretKey password, String first, final String ... files) {
        this.monitor = first != null ? new SystemResourceMonitors.FileMonitor(first){
            private SecretKey monitorPassword;
            {
                super(arg0);
                this.monitorPassword = SecretUtil.create(password);
            }

            @Override
            protected void update(SystemResourceMonitors.FileMonitor.MonitoredValues values, SystemResourceMonitors.SystemResourceCheckReady ready) {
                for (int loop = 0; loop < 3; ++loop) {
                    try {
                        if (this.monitorPassword != null) {
                            CredentialsStore.this.load(this.monitorPassword, files);
                        } else {
                            CredentialsStore.this.load(files);
                        }
                        this.ready(values);
                        break;
                    }
                    catch (IllegalArgumentException ex) {
                        if (loop < 2) {
                            LOGGER.info("{}error: {}, retry in 1s ...", (Object)CredentialsStore.this.tag, (Object)ex.getMessage());
                        } else {
                            LOGGER.info("{}error: {}", (Object)CredentialsStore.this.tag, (Object)ex.getMessage());
                        }
                        try {
                            Thread.sleep(1000L);
                        }
                        catch (InterruptedException interruptedException) {
                            // empty catch block
                        }
                        continue;
                    }
                }
                ready.ready(false);
            }
        } : null;
        return this.monitor;
    }

    public SystemResourceMonitors.SystemResourceMonitor getMonitor() {
        return this.monitor;
    }

    public void clearSeed() {
        if (this.seed != null) {
            Bytes.clear(this.seed);
            this.seed = null;
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private SslContextUtil.Credentials loadCredentials(String file) {
        try (FileInputStream in = new FileInputStream(file);){
            SslContextUtil.Credentials credentials;
            try (InputStreamReader reader = new InputStreamReader((InputStream)in, StandardCharsets.UTF_8);){
                credentials = this.load(reader);
            }
            return credentials;
        }
        catch (IOException e) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("{}read credentials {}:", this.tag, file, e);
            } else {
                LOGGER.info("{}read credentials {}: {}", this.tag, file, e.getMessage());
            }
            return null;
        }
    }

    public String load(String ... files) {
        String firstFile = null;
        StringBuilder names = new StringBuilder();
        SslContextUtil.Credentials newCredentials = null;
        for (String file : files) {
            SslContextUtil.Credentials credentials = this.loadCredentials(file);
            if (credentials == null) continue;
            if (firstFile == null) {
                firstFile = file;
            }
            names.append(file).append(", ");
            newCredentials = CredentialsStore.merge(newCredentials, credentials);
            if (this.complete(newCredentials)) break;
        }
        if (this.changed(newCredentials)) {
            this.applyCredentials(newCredentials);
        } else if (newCredentials != null) {
            names.setLength(names.length() - 2);
            LOGGER.info("{}read credentials {}: not changed", (Object)this.tag, (Object)names);
        }
        return firstFile;
    }

    private SslContextUtil.Credentials loadCredentials(String file, SecretKey password) {
        SslContextUtil.Credentials credentials;
        FileInputStream in = new FileInputStream(file);
        try {
            credentials = this.load(in, password);
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((InputStream)in).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                LOGGER.warn("{}read encrypted credentials {}:", this.tag, file, e);
                return null;
            }
        }
        ((InputStream)in).close();
        return credentials;
    }

    public String load(SecretKey password, String ... files) {
        String firstFile = null;
        StringBuilder names = new StringBuilder();
        SslContextUtil.Credentials newCredentials = null;
        for (String file : files) {
            SslContextUtil.Credentials credentials = this.loadCredentials(file, password);
            if (credentials == null) continue;
            if (firstFile == null) {
                firstFile = file;
            }
            names.append(file).append(", ");
            newCredentials = CredentialsStore.merge(newCredentials, credentials);
            if (this.complete(newCredentials)) break;
        }
        if (this.changed(newCredentials)) {
            this.applyCredentials(newCredentials);
        } else if (newCredentials != null) {
            names.setLength(names.length() - 2);
            LOGGER.info("{}read encrypted credentials {}: not changed", (Object)this.tag, (Object)names);
        }
        return firstFile;
    }

    private void applyCredentials(SslContextUtil.Credentials newCredentials) {
        Consumer<SslContextUtil.Credentials> observer;
        if (!this.complete(newCredentials)) {
            LOGGER.info("Credentials are not complete!");
            this.clearSeed();
            return;
        }
        if (newCredentials.getPrivateKey() != null) {
            CertificateConfigurationHelper helper = new CertificateConfigurationHelper();
            helper.verifyKeyPair(newCredentials.getPrivateKey(), newCredentials.getPublicKey());
        }
        if ((observer = this.observer) != null) {
            observer.accept(newCredentials);
        }
        this.currentCredentials = newCredentials;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private SslContextUtil.Credentials load(InputStream in, SecretKey password) {
        byte[] seed = this.encryptionUtility.readSeed(in);
        if (this.seed != null && Arrays.equals(this.seed, seed)) {
            LOGGER.debug("{}encrypted credentials not changed, (same seed).", (Object)this.tag);
            return null;
        }
        try (InputStream inEncrypted = this.encryptionUtility.prepare(seed, in, password);
             InputStreamReader reader = new InputStreamReader(inEncrypted, StandardCharsets.UTF_8);){
            SslContextUtil.Credentials newCredentials = this.load(reader);
            if (newCredentials == null) return null;
            this.seed = seed;
            SslContextUtil.Credentials credentials = newCredentials;
            return credentials;
        }
        catch (IOException e) {
            LOGGER.warn("{}read encrypted credentials:", (Object)this.tag, (Object)e);
            return null;
        }
    }

    private SslContextUtil.Credentials load(Reader reader) throws IOException {
        block4: {
            try {
                SslContextUtil.Credentials newCredentials = SslContextUtil.loadPemCredentials(new PemReader(reader));
                if (newCredentials.isDestroyed()) {
                    LOGGER.info("Loaded credentials are empty!", (Object)newCredentials);
                    if (this.currentCredentials.isDestroyed()) {
                        this.destroyed = true;
                    }
                    break block4;
                }
                this.seed = null;
                LOGGER.info("Loaded {}", (Object)newCredentials);
                return newCredentials;
            }
            catch (GeneralSecurityException e) {
                LOGGER.warn("Loading credentials failed {}", (Object)e.getMessage());
                if (!this.currentCredentials.isDestroyed()) break block4;
                this.destroyed = true;
            }
        }
        return null;
    }

    protected boolean complete(SslContextUtil.Credentials newCredentials) {
        return !newCredentials.isDestroyed() && newCredentials.getPublicKey() != null == (newCredentials.getPrivateKey() != null);
    }

    protected boolean changed(SslContextUtil.Credentials newCredentials) {
        if (newCredentials == null) {
            return false;
        }
        if (CredentialsStore.changed(this.currentCredentials.getPrivateKey(), newCredentials.getPrivateKey())) {
            LOGGER.info("Private key changed");
            return true;
        }
        if (CredentialsStore.changed(this.currentCredentials.getPublicKey(), newCredentials.getPublicKey())) {
            LOGGER.info("Public key changed");
            return true;
        }
        if (CredentialsStore.changed(this.currentCredentials.getCertificateChain(), newCredentials.getCertificateChain())) {
            LOGGER.info("Certificate chain changed");
            return true;
        }
        if (CredentialsStore.changed(this.currentCredentials.getTrustedCertificates(), newCredentials.getTrustedCertificates())) {
            LOGGER.info("Trusted certificates changed");
            return true;
        }
        return false;
    }

    private static <T> boolean changed(T currentItem, T newItem) {
        if (newItem == null || newItem == currentItem) {
            return false;
        }
        if (currentItem == null) {
            return true;
        }
        if (currentItem.getClass().getComponentType() == null) {
            return !currentItem.equals(newItem);
        }
        if (newItem.getClass().getComponentType() != null) {
            return !Arrays.deepEquals((Object[])currentItem, (Object[])newItem);
        }
        return true;
    }

    private static SslContextUtil.Credentials merge(SslContextUtil.Credentials credentials1, SslContextUtil.Credentials credentials2) {
        if (credentials1 == null) {
            return credentials2;
        }
        if (credentials2 == null) {
            return credentials1;
        }
        PrivateKey privateKey = credentials1.getPrivateKey();
        PublicKey publicKey = credentials1.getPublicKey();
        Object[] chain = credentials1.getCertificateChain();
        Object[] trusts = credentials1.getTrustedCertificates();
        if (credentials2.getPrivateKey() != null) {
            if (privateKey != null && !privateKey.equals(credentials2.getPrivateKey())) {
                throw new IllegalArgumentException("Ambiguous private key!");
            }
            privateKey = credentials2.getPrivateKey();
        }
        if (credentials2.getPublicKey() != null) {
            if (publicKey != null && !publicKey.equals(credentials2.getPublicKey())) {
                throw new IllegalArgumentException("Ambiguous public key!");
            }
            publicKey = credentials2.getPublicKey();
        }
        if (credentials2.getCertificateChain() != null) {
            if (chain != null && !Arrays.deepEquals(chain, credentials2.getCertificateChain())) {
                throw new IllegalArgumentException("Ambiguous chain!");
            }
            chain = credentials2.getCertificateChain();
        }
        if (credentials2.getTrustedCertificates() != null) {
            if (trusts != null && !Arrays.deepEquals(trusts, credentials2.getTrustedCertificates())) {
                throw new IllegalArgumentException("Ambiguous trusts!");
            }
            trusts = credentials2.getTrustedCertificates();
        }
        if (privateKey == null && publicKey == null) {
            return new SslContextUtil.Credentials((Certificate[])trusts);
        }
        return new SslContextUtil.Credentials(privateKey, publicKey, (X509Certificate[])chain);
    }

    public SslContextUtil.Credentials loadAndCreateMonitor(String password64, boolean createMonitor, String ... files) {
        if (password64 != null) {
            byte[] secret = StringUtil.base64ToByteArray(password64);
            SecretKey key = SecretUtil.create(secret, "PW");
            Bytes.clear(secret);
            String first = this.load(key, files);
            if (createMonitor) {
                this.createMonitor(key, first, files);
            }
            SecretUtil.destroy(key);
            LOGGER.info("{}loaded encrypted stores {}", (Object)this.getTag(), (Object)first);
        } else {
            String first = this.load(files);
            if (createMonitor) {
                this.createMonitor(null, first, files);
            }
            LOGGER.info("{}loaded stores {}", (Object)this.getTag(), (Object)first);
        }
        return this.currentCredentials;
    }

    public SslContextUtil.Credentials getCredentials() {
        return this.currentCredentials;
    }

    @Override
    public void destroy() throws DestroyFailedException {
        this.currentCredentials.destroy();
        this.destroyed = true;
    }

    @Override
    public boolean isDestroyed() {
        return this.destroyed;
    }
}

