/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.cloud.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.Principal;
import java.security.PublicKey;
import java.security.spec.ECGenParameterSpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.security.auth.DestroyFailedException;
import org.eclipse.californium.cloud.util.ResourceParser;
import org.eclipse.californium.elements.auth.PreSharedKeyIdentity;
import org.eclipse.californium.elements.auth.RawPublicKeyIdentity;
import org.eclipse.californium.elements.auth.X509CertPath;
import org.eclipse.californium.elements.util.Asn1DerDecoder;
import org.eclipse.californium.elements.util.Bytes;
import org.eclipse.californium.elements.util.JceProviderUtil;
import org.eclipse.californium.elements.util.StandardCharsets;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.dtls.cipher.RandomManager;
import org.eclipse.californium.scandium.util.SecretUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DeviceParser
implements ResourceParser<DeviceParser> {
    private static final Logger LOGGER = LoggerFactory.getLogger(DeviceParser.class);
    private static final byte[] ECC_SECP256R1_HEADER;
    public static final String GROUP_POSTFIX = ".group";
    public static final String PSK_POSTFIX = ".psk";
    public static final String RPK_POSTFIX = ".rpk";
    private final ConcurrentMap<String, Device> map = new ConcurrentHashMap<String, Device>();
    private final ConcurrentMap<String, Device> psk = new ConcurrentHashMap<String, Device>();
    private final ConcurrentMap<PublicKey, Device> rpk = new ConcurrentHashMap<PublicKey, Device>();
    private final ConcurrentMap<String, Set<String>> groups = new ConcurrentHashMap<String, Set<String>>();
    private final boolean caseSensitiveNames;
    private volatile boolean destroyed;

    public DeviceParser(boolean caseSensitiveNames) {
        this.caseSensitiveNames = caseSensitiveNames;
    }

    private String getKey(String name) {
        String key;
        String string = key = name == null ? "" : name;
        if (!this.caseSensitiveNames && key != null) {
            key = key.toLowerCase();
        }
        return key;
    }

    private boolean match(String name1, String name2) {
        if (this.caseSensitiveNames) {
            return name1.equals(name2);
        }
        return name1.equalsIgnoreCase(name2);
    }

    private String prefix(String id, String postfix) {
        return StringUtil.truncateTail(this.caseSensitiveNames, id, postfix);
    }

    private boolean isName(String id) {
        String name = this.prefix(id, GROUP_POSTFIX);
        if (name != id) {
            return true;
        }
        name = this.prefix(id, PSK_POSTFIX);
        if (name != id) {
            return false;
        }
        name = this.prefix(id, RPK_POSTFIX);
        return name == id;
    }

    private boolean match(Device.Builder builder, String name) {
        if (builder.name == null) {
            return false;
        }
        if (name.isEmpty()) {
            return !builder.name.isEmpty();
        }
        return this.match(builder.name, name);
    }

    public boolean add(Device.Builder builder) {
        return this.add(builder.build());
    }

    public boolean add(Device device) {
        String key = this.getKey(device.name);
        if (this.map.putIfAbsent(key, device) == null) {
            Device previous = null;
            if (device.pskIdentity != null && (previous = this.psk.putIfAbsent(device.pskIdentity, device)) != null) {
                LOGGER.info("psk {} {} ambiguous {}", device.pskIdentity, device.name, previous.name);
                this.map.remove(key, device);
                return false;
            }
            if (device.publicKey != null && (previous = this.rpk.putIfAbsent(device.publicKey, device)) != null) {
                LOGGER.info("rpk {} ambiguous {}", (Object)device.name, (Object)previous.name);
                this.remove(device);
                return false;
            }
            LOGGER.info("added {} {} {}", device.name, device.pskIdentity != null ? "psk" : "", device.publicKey != null ? "rpk" : "");
            Set<String> group = new HashSet<String>();
            Set prev = this.groups.putIfAbsent(device.group, group);
            if (prev != null) {
                group = prev;
            }
            group.add(device.name);
            return true;
        }
        return false;
    }

    public Set<String> getGroup(String group) {
        Set<String> devices = (Set<String>)this.groups.get(group);
        if (devices == null) {
            devices = Collections.emptySet();
        }
        return devices;
    }

    public Device get(String name) {
        return (Device)this.map.get(this.getKey(name));
    }

    public Device getByPreSharedKeyIdentity(String identity) {
        return (Device)this.psk.get(identity);
    }

    public Device getByRawPublicKey(PublicKey publicKey) {
        return (Device)this.rpk.get(publicKey);
    }

    public Device getByPrincipal(Principal principal) {
        if (principal instanceof PreSharedKeyIdentity) {
            PreSharedKeyIdentity pskIdentity = (PreSharedKeyIdentity)principal;
            return this.getByPreSharedKeyIdentity(pskIdentity.getIdentity());
        }
        if (principal instanceof RawPublicKeyIdentity) {
            RawPublicKeyIdentity rpkIdentity = (RawPublicKeyIdentity)principal;
            return this.getByRawPublicKey(rpkIdentity.getKey());
        }
        if (principal instanceof X509CertPath) {
            // empty if block
        }
        return null;
    }

    public boolean remove(String name) {
        return this.remove(this.get(name));
    }

    public boolean remove(Device device) {
        boolean removed = this.map.remove(this.getKey(device.name), device);
        if (removed) {
            Set group;
            if (device.pskIdentity != null) {
                this.psk.remove(device.pskIdentity, device);
            }
            if (device.publicKey != null) {
                this.rpk.remove(device.publicKey, device);
            }
            if ((group = (Set)this.groups.get(device.group)) != null) {
                group.remove(device.name);
            }
            return true;
        }
        return removed;
    }

    public int size() {
        return this.map.size();
    }

    @Override
    public void save(Writer writer) throws IOException {
        ArrayList names = new ArrayList(this.map.keySet());
        Collections.sort(names);
        for (String name : names) {
            Device credentials = (Device)this.map.get(name);
            if (credentials == null) continue;
            writer.write(credentials.name);
            writer.write(61);
            writer.write(credentials.group);
            writer.write(StringUtil.lineSeparator());
            if (credentials.pskIdentity != null && credentials.pskSecret != null) {
                writer.write(credentials.name + PSK_POSTFIX);
                writer.write(61);
                writer.write(credentials.pskIdentity);
                writer.write(44);
                writer.write(DeviceParser.encode64(credentials.pskSecret));
                writer.write(StringUtil.lineSeparator());
            }
            if (credentials.publicKey == null) continue;
            writer.write(credentials.name + RPK_POSTFIX);
            writer.write(61);
            writer.write(DeviceParser.encode64(credentials.publicKey.getEncoded()));
            writer.write(StringUtil.lineSeparator());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void load(Reader reader) throws IOException {
        block27: {
            BufferedReader lineReader = new BufferedReader(reader);
            try {
                String line;
                int lineNumber = 0;
                int errors = 0;
                int comments = 0;
                Device.Builder builder = Device.builder();
                while ((line = lineReader.readLine()) != null) {
                    ++lineNumber;
                    try {
                        if (!line.isEmpty() && !line.startsWith("#")) {
                            String[] entry = line.split("=", 2);
                            if (entry.length == 2) {
                                String name = entry[0];
                                String[] values = entry[1].split(",");
                                String prefix = this.prefix(name, RPK_POSTFIX);
                                if (prefix != name) {
                                    if (this.parseRPK(builder, prefix, values)) continue;
                                    ++errors;
                                    LOGGER.warn("{}: '{}' invalid line!", (Object)lineNumber, (Object)line);
                                    continue;
                                }
                                prefix = this.prefix(name, PSK_POSTFIX);
                                if (prefix != name) {
                                    if (this.parsePSK(builder, prefix, values)) continue;
                                    ++errors;
                                    LOGGER.warn("{}: '{}' invalid line!", (Object)lineNumber, (Object)line);
                                    continue;
                                }
                                prefix = this.prefix(name, GROUP_POSTFIX);
                                if (prefix == name && !this.isName(name)) continue;
                                if (builder.name != null) {
                                    this.add(builder);
                                    builder = Device.builder();
                                }
                                if (values.length != 1) {
                                    ++errors;
                                    LOGGER.warn("{}: '{}' invalid line!", (Object)lineNumber, (Object)line);
                                    continue;
                                }
                                builder.name = prefix == null ? name : prefix;
                                builder.group = DeviceParser.decodeText(values[0]);
                                continue;
                            }
                            ++errors;
                            LOGGER.warn("{}: '{}' invalid line!", (Object)lineNumber, (Object)line);
                            continue;
                        }
                        ++comments;
                    }
                    catch (IllegalArgumentException ex) {
                        ++errors;
                        LOGGER.warn("{}: '{}' invalid line!", lineNumber, line, ex);
                    }
                }
                if (builder.name != null) {
                    this.add(builder);
                }
                if (this.size() == 0 && errors > 0 && lineNumber == comments + errors) {
                    LOGGER.warn("read store, only errors, wrong password?");
                    SecretUtil.destroy(this);
                }
            }
            catch (RuntimeException e) {
                LOGGER.warn("read store, unexpected error occurred!", e);
            }
            catch (IOException e) {
                if (e.getCause() instanceof GeneralSecurityException) {
                    LOGGER.warn("read store, wrong password?", e);
                    SecretUtil.destroy(this);
                    break block27;
                }
                throw e;
            }
            finally {
                try {
                    lineReader.close();
                }
                catch (IOException e) {}
            }
        }
        LOGGER.info("read {} device credentials.", (Object)this.size());
    }

    private boolean parsePSK(Device.Builder builder, String name, String[] values) {
        if (values.length != 2 || !this.match(builder, name)) {
            return false;
        }
        builder.pskIdentity = DeviceParser.decodeText(values[0]);
        builder.pskSecret = DeviceParser.binDecodeTextOr64(values[1]);
        return !builder.pskIdentity.isEmpty() && builder.pskSecret.length > 0;
    }

    private boolean parseRPK(Device.Builder builder, String name, String[] values) {
        if (values.length != 1 || !this.match(builder, name)) {
            return false;
        }
        byte[] publicKey = DeviceParser.binDecodeTextOr64(values[0]);
        Exception error = null;
        try {
            builder.publicKey = Asn1DerDecoder.readSubjectPublicKey(publicKey);
            return true;
        }
        catch (GeneralSecurityException e) {
            error = e;
        }
        catch (IllegalArgumentException e) {
            error = e;
        }
        if (error != null) {
            if (publicKey.length == 64) {
                publicKey = Bytes.concatenate(ECC_SECP256R1_HEADER, publicKey);
                try {
                    builder.publicKey = Asn1DerDecoder.readSubjectPublicKey(publicKey);
                    return true;
                }
                catch (GeneralSecurityException generalSecurityException) {
                }
                catch (IllegalArgumentException illegalArgumentException) {
                    // empty catch block
                }
            }
            LOGGER.warn("RPK:", error);
        }
        return false;
    }

    @Override
    public void destroy() throws DestroyFailedException {
        this.map.clear();
        this.destroyed = true;
    }

    @Override
    public boolean isDestroyed() {
        return this.destroyed;
    }

    private static String encode64(byte[] value) {
        return StringUtil.byteArrayToBase64(value);
    }

    private static byte[] binDecodeTextOr64(String value) {
        if (value.isEmpty()) {
            return Bytes.EMPTY;
        }
        char c = value.charAt(0);
        if (c == '\'' || c == '\"') {
            int end;
            char e;
            if (value.length() > 2 && (e = value.charAt(end = value.length() - 1)) == c) {
                value = value.substring(1, end);
                return value.getBytes(StandardCharsets.UTF_8);
            }
        } else if (c == ':' && value.startsWith(":0x")) {
            return StringUtil.hex2ByteArray(value.substring(3));
        }
        return StringUtil.base64ToByteArray(value);
    }

    private static String decodeText(String value) {
        if (!value.isEmpty()) {
            int end;
            char e;
            char c = value.charAt(0);
            if (value.length() > 2 && (c == '\'' || c == '\"') && (e = value.charAt(end = value.length() - 1)) == c) {
                value = value.substring(1, end);
            }
        }
        return value;
    }

    @Override
    public DeviceParser create() {
        return new DeviceParser(this.caseSensitiveNames);
    }

    static {
        byte[] header = null;
        try {
            String oid = JceProviderUtil.getEdDsaStandardAlgorithmName("EC", "EC");
            KeyPairGenerator generator = KeyPairGenerator.getInstance(oid);
            generator.initialize(new ECGenParameterSpec("secp256r1"), RandomManager.currentSecureRandom());
            KeyPair keyPair = generator.generateKeyPair();
            byte[] encoded = keyPair.getPublic().getEncoded();
            header = Arrays.copyOf(encoded, encoded.length - 64);
        }
        catch (GeneralSecurityException ex) {
            header = null;
            LOGGER.error("EC failed!", ex);
        }
        ECC_SECP256R1_HEADER = header;
    }

    public static class Device {
        public final String name;
        public final String group;
        public final String pskIdentity;
        public final byte[] pskSecret;
        public final PublicKey publicKey;

        public Device(String name, String group, String pskIdentity, byte[] pskSecret, PublicKey publicKey) {
            if (name == null) {
                throw new NullPointerException("name must not be null!");
            }
            if (group == null) {
                throw new NullPointerException("group must not be null!");
            }
            if (pskIdentity == null && publicKey == null) {
                throw new NullPointerException("Either pskIdentity or publicKey must not be null!");
            }
            if (pskIdentity != null && pskSecret == null) {
                throw new NullPointerException("pskSecret must not be null, if pskIdentity is provided!");
            }
            this.name = name;
            this.pskIdentity = pskIdentity;
            this.pskSecret = pskSecret;
            this.publicKey = publicKey;
            this.group = group;
        }

        public int hashCode() {
            return this.name.hashCode();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            Device other = (Device)obj;
            return this.name.equals(other.name);
        }

        public static Builder builder() {
            return new Builder();
        }

        public static class Builder {
            public String name;
            public String group;
            public String pskIdentity;
            public byte[] pskSecret;
            public PublicKey publicKey;

            private Builder() {
            }

            public Device build() {
                return new Device(this.name, this.group, this.pskIdentity, this.pskSecret, this.publicKey);
            }
        }
    }
}

