/*
 * Copyright (C) 2016 Marten Gajda <marten@dmfs.org>
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.dmfs.iterators;

import java.util.Iterator;
import java.util.NoSuchElementException;


/**
 * An {@link Iterator} that iterates the elements of a string of a comma (or other character) separated value list .
 * <p />
 * Example:
 * 
 * <pre>
 * <code>
 * 	Iterator<String> i = new CsvIterator<String>("a, b,def,123", ',');
 * 	i.next(); // returns &quot;a&quot;
 * 	i.next(); // returns &quot; b&quot;
 * 	i.next(); // returns &quot;def&quot;
 * 	i.next(); // returns &quot;123&quot;
 * 	i.hasNext(); // false
 * </code>
 * </pre>
 * 
 * Separators between quote characters (") will be ignored.
 * <p />
 * Example:
 * 
 * <pre>
 * <code>
 * 	Iterator<String> i = new CsvIterator<String>("\"a, b\",\"def,123\"", ',');
 * 	i.next(); // returns &quot;a , b&quot;
 * 	i.next(); // returns &quot;def,123&quot;
 * 	i.hasNext(); // false
 * </code>
 * </pre>
 * 
 * Iterating an empty String or a string without (unquoted) separators will return exactly one element.
 * <p />
 * Example:
 * 
 * <pre>
 * <code>
 * 	Iterator<String> i = new CsvIterator<String>("", ',');
 * 	i.next(); // returns &quot;&quot;
 * 	i.hasNext(); // false
 * 
 * 	Iterator<String> i2 = new CsvIterator<String>("\"abc,def\"", ',');
 * 	i2.next(); // returns &quot;abc,def&quot;
 * 	i2.hasNext(); // false
 * </code>
 * </pre>
 * 
 * 
 * @author Marten Gajda <marten@dmfs.org>
 */
public final class CsvIterator implements Iterator<String>
{
	private final String mValue;
	private final char mSeparator;

	private int mLastSeparatorPos = -1;
	private int mNextSeparatorPos = -1;


	/**
	 * Creates an {@link Iterator} that iterates all elements of the given string which are separated by the given <code>separator</code>.
	 * 
	 * @param value
	 *            The string that contains a list of values.
	 * @param separator
	 *            The separator that separates the values.
	 */
	public CsvIterator(String value, char separator)
	{
		mValue = value;
		mSeparator = separator;
	}


	@Override
	public boolean hasNext()
	{
		if (mNextSeparatorPos == -1)
		{
			findNextSeparator();
		}
		return mLastSeparatorPos < mValue.length();
	}


	@Override
	public String next()
	{
		if (mLastSeparatorPos >= mValue.length())
		{
			throw new NoSuchElementException("Last element has already been iterated.");
		}
		if (mNextSeparatorPos == -1)
		{
			findNextSeparator();
		}
		String result = mValue.substring(mLastSeparatorPos + 1, mNextSeparatorPos);
		findNextSeparator();
		return result;
	}


	@Override
	public void remove()
	{
		throw new UnsupportedOperationException("Remove not supported by this iterator");
	}


	/**
	 * Move {@link #mNextSeparatorPos} to the next (unquoted) separator (or the end of the {@link String} if no other separator exists in {@link #mValue}).
	 */
	private void findNextSeparator()
	{
		boolean isQuoted = false;
		mLastSeparatorPos = mNextSeparatorPos;
		while (++mNextSeparatorPos < mValue.length())
		{
			char c = mValue.charAt(mNextSeparatorPos);
			if (c == mSeparator)
			{
				if (!isQuoted)
				{
					return;
				}
				// else: ignore quoted speparator
			}
			else if (c == '"')
			{
				isQuoted = !isQuoted;
			}
		}
	}

}
