/*******************************************************************************
 * Copyright (c) 2015-2018 Skymind, Inc.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Apache License, Version 2.0 which is available at
 * https://www.apache.org/licenses/LICENSE-2.0.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 ******************************************************************************/

package org.deeplearning4j.ui;

import lombok.NonNull;
import org.nd4j.linalg.primitives.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.concurrent.LinkedBlockingQueue;

/**
 * This is simple wrapper for sending state updates generated by `TrainingListener`s.
 * Basic idea is simple: network processing should be handled in background, without slowing caller thread
 *
 * @author raver119@gmail.com
 */
public class WebReporter {
    private static final WebReporter INSTANCE = new WebReporter();
    protected LinkedBlockingQueue<Pair<WebTarget, Entity>> queue = new LinkedBlockingQueue<>();

    private static final Logger log = LoggerFactory.getLogger(WebReporter.class);

    private WebReporter() {
        ReporterThread thread = new ReporterThread(queue);
        thread.start();
        throw new RuntimeException("Not implemented");
    }

    public static WebReporter getInstance() {
        return INSTANCE;
    }

    /**
     * This method queues UI report for sending
     *
     * @param target
     * @param entity
     */
    public void queueReport(WebTarget target, Entity entity) {
        queue.add(Pair.makePair(target, entity));
    }

    /**
     * This method immediately sends UI report to specified target using POST request
     *
     * @param target
     * @param entity
     */
    public void postReport(WebTarget target, Entity entity) {
        Response resp = target.request(MediaType.APPLICATION_JSON).accept(MediaType.APPLICATION_JSON).post(entity);
        log.debug("{}", resp);
    }

    private class ReporterThread extends Thread implements Runnable {
        private LinkedBlockingQueue<Pair<WebTarget, Entity>> queue;

        public ReporterThread(@NonNull LinkedBlockingQueue<Pair<WebTarget, Entity>> queue) {
            this.queue = queue;
            this.setName("DL4j Ui WebReporter thread");
            this.setDaemon(true);
        }

        @Override
        public void run() {
            while (true) {
                try {
                    Pair<WebTarget, Entity> pair = queue.take();
                    postReport(pair.getFirst(), pair.getSecond());
                } catch (Exception e) {
                    log.error("Exception caught but ignored: " + e.getMessage());
                    e.printStackTrace();
                } finally {
                    try {
                        Thread.sleep(100);
                    } catch (Exception e) {
                    }
                }
            }
        }
    }
}
