/*
 *  ******************************************************************************
 *  *
 *  *
 *  * This program and the accompanying materials are made available under the
 *  * terms of the Apache License, Version 2.0 which is available at
 *  * https://www.apache.org/licenses/LICENSE-2.0.
 *  *
 *  *  See the NOTICE file distributed with this work for additional
 *  *  information regarding copyright ownership.
 *  * Unless required by applicable law or agreed to in writing, software
 *  * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 *  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 *  * License for the specific language governing permissions and limitations
 *  * under the License.
 *  *
 *  * SPDX-License-Identifier: Apache-2.0
 *  *****************************************************************************
 */

package org.deeplearning4j.nn.conf.layers.convolutional;

import lombok.*;
import org.deeplearning4j.nn.conf.InputPreProcessor;
import org.deeplearning4j.nn.conf.NeuralNetConfiguration;
import org.deeplearning4j.nn.conf.inputs.InputType;
import org.deeplearning4j.nn.conf.layers.InputTypeUtil;
import org.deeplearning4j.nn.conf.layers.Layer;
import org.deeplearning4j.nn.conf.layers.NoParamLayer;
import org.deeplearning4j.nn.conf.memory.LayerMemoryReport;
import org.deeplearning4j.nn.layers.convolution.Cropping3DLayer;
import org.deeplearning4j.optimize.api.TrainingListener;
import org.deeplearning4j.util.ValidationUtils;
import org.nd4j.common.base.Preconditions;
import org.nd4j.linalg.api.buffer.DataType;
import org.nd4j.linalg.api.ndarray.INDArray;

import java.util.Collection;
import java.util.Map;

@Data
@NoArgsConstructor
@EqualsAndHashCode(callSuper = true)
public class Cropping3D extends NoParamLayer {

    private int[] cropping;

    /**
     * @param cropDepth Amount of cropping to apply to both depth boundaries of the input activations
     * @param cropHeight Amount of cropping to apply to both height boundaries of the input activations
     * @param cropWidth Amount of cropping to apply to both width boundaries of the input activations
     */
    public Cropping3D(int cropDepth, int cropHeight, int cropWidth) {
        this(cropDepth, cropDepth, cropHeight, cropHeight, cropWidth, cropWidth);
    }

    /**
     * @param cropLeftD Amount of cropping to apply to the left of the depth dimension
     * @param cropRightD Amount of cropping to apply to the right of the depth dimension
     * @param cropLeftH Amount of cropping to apply to the left of the height dimension
     * @param cropRightH Amount of cropping to apply to the right of the height dimension
     * @param cropLeftW Amount of cropping to apply to the left of the width dimension
     * @param cropRightW Amount of cropping to apply to the right of the width dimension
     */
    public Cropping3D(int cropLeftD, int cropRightD, int cropLeftH, int cropRightH, int cropLeftW, int cropRightW) {
        this(new Builder(cropLeftD, cropRightD, cropLeftH, cropRightH, cropLeftW, cropRightW));
    }

    /**
     * @param cropping Cropping as either a length 3 array, with values {@code [cropDepth, cropHeight, cropWidth]}, or
     * as a length 4 array, with values {@code [cropLeftDepth, cropRightDepth, cropLeftHeight, cropRightHeight,
     * cropLeftWidth, cropRightWidth]}
     */
    public Cropping3D(int[] cropping) {
        this(new Builder(cropping));
    }

    protected Cropping3D(Builder builder) {
        super(builder);
        this.cropping = builder.cropping;
    }

    @Override
    public org.deeplearning4j.nn.api.Layer instantiate(NeuralNetConfiguration conf,
                                                       Collection<TrainingListener> iterationListeners, int layerIndex, INDArray layerParamsView,
                                                       boolean initializeParams, DataType networkDataType) {
        Cropping3DLayer ret = new Cropping3DLayer(conf, networkDataType);
        ret.setListeners(iterationListeners);
        ret.setIndex(layerIndex);
        Map<String, INDArray> paramTable = initializer().init(conf, layerParamsView, initializeParams);
        ret.setParamTable(paramTable);
        ret.setConf(conf);
        return ret;
    }

    @Override
    public InputType getOutputType(int layerIndex, InputType inputType) {
        if (inputType == null || inputType.getType() != InputType.Type.CNN3D) {
            throw new IllegalStateException("Invalid input for 3D cropping layer (layer index = " + layerIndex
                            + ", layer name = \"" + getLayerName() + "\"): expect CNN3D input type with size > 0. Got: "
                            + inputType);
        }
        InputType.InputTypeConvolutional3D c = (InputType.InputTypeConvolutional3D) inputType;
        return InputType.convolutional3D(c.getDepth() - cropping[0] - cropping[1],
                        c.getHeight() - cropping[2] - cropping[3], c.getWidth() - cropping[4] - cropping[5],
                        c.getChannels());
    }

    @Override
    public InputPreProcessor getPreProcessorForInputType(InputType inputType) {
        Preconditions.checkArgument(inputType != null, "Invalid input for Cropping3D " + "layer (layer name=\""
                        + getLayerName() + "\"): InputType is null");
        return InputTypeUtil.getPreProcessorForInputTypeCnn3DLayers(inputType, getLayerName());
    }

    @Override
    public LayerMemoryReport getMemoryReport(InputType inputType) {
        return null;
    }


    @Getter
    @Setter
    public static class Builder extends Layer.Builder<Builder> {

        /**
         * Cropping amount, a length 6 array, i.e. crop left depth, crop right depth, crop left height, crop right height, crop left width, crop right width
         */
        @Setter(AccessLevel.NONE)
        private int[] cropping = new int[] {0, 0, 0, 0, 0, 0};

        /**
         * @param cropping Cropping amount, must be length 1, 3, or 6 array, i.e. either all values, crop depth, crop height, crop width
         * or crop left depth, crop right depth, crop left height, crop right height, crop left width, crop right width
         */
        public void setCropping(int... cropping) {
            this.cropping = ValidationUtils.validate6NonNegative(cropping, "cropping");
        }

        public Builder() {

        }

        /**
         * @param cropping Cropping amount, must be length 3 or 6 array, i.e. either crop depth, crop height, crop width
         * or crop left depth, crop right depth, crop left height, crop right height, crop left width, crop right width
         */
        public Builder(@NonNull int[] cropping) {
            this.setCropping(cropping);
        }

        /**
         * @param cropDepth Amount of cropping to apply to both depth boundaries of the input activations
         * @param cropHeight Amount of cropping to apply to both height boundaries of the input activations
         * @param cropWidth Amount of cropping to apply to both width boundaries of the input activations
         */
        public Builder(int cropDepth, int cropHeight, int cropWidth) {
            this(cropDepth, cropDepth, cropHeight, cropHeight, cropWidth, cropWidth);
        }

        /**
         * @param cropLeftD Amount of cropping to apply to the left of the depth dimension
         * @param cropRightD Amount of cropping to apply to the right of the depth dimension
         * @param cropLeftH Amount of cropping to apply to the left of the height dimension
         * @param cropRightH Amount of cropping to apply to the right of the height dimension
         * @param cropLeftW Amount of cropping to apply to the left of the width dimension
         * @param cropRightW Amount of cropping to apply to the right of the width dimension
         */
        public Builder(int cropLeftD, int cropRightD, int cropLeftH, int cropRightH, int cropLeftW, int cropRightW) {
            this.setCropping(new int[] {cropLeftD, cropRightD, cropLeftH, cropRightH, cropLeftW, cropRightW});
        }

        public Cropping3D build() {
            return new Cropping3D(this);
        }
    }
}
