/**********************************************************************
Copyright (c) 2002 Kelly Grizzle (TJDO) and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2003 Andy Jefferson - coding standards
2005 Andy Jefferson - added support for single quoted StringLiteral 
                      (including contrib from Tony Lai)
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.query.legacy;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.CharacterIterator;
import java.text.StringCharacterIterator;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.exceptions.ClassNotResolvedException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.util.Imports;
import org.datanucleus.util.Localiser;

/**
 * Parser for a Query. The query can be JDOQL, or JPQL.
 * Allows a class to work its way through the parsed string, obtaining relevant
 * components with each call, or peeking ahead before deciding what component
 * to parse next.
 **/
public class Parser
{
    /** Localiser for messages. */
    protected static final Localiser LOCALISER=Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    protected final String input;
    protected final Imports imports;
    protected final CharacterIterator ci;

    /**
     * Constructor
     * @param input The input string
     * @param imports Necessary imports to use in the parse.
     **/
    public Parser(String input, Imports imports)
    {
        this.input = input;
        this.imports = imports;

        ci = new StringCharacterIterator(input);
    }

    /**
     * Accessor for the input string.
     * @return The input string.
     */
    public String getInput()
    {
        return input;
    }

    /**
     * Accessor for the current index in the input string.
     * @return The current index.
     */
    public int getIndex()
    {
        return ci.getIndex();
    }

    /**
     * Skip over any whitespace from the current position.
     * @return The new position
     */
    public int skipWS()
    {
        int startIdx = ci.getIndex();
        char c = ci.current();

        while (Character.isWhitespace(c) || 
               c == '\t' || c == '\f' ||
               c == '\n' || c == '\r' ||
               c == '\u0009' || c == '\u000c' ||
               c == '\u0020' || c == '\11' ||
               c == '\12' || c == '\14' ||
               c == '\15' || c == '\40')
        {
            c = ci.next();
        }

        return startIdx;
    }

    /**
     * Check if END OF TEXT is reach
     * @return true if END OF TEXT is reach
     */
    public boolean parseEOS()
    {
        skipWS();

        return ci.current() == CharacterIterator.DONE;
    }

    /**
     * Check if char <code>c</code> is found
     * @param c the Character to find
     * @return true if <code>c</code> is found
     */
    public boolean parseChar(char c)
    {
        skipWS();

        if (ci.current() == c)
        {
            ci.next();
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     * Check if char <code>c</code> is found
     * @param c the Character to find
     * @param unlessFollowedBy the character to validate it does not follow <code>c</code>
     * @return true if <code>c</code> is found and not followed by <code>unlessFollowedBy</code>
     */
    public boolean parseChar(char c, char unlessFollowedBy)
    {
        int savedIdx = skipWS();

        if (ci.current() == c && ci.next() != unlessFollowedBy)
        {
            return true;
        }
        else
        {
            ci.setIndex(savedIdx);
            return false;
        }
    }

    /**
     * Check if String <code>s</code> is found
     * @param s the String to find
     * @return true if <code>s</code> is found
     */
    public boolean parseString(String s)
    {
        int savedIdx = skipWS();

        int len = s.length();
        char c = ci.current(); 

        for (int i = 0; i < len; ++i)
        {
            if (c != s.charAt(i))
            {
                ci.setIndex(savedIdx);
                return false;
            }

            c = ci.next();
        }

        return true;
    }

    /**
     * Check if String <code>s</code> is found ignoring the case
     * @param s the String to find
     * @return true if <code>s</code> is found
     */
    public boolean parseStringIgnoreCase(String s)
    {
        String lowerCasedString = s.toLowerCase();
        
        int savedIdx = skipWS();

        int len = lowerCasedString.length();
        char c = ci.current(); 

        for (int i = 0; i < len; ++i)
        {
            if (Character.toLowerCase(c) != lowerCasedString.charAt(i))
            {
                ci.setIndex(savedIdx);
                return false;
            }

            c = ci.next();
        }

        return true;
    }

    /**
     * Check if String "s" is found ignoring the case, and not moving the cursor position.
     * @param s the String to find
     * @return true if string is found
     */
    public boolean peekStringIgnoreCase(String s)
    {
        String lowerCasedString = s.toLowerCase();
        
        int savedIdx = skipWS();

        int len = lowerCasedString.length();
        char c = ci.current(); 

        for (int i = 0; i < len; ++i)
        {
            if (Character.toLowerCase(c) != lowerCasedString.charAt(i))
            {
                ci.setIndex(savedIdx);
                return false;
            }
            c = ci.next();
        }
        ci.setIndex(savedIdx);

        return true;
    }

    /**
     * Parse a java identifier from the current position.
     * @return The identifier
     */
    public String parseIdentifier()
    {
        skipWS();
        char c = ci.current();

        if (!Character.isJavaIdentifierStart(c) && !(c == ':'))
        {
            // Current character is not a valid identifier char, and isn't a
            // valid JDOQL parameter start character
            return null;
        }

        StringBuffer id = new StringBuffer();
        id.append(c);
        while (Character.isJavaIdentifierPart(c = ci.next()))
        {
            id.append(c);
        }

        return id.toString();
    }

    /**
     * Checks if a java Method is found
     * @return true if a Method is found
     */
    public String parseMethod()
    {
        int savedIdx = ci.getIndex();
        
        String id;

        if ((id = parseIdentifier()) == null)
        {
            ci.setIndex(savedIdx);
            return null;
        }
        
        skipWS();
        
        if (!parseChar('(') )
    	{
            ci.setIndex(savedIdx);
            return null;
    	}
        ci.setIndex(ci.getIndex()-1);        
        return id;
    }

    /**
     * Parses the text string (up to the next space) and
     * returns it. The name includes '.' characters.
     * This can be used, for example, when parsing a class name wanting to
     * read in the full name (including package) so that it can then be
     * checked for existence in the CLASSPATH.
     * @return The name
     */
    public String parseName()
    {
        int savedIdx = skipWS();
        String id;

        if ((id = parseIdentifier()) == null)
        {
            return null;
        }

        StringBuffer qn = new StringBuffer(id);

        while (parseChar('.'))
        {
            if ((id = parseIdentifier()) == null)
            {
                ci.setIndex(savedIdx);
                return null;
            }

            qn.append('.').append(id);
        }

        return qn.toString();
    }

    /**
     * Parse a cast in the query from the current position, returning
     * the class that is being cast to. Returns null if the current position
     * doesnt have a cast.
     * @param clr The ClassLoaderResolver
     * @param primary The primary class loader to use (if any)
     * @return The class to cast to
     */
    public Class parseCast(ClassLoaderResolver clr, ClassLoader primary)
    {
        int savedIdx = skipWS();
        String typeName;

        if (!parseChar('(') ||
            (typeName = parseName()) == null || !parseChar(')'))
        {
            ci.setIndex(savedIdx);
            return null;
        }

        try
        {
            return imports.resolveClassDeclaration(typeName, clr, primary);
        }
        catch (ClassNotResolvedException e)
        {
            ci.setIndex(savedIdx);
            throw new NucleusUserException(LOCALISER.msg("021053", typeName));
        }
    }

    /**
     * Utility to return if a character is a decimal digit.
     * @param c The character
     * @return Whether it is a decimal digit
     */
    private final static boolean isDecDigit(char c)
    {
        return c >= '0' && c <= '9';
    }

    /**
     * Utility to return if a character is a octal digit.
     * @param c The character
     * @return Whether it is a octal digit
     */
    private final static boolean isOctDigit(char c)
    {
        return c >= '0' && c <= '7';
    }

    /**
     * Utility to return if a character is a hexadecimal digit.
     * @param c The character
     * @return Whether it is a hexadecimal digit
     */
    private final static boolean isHexDigit(char c)
    {
        return c >= '0' && c <= '9' || c >= 'a' && c <= 'f' || c >= 'A' && c <= 'F';
    }

    /**
     * Parse an integer number from the current position.
     * @return The integer number parsed (null if not valid).
     */
    public BigInteger parseIntegerLiteral()
    {
        int savedIdx = skipWS();

        StringBuffer digits = new StringBuffer();
        int radix;
        char c = ci.current();

        if (c == '0')
        {
            c = ci.next();

            if (c == 'x' || c == 'X')
            {
                radix = 16;
                c = ci.next();

                while (isHexDigit(c))
                {
                    digits.append(c);
                    c = ci.next();
                }
            }
            else if (isOctDigit(c))
            {
                radix = 8;

                do
                {
                    digits.append(c);
                    c = ci.next();
                } while (isOctDigit(c));
            }
            else
            {
                radix = 10;
                digits.append('0');
            }
        }
        else
        {
            radix = 10;

            while (isDecDigit(c))
            {
                digits.append(c);
                c = ci.next();
            }
        }

        if (digits.length() == 0)
        {
            ci.setIndex(savedIdx);
            return null;
        }

        if (c == 'l' || c == 'L')
        {
            ci.next();
        }

        return new BigInteger(digits.toString(), radix);
    }

    /**
     * Parse a floating point number from the current position.
     * @return The floating point number parsed (null if not valid).
     */
    public BigDecimal parseFloatingPointLiteral()
    {
        int savedIdx = skipWS();
        StringBuffer val = new StringBuffer();
        boolean dotSeen = false;
        boolean expSeen = false;
        boolean sfxSeen = false;

        char c = ci.current();

        while (isDecDigit(c))
        {
            val.append(c);
            c = ci.next();
        }

        if (c == '.')
        {
            dotSeen = true;
            val.append(c);
            c = ci.next();

            while (isDecDigit(c))
            {
                val.append(c);
                c = ci.next();
            }
        }

        if (val.length() < (dotSeen ? 2 : 1))
        {
            ci.setIndex(savedIdx);
            return null;
        }

        if (c == 'e' || c == 'E')
        {
            expSeen = true;
            val.append(c);
            c = ci.next();

            if (c != '+' && c != '-' && !isDecDigit(c))
            {
                ci.setIndex(savedIdx);
                return null;
            }

            do
            {
                val.append(c);
                c = ci.next();
            } while (isDecDigit(c));
        }

        if (c == 'f' || c == 'F' || c == 'd' || c == 'D')
        {
            sfxSeen = true;
            ci.next();
        }

        if (!dotSeen && !expSeen && !sfxSeen)
        {
            ci.setIndex(savedIdx);
            return null;
        }

        return new BigDecimal(val.toString());
    }

    /**
     * Parse a boolean from the current position.
     * @return The boolean parsed (null if not valid).
     */
    public Boolean parseBooleanLiteral()
    {
        int savedIdx = skipWS();
        String id;

        if ((id = parseIdentifier()) == null)
        {
            return null;
        }

        if (id.equals("true"))
        {
             return Boolean.TRUE;
        }
        else if (id.equals("false"))
        {
            return Boolean.FALSE;
        }
        else
        {
            ci.setIndex(savedIdx);
            return null;
        }
    }

    /**
     * Utility to return if the next non-whitespace character is a single quote.
     * @return Whether it is a single quote at the current point (ignoring whitespace)
     */
    public boolean nextIsSingleQuote()
    {
        skipWS();
        return (ci.current() == '\'');
    }

    /**
     * Utility to return if the next character is a dot.
     * @return Whether it is a dot at the current point
     */
    public boolean nextIsDot()
    {
        return (ci.current() == '.');
    }

    /**
     * Parse a Character literal.
     * @return the Character parsed. null if single quotes is found
     * @throws NucleusUserException if an invalid character is found or the CharacterIterator is finished
     */
    public Character parseCharacterLiteral()
    {
        skipWS();

        if (ci.current() != '\'')
        {
            return null;
        }

        char c = ci.next();

        if (c == CharacterIterator.DONE)
        {
            throw new NucleusUserException("Invalid character literal: " + input);
        }

        if (c == '\\')
        {
            // Why are we doing this exactly ? If the string is "\\_" then this should be "\_" but doing this
            // we get an Exception thrown for invalid escape expression. See JPQLParser for correct version IMHO
            c = parseEscapedCharacter();
        }

        if (ci.next() != '\'')
        {
            throw new NucleusUserException("Invalid character literal: " + input);
        }

        ci.next();

        return new Character(c);
    }

    /**
     * Parse a String literal.
     * @return the String parsed. null if single quotes or double quotes is found
     * @throws NucleusUserException if an invalid character is found or the CharacterIterator is finished
     */
    public String parseStringLiteral()
    {
        skipWS();

        // Strings can be surrounded by single or double quotes
        char quote = ci.current();
        if (quote != '"' && quote != '\'')
        {
            return null;
        }

        StringBuffer lit = new StringBuffer();
        char c;

        while ((c = ci.next()) != quote)
        {
            if (c == CharacterIterator.DONE)
            {
                throw new NucleusUserException("Invalid string literal: " + input);
            }

            if (c == '\\')
            {
                // Why are we doing this exactly ? If the string is "\\_" then this should be "\_" but doing this
                // we get an Exception thrown for invalid escape expression. See JPQLParser for correct version IMHO
                c = parseEscapedCharacter();
            }

            lit.append(c);
        }

        ci.next();

        return lit.toString();
    }

    /**
     * Parse a escaped character.
     * @return the escaped char
     * @throws NucleusUserException if a escaped character is not valid
     */
    protected char parseEscapedCharacter()
    {
        char c;

        if (isOctDigit(c = ci.next()))
        {
            int i = (c - '0');

            if (isOctDigit(c = ci.next()))
            {
                i = i * 8 + (c - '0');

                if (isOctDigit(c = ci.next()))
                {
                    i = i * 8 + (c - '0');
                }
                else
                {
                    ci.previous();
                }
            }
            else
            {
                ci.previous();
            }

            if (i > 0xff)
            {
                throw new NucleusUserException("Invalid character escape: '\\" + Integer.toOctalString(i) + "'");
            }

            return (char)i;
        }
        else
        {
            switch (c)
            {
                case 'b':   return '\b';
                case 't':   return '\t';
                case 'n':   return '\n';
                case 'f':   return '\f';
                case 'r':   return '\r';
                case '"':   return '"';
                case '\'':  return '\'';
                case '\\':  return '\\';
                default:
                    throw new NucleusUserException("Invalid character escape: '\\" + c + "'");
            }
        }
    }

    /**
     * Checks if null literal is parsed
     * @return true if null literal is found
     */
    public boolean parseNullLiteral()
    {
        int savedIdx = skipWS();
        String id;

        if ((id = parseIdentifier()) == null)
        {
            return false;
        }
        else if (id.equals("null"))
        {
            return true;
        }
        else
        {
            ci.setIndex(savedIdx);
            return false;
        }
    }

    public String remaining()
    {
        StringBuffer sb = new StringBuffer();
        char c = ci.current();
        while (c != CharacterIterator.DONE)
        {
            sb.append(c);
            c = ci.next();
        }
        return sb.toString();
    }
    
    public String toString()
    {
        return input;
    }
}