/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.rdbms;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManager;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.PersistenceConfiguration;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusObjectNotFoundException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.InheritanceStrategy;
import org.datanucleus.plugin.PluginManager;
import org.datanucleus.store.AbstractPersistenceHandler;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.MappedStoreManager;
import org.datanucleus.store.rdbms.fieldmanager.DynamicSchemaFieldManager;
import org.datanucleus.store.rdbms.request.Request;
import org.datanucleus.store.rdbms.request.RequestIdentifier;
import org.datanucleus.store.rdbms.request.RequestType;
import org.datanucleus.store.rdbms.table.ClassView;
import org.datanucleus.store.rdbms.table.SecondaryTable;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.SoftValueMap;

/**
 * Handler for persistence for RDBMS datastores.
 * Makes use of the plugin-point "org.datanucleus.store.rdbms.rdbms_request" defining which
 * request handlers we should use for the different operations.
 */
public class RDBMSPersistenceHandler extends AbstractPersistenceHandler
{
    /** Localiser for messages. */
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    /** Manager for the store. */
    protected final MappedStoreManager storeMgr;

    /** The cache of database requests. Access is synchronized on the map object itself. */
    private Map<RequestIdentifier, Request> requestsByID = Collections.synchronizedMap(new SoftValueMap());

    /**
     * Constructor.
     * @param storeMgr StoreManager
     */
    public RDBMSPersistenceHandler(StoreManager storeMgr)
    {
        this.storeMgr = (MappedStoreManager)storeMgr;
    }

    /**
     * Method to close the handler and release any resources.
     */
    public void close()
    {
        requestsByID.clear();
        requestsByID = null;
    }

    // ------------------------------ Insert ----------------------------------

    /**
     * Inserts a persistent object into the database.
     * @param sm The state manager of the object to be inserted.
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void insertObject(StateManager sm)
    {
        // Check if read-only so update not permitted
        storeMgr.assertReadOnlyForUpdateOfObject(sm);

        // Check if we need to do any updates to the schema before inserting this object
        checkForSchemaUpdatesForFieldsOfObject(sm, sm.getLoadedFieldNumbers());

        ClassLoaderResolver clr = sm.getObjectManager().getClassLoaderResolver();
        String className = sm.getObject().getClass().getName();
        DatastoreClass dc = storeMgr.getDatastoreClass(className, clr);
        if (dc == null)
        {
            if (sm.getClassMetaData().getInheritanceMetaData().getStrategy() == InheritanceStrategy.SUBCLASS_TABLE)
            {
                throw new NucleusUserException(LOCALISER.msg("032013", className));
            }
            throw new NucleusException(LOCALISER.msg("032014", className, 
                sm.getClassMetaData().getInheritanceMetaData().getStrategy())).setFatal();
        }

        if (storeMgr.getRuntimeManager() != null)
        {
            storeMgr.getRuntimeManager().incrementInsertCount();
        }

        insertTable(dc, sm, clr);
    }

    /**
     * Convenience method to handle the insert into the various tables that this object is persisted into.
     * @param table The table to process
     * @param sm StateManager for the object being inserted
     * @param clr ClassLoader resolver
     */
    private void insertTable(DatastoreClass table, StateManager sm, ClassLoaderResolver clr)
    {
        if (table instanceof ClassView)
        {
            throw new NucleusUserException("Cannot perform InsertRequest on RDBMS view " + table);
        }

        DatastoreClass supertable = table.getSuperDatastoreClass();
        if (supertable != null)
        {
            // Process the superclass table first
            insertTable(supertable, sm, clr);
        }

        // Do the actual insert of this table
        Request req = getInsertRequest(table, sm.getObject().getClass(), clr);
        req.execute(sm);

        // Process any secondary tables
        Collection secondaryTables = table.getSecondaryDatastoreClasses();
        if (secondaryTables != null)
        {
            Iterator tablesIter = secondaryTables.iterator();
            while (tablesIter.hasNext())
            {
                // Process the secondary table
                SecondaryTable secTable = (SecondaryTable)tablesIter.next();
                insertTable(secTable, sm, clr);
            }
        }
    }

    /**
     * Returns a request object that will insert a row in the given table. 
     * The store manager will cache the request object for re-use by subsequent requests to the same table.
     * @param table The table into which to insert.
     * @param cls class of the object of the request
     * @param clr ClassLoader resolver
     * @return An insertion request object.
     */
    private Request getInsertRequest(DatastoreClass table, Class cls, ClassLoaderResolver clr)
    {
        RequestIdentifier reqID = new RequestIdentifier(table, null, RequestType.INSERT, cls.getName());
        Request req = requestsByID.get(reqID);
        if (req == null)
        {
            PluginManager pluginMgr = storeMgr.getOMFContext().getPluginManager();
            PersistenceConfiguration conf = storeMgr.getOMFContext().getPersistenceConfiguration();
            String key = conf.getStringProperty("datanucleus.rdbms.request.insert");
            try
            {
                req = (Request)pluginMgr.createExecutableExtension("org.datanucleus.store.rdbms.rdbms_request",
                    new String[]{"type", "name"}, new String[] {RequestType.INSERT.toString(), key}, "class",
                    new Class[] {DatastoreClass.class, Class.class, ClassLoaderResolver.class},
                    new Object[] {table, cls, clr});
            }
            catch (Exception e)
            {
                throw new NucleusException("Attempt to instantiate insert request gave an error", e);
            }
            requestsByID.put(reqID, req);
        }
        return req;
    }

    // ------------------------------ Fetch ----------------------------------

    /**
     * Fetches a persistent object from the database.
     * @param sm The state manager of the object to be fetched.
     * @param fieldNumbers The numbers of the fields to be fetched.
     * @throws NucleusObjectNotFoundException if the object doesnt exist
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void fetchObject(StateManager sm, int fieldNumbers[])
    {
        AbstractMemberMetaData[] fmds = null;
        if (fieldNumbers != null && fieldNumbers.length > 0)
        {
            // Convert the field numbers for this class into their metadata for the table
            fmds = new AbstractMemberMetaData[fieldNumbers.length];
            for (int i=0;i<fmds.length;i++)
            {
                fmds[i] = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumbers[i]);
            }

            if (sm.getPcObjectType() != StateManager.PC)
            {
                StringBuffer str = new StringBuffer();
                for (int i=0;i<fmds.length;i++)
                {
                    if (i > 0)
                    {
                        str.append(',');
                    }
                    str.append(fmds[i].getName());
                }
                NucleusLogger.PERSISTENCE.info("Request to load fields \"" + str.toString() +
                    "\" of class " + sm.getClassMetaData().getFullClassName() + " but object is embedded, so ignored");
            }
            else
            {
                if (storeMgr.getRuntimeManager() != null)
                {
                    storeMgr.getRuntimeManager().incrementFetchCount();
                }

                ClassLoaderResolver clr = sm.getObjectManager().getClassLoaderResolver();
                DatastoreClass table = storeMgr.getDatastoreClass(sm.getObject().getClass().getName(), clr);
                Request req = getFetchRequest(table, fmds, sm.getObject().getClass(), clr);
                req.execute(sm);
            }
        }
    }

    /**
     * Returns a request object that will fetch a row from the given table. 
     * The store manager will cache the request object for re-use by subsequent requests to the same table.
     * @param table The table from which to fetch.
     * @param mmds MetaData for the fields corresponding to the columns to be fetched.
     * @param cls class of the object of the request
     * @param clr ClassLoader resolver
     * @return A fetch request object.
     */
    private Request getFetchRequest(DatastoreClass table, AbstractMemberMetaData[] mmds, Class cls, 
            ClassLoaderResolver clr)
    {
        RequestIdentifier reqID = new RequestIdentifier(table, mmds, RequestType.FETCH, 
            cls.getName());
        Request req = requestsByID.get(reqID);
        if (req == null)
        {
            PluginManager pluginMgr = storeMgr.getOMFContext().getPluginManager();
            PersistenceConfiguration conf = storeMgr.getOMFContext().getPersistenceConfiguration();
            String key = conf.getStringProperty("datanucleus.rdbms.request.fetch");
            try
            {
                req = (Request)pluginMgr.createExecutableExtension("org.datanucleus.store.rdbms.rdbms_request",
                    new String[]{"type", "name"}, new String[] {RequestType.FETCH.toString(), key}, "class",
                    new Class[] {DatastoreClass.class, AbstractMemberMetaData[].class, Class.class, ClassLoaderResolver.class},
                    new Object[] {table, mmds, cls, clr});
            }
            catch (Exception e)
            {
                throw new NucleusException("Attempt to instantiate insert request gave an error", e);
            }
            requestsByID.put(reqID, req);
        }
        return req;
    }

    // ------------------------------ Update ----------------------------------

    /**
     * Updates a persistent object in the database.
     * @param sm The state manager of the object to be updated.
     * @param fieldNumbers The numbers of the fields to be updated.
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void updateObject(StateManager sm, int fieldNumbers[])
    {
        // Check if read-only so update not permitted
        storeMgr.assertReadOnlyForUpdateOfObject(sm);

        // Check if we need to do any updates to the schema before updating this object
        checkForSchemaUpdatesForFieldsOfObject(sm, fieldNumbers);

        AbstractMemberMetaData[] fmds = null;
        if (fieldNumbers != null && fieldNumbers.length > 0)
        {
            // Convert the field numbers for this class into their metadata for the table
            fmds = new AbstractMemberMetaData[fieldNumbers.length];
            for (int i=0;i<fmds.length;i++)
            {
                fmds[i] = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumbers[i]);
            }

            if (storeMgr.getRuntimeManager() != null)
            {
                storeMgr.getRuntimeManager().incrementUpdateCount();
            }

            ClassLoaderResolver clr = sm.getObjectManager().getClassLoaderResolver();
            DatastoreClass dc = storeMgr.getDatastoreClass(sm.getObject().getClass().getName(), clr);
            updateTable(dc, sm, clr, fmds);
        }
    }

    /**
     * Convenience method to handle the update into the various tables that this object is persisted into.
     * @param table The table to process
     * @param sm StateManager for the object being updated
     * @param clr ClassLoader resolver
     * @param fieldMetaData MetaData for the fields being updated
     */
    private void updateTable(DatastoreClass table, StateManager sm, ClassLoaderResolver clr,
            AbstractMemberMetaData[] fieldMetaData)
    {
        if (table instanceof ClassView)
        {
            throw new NucleusUserException("Cannot perform UpdateRequest on RDBMS view " + table);
        }

        DatastoreClass supertable = table.getSuperDatastoreClass();
        if (supertable != null)
        {
            // Process the superclass table first
            updateTable(supertable, sm, clr, fieldMetaData);
        }

        // Do the actual update of this table
        Request req = getUpdateRequest(table, fieldMetaData, sm.getObject().getClass(), clr);
        req.execute(sm);

        // Update any secondary tables
        Collection secondaryTables = table.getSecondaryDatastoreClasses();
        if (secondaryTables != null)
        {
            Iterator tablesIter = secondaryTables.iterator();
            while (tablesIter.hasNext())
            {
                // Process the secondary table
                SecondaryTable secTable = (SecondaryTable)tablesIter.next();
                updateTable(secTable, sm, clr, fieldMetaData);
            }
        }
    }

    /**
     * Returns a request object that will update a row in the given table. 
     * The store manager will cache the request object for re-use by subsequent requests to the same table.
     * @param table The table in which to update.
     * @param mmds The metadata corresponding to the columns to be updated. 
     *     MetaData whose columns exist in supertables will be ignored.
     * @param cls class of the object of the request
     * @param clr ClassLoader resolver
     * @return An update request object.
     */
    private Request getUpdateRequest(DatastoreClass table, AbstractMemberMetaData[] mmds, Class cls, 
            ClassLoaderResolver clr)
    {
        RequestIdentifier reqID = new RequestIdentifier(table, mmds, RequestType.UPDATE, cls.getName());
        Request req = requestsByID.get(reqID);
        if (req == null)
        {
            PluginManager pluginMgr = storeMgr.getOMFContext().getPluginManager();
            PersistenceConfiguration conf = storeMgr.getOMFContext().getPersistenceConfiguration();
            String key = conf.getStringProperty("datanucleus.rdbms.request.update");
            try
            {
                req = (Request)pluginMgr.createExecutableExtension("org.datanucleus.store.rdbms.rdbms_request",
                    new String[]{"type", "name"}, new String[] {RequestType.UPDATE.toString(), key}, "class",
                    new Class[] {DatastoreClass.class, AbstractMemberMetaData[].class, Class.class, ClassLoaderResolver.class},
                    new Object[] {table, mmds, cls, clr});
            }
            catch (Exception e)
            {
                throw new NucleusException("Attempt to instantiate insert request gave an error", e);
            }
            requestsByID.put(reqID, req);
        }
        return req;
    }

    // ------------------------------ Delete ----------------------------------

    /**
     * Deletes a persistent object from the database.
     * @param sm The state manager of the object to be deleted.
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void deleteObject(StateManager sm)
    {
        // Check if read-only so update not permitted
        storeMgr.assertReadOnlyForUpdateOfObject(sm);

        if (storeMgr.getRuntimeManager() != null)
        {
            storeMgr.getRuntimeManager().incrementDeleteCount();
        }

        ClassLoaderResolver clr = sm.getObjectManager().getClassLoaderResolver();
        DatastoreClass dc = storeMgr.getDatastoreClass(sm.getObject().getClass().getName(), clr);
        deleteTable(dc, sm, clr);
    }

    /**
     * Convenience method to handle the delete from the various tables that this object is persisted into.
     * @param table The table to process
     * @param sm StateManager for the object being deleted
     * @param clr ClassLoader resolver
     */
    private void deleteTable(DatastoreClass table, StateManager sm, ClassLoaderResolver clr)
    {
        if (table instanceof ClassView)
        {
            throw new NucleusUserException("Cannot perform DeleteRequest on RDBMS view " + table);
        }

        // Delete any secondary tables
        Collection secondaryTables = table.getSecondaryDatastoreClasses();
        if (secondaryTables != null)
        {
            Iterator tablesIter = secondaryTables.iterator();
            while (tablesIter.hasNext())
            {
                // Process the secondary table
                SecondaryTable secTable = (SecondaryTable)tablesIter.next();
                deleteTable(secTable, sm, clr);
            }
        }

        // Do the actual delete of this table
        Request req = getDeleteRequest(table, sm.getObject().getClass(), clr);
        req.execute(sm);

        DatastoreClass supertable = table.getSuperDatastoreClass();
        if (supertable != null)
        {
            // Process the superclass table last
            deleteTable(supertable, sm, clr);
        }
    }

    /**
     * Returns a request object that will delete a row from the given table.
     * The store manager will cache the request object for re-use by subsequent requests to the same table.
     * @param table The table from which to delete.
     * @param cls class of the object of the request
     * @param clr ClassLoader resolver
     * @return A deletion request object.
     */
    private Request getDeleteRequest(DatastoreClass table, Class cls, ClassLoaderResolver clr)
    {
        RequestIdentifier reqID = new RequestIdentifier(table, null, RequestType.DELETE, cls.getName());
        Request req = requestsByID.get(reqID);
        if (req == null)
        {
            PluginManager pluginMgr = storeMgr.getOMFContext().getPluginManager();
            PersistenceConfiguration conf = storeMgr.getOMFContext().getPersistenceConfiguration();
            String key = conf.getStringProperty("datanucleus.rdbms.request.delete");
            try
            {
                req = (Request)pluginMgr.createExecutableExtension("org.datanucleus.store.rdbms.rdbms_request",
                    new String[]{"type", "name"}, new String[] {RequestType.DELETE.toString(), key}, "class",
                    new Class[] {DatastoreClass.class, Class.class, ClassLoaderResolver.class},
                    new Object[] {table, cls, clr});
            }
            catch (Exception e)
            {
                throw new NucleusException("Attempt to instantiate insert request gave an error", e);
            }
            requestsByID.put(reqID, req);
        }
        return req;
    }

    // ------------------------------ Locate ----------------------------------

    /**
     * Locates this object in the datastore.
     * @param sm The StateManager for the object to be found
     * @throws NucleusObjectNotFoundException if the object doesnt exist
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void locateObject(StateManager sm)
    {
        ClassLoaderResolver clr = sm.getObjectManager().getClassLoaderResolver();
        DatastoreClass table = storeMgr.getDatastoreClass(sm.getObject().getClass().getName(), clr);
        Request req = getLocateRequest(table, sm.getObject().getClass().getName());
        req.execute(sm);
    }

    /**
     * Returns a request object that will locate a row from the given table.
     * The store manager will cache the request object for re-use by subsequent requests to the same table.
     * @param table The table from which to locate.
     * @param className the class name of the object of the request
     * @return A locate request object.
     */
    private Request getLocateRequest(DatastoreClass table, String className)
    {
        RequestIdentifier reqID = new RequestIdentifier(table, null, RequestType.LOCATE, className);
        Request req = requestsByID.get(reqID);
        if (req == null)
        {
            PluginManager pluginMgr = storeMgr.getOMFContext().getPluginManager();
            PersistenceConfiguration conf = storeMgr.getOMFContext().getPersistenceConfiguration();
            String key = conf.getStringProperty("datanucleus.rdbms.request.locate");
            try
            {
                req = (Request)pluginMgr.createExecutableExtension("org.datanucleus.store.rdbms.rdbms_request",
                    new String[]{"type", "name"}, new String[] {RequestType.LOCATE.toString(), key}, "class",
                    new Class[] {DatastoreClass.class}, new Object[] {table});
            }
            catch (Exception e)
            {
                throw new NucleusException("Attempt to instantiate insert request gave an error", e);
            }
            requestsByID.put(reqID, req);
        }
        return req;
    }

    // ------------------------------ Find ----------------------------------

    /**
     * Method to return a persistable object with the specified id. Optional operation for StoreManagers.
     * Should return a (at least) hollow PersistenceCapable object if the store manager supports the operation.
     * If the StoreManager is managing the in-memory object instantiation (as part of co-managing the object 
     * lifecycle in general), then the StoreManager has to create the object during this call (if it is not 
     * already created). Most relational databases leave the in-memory object instantion to Core, but some 
     * object databases may manage the in-memory object instantion, effectively preventing Core of doing this.
     * <p>
     * StoreManager implementations may simply return null, indicating that they leave the object instantiate to 
     * us. Other implementations may instantiate the object in question (whether the implementation may trust 
     * that the object is not already instantiated has still to be determined). If an implementation believes
     * that an object with the given ID should exist, but in fact does not exist, then the implementation should 
     * throw a RuntimeException. It should not silently return null in this case.
     * </p>
     * @param om the ObjectManager which will manage the object
     * @param id the id of the object in question.
     * @return a persistable object with a valid object state (for example: hollow) or null, 
     *     indicating that the implementation leaves the instantiation work to us.
     */
    public Object findObject(ObjectManager om, Object id)
    {
        return null;
    }

    // ------------------------------ Convenience ----------------------------------

    /**
     * Convenience method to remove all requests since the schema has changed.
     */
    public void removeAllRequests()
    {
        synchronized (requestsByID)
        {
            requestsByID.clear();
        }
    }

    /**
     * Convenience method to remove all requests that use a particular table since the structure
     * of the table has changed potentially leading to missing columns in the cached version.
     * @param table The table
     */
    public void removeRequestsForTable(DatastoreClass table)
    {
        synchronized(requestsByID)
        {
            // Synchronise on the "requestsById" set since while it is "synchronised itself, all iterators needs this sync
            Set keySet = new HashSet(requestsByID.keySet());
            Iterator<RequestIdentifier> keyIter = keySet.iterator();
            while (keyIter.hasNext())
            {
                RequestIdentifier reqId = keyIter.next();
                if (reqId.getTable() == table)
                {
                    requestsByID.remove(reqId);
                }
            }
        }
    }

    /**
     * Check if we need to update the schema before performing an insert/update.
     * This is typically of use where the user has an interface field and some new implementation
     * is trying to be persisted to that field, so we need to update the schema.
     * @param sm StateManager for the object
     * @param fieldNumbers The fields to check for required schema updates
     */
    private void checkForSchemaUpdatesForFieldsOfObject(StateManager sm, int[] fieldNumbers)
    {
        if (storeMgr.getOMFContext().getPersistenceConfiguration().getBooleanObjectProperty("datanucleus.rdbms.dynamicSchemaUpdates").booleanValue())
        {
            DynamicSchemaFieldManager dynamicSchemaFM = new DynamicSchemaFieldManager((RDBMSStoreManager)storeMgr, sm);
            sm.provideFields(fieldNumbers, dynamicSchemaFM);
            if (dynamicSchemaFM.hasPerformedSchemaUpdates())
            {
                requestsByID.clear();
            }
        }
    }
}