/**********************************************************************
Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.enhancer.asm;

import javax.jdo.PersistenceManager;
import javax.jdo.identity.ByteIdentity;
import javax.jdo.identity.CharIdentity;
import javax.jdo.identity.IntIdentity;
import javax.jdo.identity.LongIdentity;
import javax.jdo.identity.ObjectIdentity;
import javax.jdo.identity.ShortIdentity;
import javax.jdo.identity.StringIdentity;
import javax.jdo.spi.Detachable;
import javax.jdo.spi.JDOPermission;
import javax.jdo.spi.PersistenceCapable;
import javax.jdo.spi.StateManager;

import org.datanucleus.enhancer.ClassEnhancer;
import org.datanucleus.util.Localiser;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;

/**
 * Utility class for ASM.
 * ASM operates around two basic pieces of information about any type.
 * <ul>
 * <li><b>ASM class name</b> : this is the normal fully qualified class name but replacing
 * the dots with slashes. So java.lang.String will have an ASM class name of java/lang/String</li>
 * <li><b>Class descriptor</b> : this is used where a type is referred to in a calling sequence etc.
 * and for object types is typically things like "Ljava.lang.String;", but there are variants for primitives.</li>
 * </ul>
 */
public final class ASMUtils
{
    /** Localiser for messages. */
    protected static Localiser LOCALISER = Localiser.getInstance("org.datanucleus.enhancer.Localisation",
        ClassEnhancer.class.getClassLoader());

    /** ASM class name for boolean. */
    public final static String ACN_boolean = boolean.class.getName();

    /** ASM class name for byte. */
    public final static String ACN_byte = byte.class.getName();

    /** ASM class name for char. */
    public final static String ACN_char = char.class.getName();

    /** ASM class name for double. */
    public final static String ACN_double = double.class.getName();

    /** ASM class name for float. */
    public final static String ACN_float = float.class.getName();

    /** ASM class name for int. */
    public final static String ACN_int = int.class.getName();

    /** ASM class name for long. */
    public final static String ACN_long = long.class.getName();

    /** ASM class name for short. */
    public final static String ACN_short = short.class.getName();

    /** ASM class name for Boolean. */
    public final static String ACN_Boolean = Boolean.class.getName().replace('.', '/');

    /** ASM class name for Byte. */
    public final static String ACN_Byte = Byte.class.getName().replace('.', '/');

    /** ASM class name for Character. */
    public final static String ACN_Character = Character.class.getName().replace('.', '/');

    /** ASM class name for Double. */
    public final static String ACN_Double = Double.class.getName().replace('.', '/');

    /** ASM class name for Float. */
    public final static String ACN_Float = Float.class.getName().replace('.', '/');

    /** ASM class name for Integer. */
    public final static String ACN_Integer = Integer.class.getName().replace('.', '/');

    /** ASM class name for Long. */
    public final static String ACN_Long = Long.class.getName().replace('.', '/');

    /** ASM class name for Short. */
    public final static String ACN_Short = Short.class.getName().replace('.', '/');

    /** ASM class name for java.lang.String. */
    public final static String ACN_String = String.class.getName().replace('.', '/');

    /** ASM class name for java.lang.Object. */
    public final static String ACN_Object = Object.class.getName().replace('.', '/');

    /** ASM class name for javax.jdo.spi.StateManager. */
    public final static String ACN_StateManager = StateManager.class.getName().replace('.', '/');

    /** ASM class name for javax.jdo.PersistenceManager. */
    public final static String ACN_PersistenceManager = PersistenceManager.class.getName().replace('.', '/');

    /** ASM class name for javax.jdo.spi.PersistenceCapable. */
    public final static String ACN_PersistenceCapable = PersistenceCapable.class.getName().replace('.', '/');

    /** ASM class name for javax.jdo.spi.Detachable. */
    public final static String ACN_Detachable = Detachable.class.getName().replace('.', '/');

    /** ASM class name for javax.jdo.spi.JDOPermission. */
    public final static String ACN_JDOPermission = JDOPermission.class.getName().replace('.', '/');

    /** ASM class name for java.lang.SecurityManager. */
    public final static String ACN_SecurityManager = SecurityManager.class.getName().replace('.', '/');

    /** Class descriptor for String. */
    public final static String CD_String = Type.getDescriptor(String.class);

    /** Descriptor for java.lang.Object. */
    public final static String CD_Object = Type.getDescriptor(Object.class);

    /** Descriptor for ByteIdentity. */
    public final static String CD_ByteIdentity = Type.getDescriptor(ByteIdentity.class);

    /** Descriptor for CharIdentity. */
    public final static String CD_CharIdentity = Type.getDescriptor(CharIdentity.class);

    /** Descriptor for IntIdentity. */
    public final static String CD_IntIdentity = Type.getDescriptor(IntIdentity.class);

    /** Descriptor for LongIdentity. */
    public final static String CD_LongIdentity = Type.getDescriptor(LongIdentity.class);

    /** Descriptor for ShortIdentity. */
    public final static String CD_ShortIdentity = Type.getDescriptor(ShortIdentity.class);

    /** Descriptor for StringIdentity. */
    public final static String CD_StringIdentity = Type.getDescriptor(StringIdentity.class);

    /** Descriptor for ObjectIdentity. */
    public final static String CD_ObjectIdentity = Type.getDescriptor(ObjectIdentity.class);

    /** Descriptor for javax.jdo.spi.StateManager. */
    public final static String CD_StateManager = Type.getDescriptor(StateManager.class);

    /** Descriptor for javax.jdo.PersistenceManager. */
    public final static String CD_PersistenceManager = Type.getDescriptor(PersistenceManager.class);

    /** Descriptor for javax.jdo.spi.PersistenceCapable. */
    public final static String CD_PersistenceCapable = Type.getDescriptor(PersistenceCapable.class);

    /** Descriptor for javax.jdo.spi.Detachable. */
    public final static String CD_Detachable = Type.getDescriptor(Detachable.class);

    /** Descriptor for java.lang.SecurityManager. */
    public final static String CD_SecurityManager = Type.getDescriptor(SecurityManager.class);

    /** Descriptor for PersistenceCapable$ObjectIdFieldConsumer. */
    public final static String CD_ObjectIdFieldConsumer = Type.getDescriptor(PersistenceCapable.ObjectIdFieldConsumer.class);

    /** Descriptor for PersistenceCapable$ObjectIdFieldConsumer. */
    public final static String CD_ObjectIdFieldSupplier = Type.getDescriptor(PersistenceCapable.ObjectIdFieldSupplier.class);

    /**
     * private constructor to prevent instantiation.
     */
    private ASMUtils()
    {
    }

    /**
     * Convenience method to add a BIPUSH-type int to the visitor.
     * @param visitor The MethodVisitor
     * @param i number
     */
    public static void addBIPUSHToMethod(MethodVisitor visitor, final int i)
    {
        if (i < 6)
        {
            switch (i)
            {
                case 0 :
                    visitor.visitInsn(Opcodes.ICONST_0);
                    break;
                case 1 :
                    visitor.visitInsn(Opcodes.ICONST_1);
                    break;
                case 2 :
                    visitor.visitInsn(Opcodes.ICONST_2);
                    break;
                case 3 :
                    visitor.visitInsn(Opcodes.ICONST_3);
                    break;
                case 4 :
                    visitor.visitInsn(Opcodes.ICONST_4);
                    break;
                case 5 :
                    visitor.visitInsn(Opcodes.ICONST_5);
                    break;
            }
        }
        else if (i < Byte.MAX_VALUE)
        {
            visitor.visitIntInsn(Opcodes.BIPUSH, i);
        }
        else if (i < Short.MAX_VALUE)
        {
            visitor.visitIntInsn(Opcodes.SIPUSH, i);
        }
    }

    /**
     * Convenience method to add a return statement based on the type to be returned.
     * @param visitor The MethodVisitor
     * @param type The type to return
     */
    public static void addReturnForType(MethodVisitor visitor, Class type)
    {
        if (type == int.class || type == boolean.class || type == byte.class || type == char.class || type == short.class)
        {
            visitor.visitInsn(Opcodes.IRETURN);
        }
        else if (type == double.class)
        {
            visitor.visitInsn(Opcodes.DRETURN);
        }
        else if (type == float.class)
        {
            visitor.visitInsn(Opcodes.FRETURN);
        }
        else if (type == long.class)
        {
            visitor.visitInsn(Opcodes.LRETURN);
        }
        else
        {
            visitor.visitInsn(Opcodes.ARETURN);
        }
    }

    /**
     * Convenience method to add a load statement based on the type to be loaded.
     * @param visitor The MethodVisitor
     * @param type The type to load
     * @param number Number to load
     */
    public static void addLoadForType(MethodVisitor visitor, Class type, int number)
    {
        if (type == int.class || type == boolean.class || type == byte.class || type == char.class || type == short.class)
        {
            visitor.visitVarInsn(Opcodes.ILOAD, number);
        }
        else if (type == double.class)
        {
            visitor.visitVarInsn(Opcodes.DLOAD, number);
        }
        else if (type == float.class)
        {
            visitor.visitVarInsn(Opcodes.FLOAD, number);
        }
        else if (type == long.class)
        {
            visitor.visitVarInsn(Opcodes.LLOAD, number);
        }
        else
        {
            visitor.visitVarInsn(Opcodes.ALOAD, number);
        }
    }

    /**
     * Convenience method to give the JDO method name given the type.
     * This is for the assorted methods on the JDO StateManager called things like "replacingStringField",
     * "replacingObjectField", "providedIntField", etc. Just returns the "type" part of the name.
     * <ul>
     * <li>Boolean, bool : returns "Boolean"</li>
     * <li>Byte, byte : returns "Byte"</li>
     * <li>Character, char : returns "Char"</li>
     * <li>Double, double : returns "Double"</li>
     * <li>Float, float : returns "Float"</li>
     * <li>Integer, int : returns "Int"</li>
     * <li>Long, long : returns "Long"</li>
     * <li>Short, short : returns "Short"</li>
     * <li>String : returns "String"</li>
     * <li>all others : returns "Object"</li>
     * </ul>
     * @param cls The type of the field
     * @return Name for the method
     */
    public static String getTypeNameForJDOMethod(Class cls)
    {
        if (cls == null)
        {
            return null;
        }
        else if (cls == boolean.class)
        {
            return "Boolean";
        }
        else if (cls == byte.class)
        {
            return "Byte";
        }
        else if (cls == char.class)
        {
            return "Char";
        }
        else if (cls == double.class)
        {
            return "Double";
        }
        else if (cls == float.class)
        {
            return "Float";
        }
        else if (cls == int.class)
        {
            return "Int";
        }
        else if (cls == long.class)
        {
            return "Long";
        }
        else if (cls == short.class)
        {
            return "Short";
        }
        else if (cls == String.class)
        {
            return "String";
        }
        // Byte, Boolean, Character, Double, Float, Integer, Long, Short go through Object too
        return "Object";
    }

    /**
     * Return the ASM type descriptor for the input class.
     * @param clsName The input class name
     * @return The ASM type descriptor name
     */
    public static String getTypeDescriptorForType(String clsName)
    {
        if (clsName == null)
        {
            return null;
        }
        else if (clsName.equals(boolean.class.getName()))
        {
            return Type.BOOLEAN_TYPE.getDescriptor();
        }
        else if (clsName.equals(byte.class.getName()))
        {
            return Type.BYTE_TYPE.getDescriptor();
        }
        else if (clsName.equals(char.class.getName()))
        {
            return Type.CHAR_TYPE.getDescriptor();
        }
        else if (clsName.equals(double.class.getName()))
        {
            return Type.DOUBLE_TYPE.getDescriptor();
        }
        else if (clsName.equals(float.class.getName()))
        {
            return Type.FLOAT_TYPE.getDescriptor();
        }
        else if (clsName.equals(int.class.getName()))
        {
            return Type.INT_TYPE.getDescriptor();
        }
        else if (clsName.equals(long.class.getName()))
        {
            return Type.LONG_TYPE.getDescriptor();
        }
        else if (clsName.equals(short.class.getName()))
        {
            return Type.SHORT_TYPE.getDescriptor();
        }
        else if (clsName.equals(String.class.getName()))
        {
            return CD_String;
        }
        else
        {
            return "L" + clsName.replace('.', '/') + ";";
        }
    }
    /**
     * Convenience method to give the descriptor for use in a JDO "field" method.
     * This is for the assorted methods on the JDO StateManager called things like "replacingStringField",
     * "replacingObjectField", "providedIntField", etc. Returns the ASM descriptor equivalent for the method used
     * <ul>
     * <li>Boolean, bool : returns "Boolean"</li>
     * <li>Byte, byte : returns "Byte"</li>
     * <li>Character, char : returns "Char"</li>
     * <li>Double, double : returns "Double"</li>
     * <li>Float, float : returns "Float"</li>
     * <li>Integer, int : returns "Int"</li>
     * <li>Long, long : returns "Long"</li>
     * <li>Short, short : returns "Short"</li>
     * <li>String : returns "String"</li>
     * <li>all others : returns "Object"</li>
     * </ul>
     * TODO Cache these descriptors/classes etc
     * @param cls The type of the field
     * @return Name for the method
     */
    public static String getTypeDescriptorForJDOMethod(Class cls)
    {
        if (cls == null)
        {
            return null;
        }
        else if (cls == boolean.class)
        {
            return Type.BOOLEAN_TYPE.getDescriptor();
        }
        else if (cls == byte.class)
        {
            return Type.BYTE_TYPE.getDescriptor();
        }
        else if (cls == char.class)
        {
            return Type.CHAR_TYPE.getDescriptor();
        }
        else if (cls == double.class)
        {
            return Type.DOUBLE_TYPE.getDescriptor();
        }
        else if (cls == float.class)
        {
            return Type.FLOAT_TYPE.getDescriptor();
        }
        else if (cls == int.class)
        {
            return Type.INT_TYPE.getDescriptor();
        }
        else if (cls == long.class)
        {
            return Type.LONG_TYPE.getDescriptor();
        }
        else if (cls == short.class)
        {
            return Type.SHORT_TYPE.getDescriptor();
        }
        else if (cls == String.class)
        {
            return CD_String;
        }
        return CD_Object;
    }

    /**
     * Accessor for the descriptor for the return type from SingleFieldIdentity.getKey()
     * for the passed identity class name.
     * @param oidClassName Name of the SingleFieldIdentity class
     * @return The descriptor of the return type from getKey
     */
    public static String getTypeDescriptorForSingleFieldIdentityGetKey(String oidClassName)
    {
        if (oidClassName.equals(LongIdentity.class.getName()))
        {
            return Type.LONG_TYPE.getDescriptor();
        }
        else if (oidClassName.equals(IntIdentity.class.getName()))
        {
            return Type.INT_TYPE.getDescriptor();
        }
        else if (oidClassName.equals(ShortIdentity.class.getName()))
        {
            return Type.SHORT_TYPE.getDescriptor();
        }
        else if (oidClassName.equals(CharIdentity.class.getName()))
        {
            return Type.CHAR_TYPE.getDescriptor();
        }
        else if (oidClassName.equals(ByteIdentity.class.getName()))
        {
            return Type.BYTE_TYPE.getDescriptor();
        }
        else if (oidClassName.equals(StringIdentity.class.getName()))
        {
            return CD_String;
        }
        else if (oidClassName.equals(ObjectIdentity.class.getName()))
        {
            return CD_Object;
        }
        return null;
    }

    /**
     * Accessor for the descriptor for a SingleFieldIdentity type.
     * @param oidClassName Name of the SingleFieldIdentity class
     * @return The descriptor of the SingleFieldIdentity type
     */
    public static String getSingleFieldIdentityDescriptor(String oidClassName)
    {
        if (oidClassName.equals(LongIdentity.class.getName()))
        {
            return CD_LongIdentity;
        }
        else if (oidClassName.equals(IntIdentity.class.getName()))
        {
            return CD_IntIdentity;
        }
        else if (oidClassName.equals(StringIdentity.class.getName()))
        {
            return CD_StringIdentity;
        }
        else if (oidClassName.equals(ShortIdentity.class.getName()))
        {
            return CD_ShortIdentity;
        }
        else if (oidClassName.equals(CharIdentity.class.getName()))
        {
            return CD_CharIdentity;
        }
        else if (oidClassName.equals(ByteIdentity.class.getName()))
        {
            return CD_ByteIdentity;
        }
        else if (oidClassName.equals(ObjectIdentity.class.getName()))
        {
            return CD_ObjectIdentity;
        }
        return null;
    }

    /**
     * Convenience method to give the JDO method type name for a singleFieldIdentity class name.
     * Used for jdoCopyKeyFields[To/From]ObjectId and defines the "type name" used for things like storeXXXField.
     * <ul>
     * <li>Byte, byte : returns "Byte"</li>
     * <li>Character, char : returns "Char"</li>
     * <li>Integer, int : returns "Int"</li>
     * <li>Long, long : returns "Long"</li>
     * <li>Short, short : returns "Short"</li>
     * <li>String : returns "String"</li>
     * <li>all others : returns "Object"</li>
     * </ul>
     * @param oidClassName Name of the single field identity class
     * @return Name for the method
     */
    public static String getTypeNameForUseWithSingleFieldIdentity(String oidClassName)
    {
        if (oidClassName == null)
        {
            return null;
        }
        else if (oidClassName.equals(ByteIdentity.class.getName()))
        {
            return "Byte";
        }
        else if (oidClassName.equals(CharIdentity.class.getName()))
        {
            return "Char";
        }
        else if (oidClassName.equals(IntIdentity.class.getName()))
        {
            return "Int";
        }
        else if (oidClassName.equals(LongIdentity.class.getName()))
        {
            return "Long";
        }
        else if (oidClassName.equals(ShortIdentity.class.getName()))
        {
            return "Short";
        }
        else if (oidClassName.equals(StringIdentity.class.getName()))
        {
            return "String";
        }
        return "Object";
    }

    /**
     * Convenience method to return the ASM class name to use as input via the SingleFieldIdentity constructor.
     * Means that if the fieldType is primitive we return the ASM class name of the object wrapper.
     * @param fieldType Type of the field
     * @return ASM class name to use as input in the constructor
     */
    public static String getASMClassNameForSingleFieldIdentityConstructor(Class fieldType)
    {
        if (fieldType == null)
        {
            return null;
        }
        else if (fieldType == byte.class || fieldType == Byte.class)
        {
            return ACN_Byte;
        }
        else if (fieldType == char.class || fieldType == Character.class)
        {
            return ACN_Character;
        }
        else if (fieldType == int.class || fieldType == Integer.class)
        {
            return ACN_Integer;
        }
        else if (fieldType == long.class || fieldType == Long.class)
        {
            return ACN_Long;
        }
        else if (fieldType == short.class || fieldType == Short.class)
        {
            return ACN_Short;
        }
        else if (fieldType == String.class)
        {
            return ACN_String;
        }
        else
        {
            return ACN_Object;
        }
    }
}