/**********************************************************************
Copyright (c) 2010 Peter Dettman and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.types.queued;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;

import org.datanucleus.state.ObjectProvider;
import org.datanucleus.store.scostore.Store;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.StringUtils;

/**
 * Queue of operations to be performed on second class collections/maps.
 * When we are queueing operations until flush()/commit() they wait in this queue until the moment
 * arrives for flushing to the datastore (and <pre>performAll</pre> is called).
 * Note that this queue will contain all operations to be performed for an ExecutionContext.
 */
public class SCOOperationQueue
{
    protected static final Localiser LOCALISER = Localiser.getInstance("org.datanucleus.Localisation",
        org.datanucleus.ClassConstants.NUCLEUS_CONTEXT_LOADER);

    protected List<QueuedOperation> queuedOperations = new ArrayList<QueuedOperation>();

    /**
     * Method to add the specified operation to the operation queue.
     * @param oper Operation
     */
    public void enqueue(QueuedOperation oper)
    {
        queuedOperations.add(oper);
    }

    public void clear()
    {
        queuedOperations.clear();
    }

    /**
     * Method to provide access to inspect the queued operations. The returned list is unmodifiable.
     * @return The queued operations
     */
    public List<QueuedOperation> getOperations()
    {
        return Collections.unmodifiableList(queuedOperations);
    }

    /**
     * Method to perform all operations queued for the specified ObjectProvider and backing store.
     * Those operations are then removed from the queue.
     * @param store The backing store
     * @param op ObjectProvider
     */
    public void performAll(Store store, ObjectProvider op)
    {
        if (NucleusLogger.PERSISTENCE.isDebugEnabled())
        {
            NucleusLogger.PERSISTENCE.debug(LOCALISER.msg("023005", 
                op.getObjectAsPrintable(), store.getOwnerMemberMetaData().getFullFieldName()));
        }

        // Extract those operations for the specified backing store
        List<QueuedOperation> flushOperations = new ArrayList<QueuedOperation>();
        synchronized (this)
        {
            ListIterator<QueuedOperation> operIter = queuedOperations.listIterator();
            while (operIter.hasNext())
            {
                QueuedOperation oper = operIter.next();
                if (oper.getStore() == store)
                {
                    flushOperations.add(oper);
                    operIter.remove();
                }
            }
        }

        ListIterator<QueuedOperation> flushOperIter = flushOperations.listIterator();
        while (flushOperIter.hasNext())
        {
            QueuedOperation oper = flushOperIter.next();
            if (isAddFollowedByRemoveOnSameSCO(store, op, oper, flushOperIter))
            {
                // add+remove of the same element - ignore this and the next one
                flushOperIter.next();
            }
            else if (isRemoveFollowedByAddOnSameSCO(store, op, oper, flushOperIter))
            {
                // remove+add of the same element - ignore this and the next one
                flushOperIter.next();
            }
            else
            {
                oper.perform();
            }
        }
    }

    /**
     * Convenience optimisation checker to return if the current operation is ADD of an element that is
     * immediately REMOVED. Always leaves the iterator at the same position as starting
     * @param store The backing store
     * @param op The object provider
     * @param currentOper The current operation
     * @param listIter The iterator of operations
     * @return Whether this is an ADD that has a REMOVE of the same element immediately after
     */
    protected boolean isAddFollowedByRemoveOnSameSCO(Store store, ObjectProvider op, 
            QueuedOperation currentOper, ListIterator<QueuedOperation> listIter)
    {
        if (AddOperation.class.isInstance(currentOper))
        {
            // Optimisation to check that we aren't doing add+remove of the same element consecutively
            boolean addThenRemove = false;
            if (listIter.hasNext())
            {
                // Get next element
                QueuedOperation operNext = listIter.next();
                if (RemoveCollectionOperation.class.isInstance(operNext))
                {
                    Object value = AddOperation.class.cast(currentOper).getValue();
                    if (value == RemoveCollectionOperation.class.cast(operNext).getValue())
                    {
                        addThenRemove = true;
                        NucleusLogger.PERSISTENCE.info(
                            "Member " + store.getOwnerMemberMetaData().getFullFieldName() + 
                            " of " + StringUtils.toJVMIDString(op.getObject()) +
                            " had an add then a remove of element " + 
                            StringUtils.toJVMIDString(value) + " - operations ignored");
                    }
                }

                // Go back
                listIter.previous();
            }
            return addThenRemove;
        }
        return false;
    }

    /**
     * Convenience optimisation checker to return if the current operation is REMOVE of an element that is
     * immediately ADDed. Always leaves the iterator at the same position as starting
     * @param store The backing store
     * @param op The object provider
     * @param currentOper The current operation
     * @param listIter The iterator of operations
     * @return Whether this is a REMOVE that has an ADD of the same element immediately after
     */
    protected boolean isRemoveFollowedByAddOnSameSCO(Store store, ObjectProvider op, 
            QueuedOperation currentOper, ListIterator<QueuedOperation> listIter)
    {
        if (RemoveCollectionOperation.class.isInstance(currentOper))
        {
            // Optimisation to check that we aren't doing add+remove of the same element consecutively
            boolean removeThenAdd = false;
            if (listIter.hasNext())
            {
                // Get next element
                QueuedOperation opNext = listIter.next();
                if (AddOperation.class.isInstance(opNext))
                {
                    Object value = RemoveCollectionOperation.class.cast(currentOper).getValue();
                    if (value == AddOperation.class.cast(opNext).getValue())
                    {
                        removeThenAdd = true;
                        NucleusLogger.PERSISTENCE.info(
                            "Member" + store.getOwnerMemberMetaData().getFullFieldName() + 
                            " of " + StringUtils.toJVMIDString(op.getObject()) +
                            " had a remove then add of element " + 
                            StringUtils.toJVMIDString(value) + " - operations ignored");
                    }
                }

                if (!removeThenAdd)
                {
                    // Go back
                    listIter.previous();
                }
            }
            return removeThenAdd;
        }
        return false;
    }
}