/**********************************************************************
Copyright (c) 2006 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ClassNameConstants;
import org.datanucleus.ObjectManager;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.ScalarExpression;
import org.datanucleus.store.mapped.expression.StringLiteral;
import org.datanucleus.store.mapped.mapping.StringMapping;
import org.datanucleus.store.types.StringBufferStringConverter;

/**
 * Mapping for a StringBuffer type.
 * 
 * Note: A java.lang.StringBuffer is a final class and does not allow a
 * SCO implementation in order of implementing dirty detecting 
 */
public class StringBufferMapping extends StringMapping
{
    protected static StringBufferStringConverter converter = new StringBufferStringConverter();

    /**
     * Accessor for the name of the java-type actually used when mapping the particular datastore
     * field. This java-type must have an entry in the datastore mappings.
     * @param index requested datastore field index.
     * @return the name of java-type for the requested datastore field.
     */
    public String getJavaTypeForDatastoreMapping(int index)
    {
        // All of the types extending this class will be using java-type of String for the datastore
        return ClassNameConstants.JAVA_LANG_STRING;
    }

    /**
     * Delegates to StringMapping the storage with giving a String
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value)
    {
        Object v = converter.toString(value);
        super.setObject(om, preparedStatement, exprIndex, v);
    }

    /**
     * Delegates to StringMapping the retrieval of a String and constructs
     * a StringBuffer out of it
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex)
    {
        if (exprIndex == null)
        {
            return null;
        }
        Object value = getDatastoreMapping(0).getObject(resultSet, exprIndex[0]);
        if (value != null)
        {
            return converter.toObject((String)value);
        }
        return null;
    }
    
    public Class getJavaType()
    {
        return StringBuffer.class;
    }

    // TODO Delete when we drop legacy support
    protected static StringBuffer mappingSampleValue = new StringBuffer();
    public Object getSampleValue(ClassLoaderResolver clr)
    {
        return mappingSampleValue;
    }

    public ScalarExpression newLiteral(QueryExpression qs, Object value)
    {
        return new StringLiteral(qs, this, ((StringBuffer)value).toString());
    }
}