/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2003 Erik Bengtson - Important changes regarding the application identity
                     support when fetching PreparedStatements. Added an
                     inner class that should be moved away.
2004 Erik Bengtson - added commentary and Localisation
2004 Andy Jefferson - added capability to handle Abstract PC fields for
                      both SingleFieldIdentity and AID
2007 Andy Jefferson - implement RelationMappingCallbacks
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.Collection;

import javax.jdo.JDOObjectNotFoundException;
import javax.jdo.spi.PersistenceCapable;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.FetchPlan;
import org.datanucleus.OMFContext;
import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;
import org.datanucleus.api.ApiAdapter;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusObjectNotFoundException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.identity.OID;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.ClassMetaData;
import org.datanucleus.metadata.ColumnMetaData;
import org.datanucleus.metadata.FieldMetaData;
import org.datanucleus.metadata.FieldRole;
import org.datanucleus.metadata.IdentityStrategy;
import org.datanucleus.metadata.IdentityType;
import org.datanucleus.metadata.InheritanceStrategy;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.metadata.Relation;
import org.datanucleus.sco.SCOCollection;
import org.datanucleus.store.FieldValues;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.exceptions.NotYetFlushedException;
import org.datanucleus.store.exceptions.ReachableObjectNotCascadedException;
import org.datanucleus.store.fieldmanager.FieldManager;
import org.datanucleus.store.fieldmanager.SingleValueFieldManager;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.DatastoreElementContainer;
import org.datanucleus.store.mapped.DatastoreField;
import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.IdentifierType;
import org.datanucleus.store.mapped.MappedStoreManager;
import org.datanucleus.store.mapped.StatementClassMapping;
import org.datanucleus.store.mapped.StatementMappingIndex;
import org.datanucleus.store.mapped.expression.LogicSetExpression;
import org.datanucleus.store.mapped.expression.ObjectExpression;
import org.datanucleus.store.mapped.expression.ObjectLiteral;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.ScalarExpression;
import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.StringUtils;

/**
 * Maps a java field to a persistable class.
 * For persistable classes using datastore-id most of the behaviour is coded in the OIDMapping superclass.
 * TODO Split this from OIDMapping since PCMapping may represent app-id instead of datastore-id
 * TODO Move column creation for join table collection/map/array of PC to this class
 */
public class PersistableMapping extends OIDMapping implements MappingCallbacks
{
    /** Mappings for all fields necessary to represent the id of the PC object. */
    protected JavaTypeMapping[] javaTypeMappings = new JavaTypeMapping[0];

    //----------------------- convenience fields to improve performance ----------------------// 
    /** ClassMetaData for the represented class. Create a new one on each getObject invoke is expensive **/
    protected AbstractClassMetaData cmd;

    /** number of DatastoreField - convenience field to improve performance **/
    private int numberOfDatastoreFields = 0;

    /**
     * Create a new empty PersistenceCapableMapping.
     * The caller must call one of the initialize methods to initialize the instance with the
     * DatastoreAdapter and its type.
     */
    public PersistableMapping()
    {
    }

    /**
     * Initialize this JavaTypeMapping with the given DatastoreAdapter for the given metadata.
     * @param mmd MetaData for the field/property to be mapped (if any)
     * @param container The datastore container storing this mapping (if any)
     * @param clr the ClassLoaderResolver
     */
    public void initialize(AbstractMemberMetaData mmd, DatastoreContainerObject container, 
            ClassLoaderResolver clr)
    {
    	super.initialize(mmd, container, clr);

    	prepareDatastoreMapping(clr);
    }

    /**
     * Add a new JavaTypeMapping
     * @param mapping the JavaTypeMapping
     */
	public void addJavaTypeMapping(JavaTypeMapping mapping)
	{
	    if (mapping == null)
	    {
	        throw new NucleusException("mapping argument in PersistenceCapableMapping.addJavaTypeMapping is null").setFatal();
	    }
        JavaTypeMapping[] jtm = javaTypeMappings; 
        javaTypeMappings = new JavaTypeMapping[jtm.length+1]; 
        System.arraycopy(jtm, 0, javaTypeMappings, 0, jtm.length);
        javaTypeMappings[jtm.length] = mapping;
	}
    
    /**
     * Method to prepare the PC mapping and add its associated datastore mappings.
     */
    protected void prepareDatastoreMapping()
    {
        // Does nothing - added to prevent column creation by the method in OIDMapping
        // If we change the inheritance so this doesn't extend OIDMapping, then this can be deleted
    }

	/**
     * Method to prepare the PC mapping and add its associated datastore mappings.
     * @param clr The ClassLoaderResolver
     */
    protected void prepareDatastoreMapping(ClassLoaderResolver clr)
    {
        if (roleForMember == FieldRole.ROLE_COLLECTION_ELEMENT)
        {
            // TODO Handle creation of columns in join table for collection of PCs
        }
        else if (roleForMember == FieldRole.ROLE_ARRAY_ELEMENT)
        {
            // TODO Handle creation of columns in join table for array of PCs
        }
        else if (roleForMember == FieldRole.ROLE_MAP_KEY)
        {
            // TODO Handle creation of columns in join table for map of PCs as keys
        }
        else if (roleForMember == FieldRole.ROLE_MAP_VALUE)
        {
            // TODO Handle creation of columns in join table for map of PCs as values
        }
        else
        {
            // Either one end of a 1-1 relation, or the N end of a N-1
            AbstractClassMetaData refCmd = storeMgr.getOMFContext().getMetaDataManager().getMetaDataForClass(mmd.getType(), clr);
            JavaTypeMapping referenceMapping = null;
            if (refCmd.getInheritanceMetaData() != null &&
                    refCmd.getInheritanceMetaData().getStrategy() == InheritanceStrategy.SUBCLASS_TABLE)
            {
                // Find the actual tables storing the other end (can be multiple subclasses)
                AbstractClassMetaData[] cmds = storeMgr.getClassesManagingTableForClass(refCmd, clr);
                if (cmds != null && cmds.length > 0)
                {
                    if (cmds.length > 1)
                    {
                        NucleusLogger.PERSISTENCE.warn("Field " + mmd.getFullFieldName() + " represents either a 1-1 relation, " +
                            "or a N-1 relation where the other end uses \"subclass-table\" inheritance strategy and more " +
                        "than 1 subclasses with a table. This is not fully supported by JPOX");
                    }
                }
                else
                {
                    // No subclasses of the class using "subclasses-table" so no mapping!
                    // TODO Throw an exception ?
                    return;
                }
                // TODO We need a mapping for each of the possible subclass tables
                referenceMapping = storeMgr.getDatastoreClass(cmds[0].getFullClassName(), clr).getIdMapping();
            }
            else
            {
                referenceMapping = storeMgr.getDatastoreClass(mmd.getType().getName(), clr).getIdMapping();
            }

            // Generate a mapping from the columns of the referenced object to this mapping's ColumnMetaData
            CorrespondentColumnsMapper correspondentColumnsMapping = new CorrespondentColumnsMapper(mmd, referenceMapping, true);

            // Find any related field where this is part of a bidirectional relation
            int relationType = mmd.getRelationType(clr);
            boolean createDatastoreMappings = true;
            if (relationType == Relation.MANY_TO_ONE_BI)
            {
                AbstractMemberMetaData[] relatedMmds = mmd.getRelatedMemberMetaData(clr);
                // TODO Cater for more than 1 related field
                createDatastoreMappings = (relatedMmds[0].getJoinMetaData() == null);
            }
            else if (relationType == Relation.ONE_TO_ONE_BI)
            {
                // Put the FK at the end without "mapped-by"
                createDatastoreMappings = (mmd.getMappedBy() == null);
            }

            // Loop through the datastore fields in the referenced class and create a datastore field for each
            for (int i=0; i<referenceMapping.getNumberOfDatastoreMappings(); i++)
            {
                DatastoreMapping refDatastoreMapping = referenceMapping.getDatastoreMapping(i);
                JavaTypeMapping mapping = storeMgr.getMappingManager().getMapping(refDatastoreMapping.getJavaTypeMapping().getJavaType());
                this.addJavaTypeMapping(mapping);

                // Create physical datastore columns where we require a FK link to the related table.
                if (createDatastoreMappings)
                {
                    // Find the Column MetaData that maps to the referenced datastore field
                    ColumnMetaData colmd = correspondentColumnsMapping.getColumnMetaDataByIdentifier(
                        refDatastoreMapping.getDatastoreField().getIdentifier());
                    if (colmd == null)
                    {
                        throw new NucleusUserException(LOCALISER.msg("041038",
                            refDatastoreMapping.getDatastoreField().getIdentifier(), toString())).setFatal();
                    }

                    // Create a Datastore field to equate to the referenced classes datastore field
                    MappingManager mmgr = storeMgr.getMappingManager();
                    DatastoreField col = mmgr.createDatastoreField(mmd, datastoreContainer, mapping, 
                        colmd, refDatastoreMapping.getDatastoreField(), clr);

                    // Add its datastore mapping
                    DatastoreMapping datastoreMapping = mmgr.createDatastoreMapping(mapping, col, refDatastoreMapping.getJavaTypeMapping().getJavaTypeForDatastoreMapping(i));
                    this.addDatastoreMapping(datastoreMapping);
                }
                else
                {
                    mapping.setReferenceMapping(referenceMapping);
                }
            }
        }
    }

    /**
     * Accessor for the Java type mappings
     * @return The Java type mappings
     */
    public JavaTypeMapping[] getJavaTypeMapping()
    {
        return javaTypeMappings;
    }

    /**
     * Accessor for the number of datastore fields.
     * Zero datastore fields implies that the mapping uses a FK in the associated class
     * to reference back.
     * @return Number of datastore fields for this PC mapping.
     */
    public int getNumberOfDatastoreMappings()
    {
        if (numberOfDatastoreFields == 0)
        {
            for (int i=0; i<javaTypeMappings.length; i++)
            {
                numberOfDatastoreFields += javaTypeMappings[i].getNumberOfDatastoreMappings();
            }
        }
        return numberOfDatastoreFields;
    }

	/**
     * Accessor for a datastore mapping.
     * This method works through the java type mappings, and for each mapping through its datastore mappings
     * incrementing the index with each datastore mapping.
     * @param index The position of the mapping.
     * @return The datastore mapping.
	 */
	public DatastoreMapping getDatastoreMapping(int index)
	{
        int currentIndex = 0;
        int numberJavaMappings = javaTypeMappings.length;
        for (int i=0; i<numberJavaMappings; i++)
        {
            int numberDatastoreMappings = javaTypeMappings[i].getNumberOfDatastoreMappings();
            for (int j=0; j<numberDatastoreMappings; j++)
            {
                if (currentIndex == index)
                {
                    return javaTypeMappings[i].getDatastoreMapping(j);
                }
                currentIndex++;
            }
        }
        // TODO Localise this message
        throw new NucleusException("Invalid index " + index + " for DataStoreMapping.").setFatal();
	}

    /**
     * Accessor for the datastore mappings for this java type.
     * Overrides the method in JavaTypeMapping so we can add on datastore mappings of any sub mappings here.
     * @return The datastore mapping(s)
     */
    public DatastoreMapping[] getDatastoreMappings()
    {
        if (datastoreMappings.length == 0)
        {
            datastoreMappings = new DatastoreMapping[getNumberOfDatastoreMappings()];
            int currentIndex = 0;
            int numberJavaMappings = javaTypeMappings.length;
            for (int i=0; i<numberJavaMappings; i++)
            {
                int numberDatastoreMappings = javaTypeMappings[i].getNumberOfDatastoreMappings();
                for (int j=0; j<numberDatastoreMappings; j++)
                {
                    datastoreMappings[currentIndex++] = javaTypeMappings[i].getDatastoreMapping(j);
                }
            }
        }
        return super.getDatastoreMappings();
    }

    /**
     * Method to return the value to be stored in the specified datastore index given the overall
     * value for this java type.
     * @param omfCtx OMF Context
     * @param index The datastore index
     * @param value The overall value for this java type
     * @return The value for this datastore index
     */
    public Object getValueForDatastoreMapping(OMFContext omfCtx, int index, Object value)
    {
        ObjectManager om = omfCtx.getApiAdapter().getObjectManager(value);
        if (cmd == null)
        {
            cmd = omfCtx.getMetaDataManager().getMetaDataForClass(getType(), 
                om != null ? om.getClassLoaderResolver() : omfCtx.getClassLoaderResolver(null));
        }

        if (cmd.getIdentityType() == IdentityType.APPLICATION)
        {
            AbstractMemberMetaData mmd =
                cmd.getMetaDataForManagedMemberAtAbsolutePosition(cmd.getPKMemberPositions()[index]);
            StateManager sm = null;
            if (om != null)
            {
                sm = om.findStateManager(value);
            }

            if (sm == null)
            {
                // Transient or detached maybe, so use reflection to get PK field values
                if (mmd instanceof FieldMetaData)
                {
                    return ClassUtils.getValueOfFieldByReflection(value, mmd.getName());
                }
                else
                {
                    return ClassUtils.getValueOfMethodByReflection(value,
                        ClassUtils.getJavaBeanGetterName(mmd.getName(), false), null);
                }
            }

            if (!mmd.isPrimaryKey())
            {
                // Make sure the field is loaded
                omfCtx.getApiAdapter().isLoaded(sm, mmd.getAbsoluteFieldNumber());
            }
            FieldManager fm = new SingleValueFieldManager();
            sm.provideFields(new int[] {mmd.getAbsoluteFieldNumber()}, fm);
            return fm.fetchObjectField(mmd.getAbsoluteFieldNumber());
        }
        else if (cmd.getIdentityType() == IdentityType.DATASTORE)
        {
            OID oid = (OID)omfCtx.getApiAdapter().getIdForObject(value);
            return oid != null ? oid.getKeyValue() : null;
        }
        return null;
    }

    /**
	 * Method to set an object in the datastore.
	 * @param om The ObjectManager
	 * @param ps The Prepared Statement
	 * @param param The parameter ids in the statement
	 * @param value The value to put in the statement at these ids
	 * @throws NotYetFlushedException
	 */
    public void setObject(ObjectManager om, Object ps, int[] param, Object value)
    {
        setObject(om, ps, param, value, null, -1);
    }

    /**
     * Method to set an object reference (FK) in the datastore.
     * @param om The Object Manager
     * @param ps The Prepared Statement
     * @param param The parameter ids in the statement
     * @param value The value to put in the statement at these ids
     * @param ownerSM StateManager for the owner object
     * @param ownerFieldNumber Field number of this PC object in the owner
     * @throws NotYetFlushedException
     */
    public void setObject(ObjectManager om, Object ps, int[] param, Object value, StateManager ownerSM, int ownerFieldNumber)
    {
        if (value == null)
        {
            setObjectAsNull(om, ps, param);
        }
        else
        {
            setObjectAsValue(om, ps, param, value, ownerSM, ownerFieldNumber);
        }
    }

    /**
     * Populates the PreparedStatement with a null value for the mappings of this PersistenceCapableMapping.
     * @param om the Object Manager
     * @param ps the Prepared Statement
     * @param param The parameter ids in the statement
     */
    private void setObjectAsNull(ObjectManager om, Object ps, int[] param)
    {
        // Null out the PC object
        int n=0;
        for (int i=0; i<javaTypeMappings.length; i++)
        {
            JavaTypeMapping mapping = javaTypeMappings[i];
            if (mapping.getNumberOfDatastoreMappings() > 0)
            {
                // Only populate the PreparedStatement for the object if it has any datastore mappings
                int[] posMapping = new int[mapping.getNumberOfDatastoreMappings()];
                for (int j=0; j<posMapping.length; j++)
                {
                    posMapping[j] = param[n++];
                }
                mapping.setObject(om, ps, posMapping, null);
            }
        }
    }

    /**
     * Check if one of the primary key fields of the PC has value attributed by the datastore
     * @param mdm the {@link MetaDataManager}
     * @param srm the {@link StoreManager}
     * @param clr the {@link ClassLoaderResolver}
     * @return true if one of the primary key fields of the PC has value attributed by the datastore
     */
    private boolean hasDatastoreAttributedPrimaryKeyValues(MetaDataManager mdm, StoreManager srm, ClassLoaderResolver clr)
    {
        boolean hasDatastoreAttributedPrimaryKeyValues = false;
        if (this.mmd != null)
        {
            if (roleForMember != FieldRole.ROLE_ARRAY_ELEMENT &&
                roleForMember != FieldRole.ROLE_COLLECTION_ELEMENT &&
                roleForMember != FieldRole.ROLE_MAP_KEY &&
                roleForMember != FieldRole.ROLE_MAP_VALUE)
            {
                // Object is associated to a field (i.e not a join table)
                AbstractClassMetaData acmd = mdm.getMetaDataForClass(this.mmd.getType(), clr);
                if (acmd.getIdentityType() == IdentityType.APPLICATION)
                {
                    for (int i=0; i<acmd.getPKMemberPositions().length; i++)
                    {
                        IdentityStrategy strategy = 
                            acmd.getMetaDataForManagedMemberAtAbsolutePosition(acmd.getPKMemberPositions()[i]).getValueStrategy();
                        if (strategy != null)
                        {
                            //if strategy is null, then it's user attributed value
                            hasDatastoreAttributedPrimaryKeyValues |= srm.isStrategyDatastoreAttributed(strategy, false);
                        }
                    }
                }
            }
        }
        return hasDatastoreAttributedPrimaryKeyValues;
    }
    
    /**
     * Method to set an object reference (FK) in the datastore.
     * @param om The Object Manager
     * @param ps The Prepared Statement
     * @param param The parameter ids in the statement
     * @param value The value to put in the statement at these ids
     * @param ownerSM StateManager for the owner object
     * @param ownerFieldNumber Field number of this PC object in the owner
     * @throws NotYetFlushedException Just put "null" in and throw "NotYetFlushedException", 
     *                                to be caught by ParameterSetter and will signal to the PC object being inserted
     *                                that it needs to inform this object when it is inserted.
     */
    private void setObjectAsValue(ObjectManager om, Object ps, int[] param, Object value, StateManager ownerSM, 
            int ownerFieldNumber)
    {
        Object id;

        ApiAdapter api = om.getApiAdapter();
        if (!api.isPersistable(value))
        {
            throw new NucleusException(LOCALISER.msg("041016", value.getClass(), value)).setFatal();
        }

        StateManager sm = om.findStateManager(value);

        try
        {
            ClassLoaderResolver clr = om.getClassLoaderResolver();
            MappedStoreManager storeMgr = (MappedStoreManager)om.getStoreManager();

            // Check if the field is attributed in the datastore
            boolean hasDatastoreAttributedPrimaryKeyValues = hasDatastoreAttributedPrimaryKeyValues(
                om.getMetaDataManager(), storeMgr, clr);

            boolean inserted = false;
            if (ownerFieldNumber >= 0)
            {
                // Field mapping : is this field of the related object present in the datastore?
                inserted = storeMgr.isObjectInserted(sm, ownerFieldNumber);
            }
            else if (mmd == null)
            {
                // Identity mapping : is the object inserted far enough to be considered of this mapping type?
                inserted = storeMgr.isObjectInserted(sm, type);
            }

            if (sm != null)
            {
                if (om.getApiAdapter().isDetached(value) && sm.getReferencedPC() != null && ownerSM != null && mmd != null)
                {
                    // Still detached but started attaching so replace the field with what will be the attached
                    // Note that we have "fmd != null" here hence omitting any M-N relations where this is a join table 
                    // mapping
                    ownerSM.replaceField(ownerFieldNumber, sm.getReferencedPC(), true);
                }

                if (sm.isWaitingToBeFlushedToDatastore())
                {
                    // Related object is not yet flushed to the datastore so flush it so we can set the FK
                    sm.flush();
                }
            }
            else
            {
                if (om.getApiAdapter().isDetached(value))
                {
                    // Field value is detached and not yet started attaching, so attach
                    Object attachedValue = om.persistObjectInternal(value, null, null, -1, StateManager.PC);
                    if (attachedValue != value && ownerSM != null)
                    {
                        // Replace the field value if using copy-on-attach
                        ownerSM.replaceField(ownerFieldNumber, attachedValue, true);
                    }
                }
            }

            // we can execute this block when
            // 1) the pc has been inserted; OR
            // 2) is not in process of being inserted; OR
            // 3) is being inserted yet is inserted enough to use this mapping; OR
            // 4) the PC PK values are not attributed by the database and this mapping is for a PK field (compound identity)
            // 5) the value is the same object as we are inserting anyway and has its identity set
            if (inserted || !om.isInserting(value) ||
                (!hasDatastoreAttributedPrimaryKeyValues && (this.mmd != null && this.mmd.isPrimaryKey())) ||
                (!hasDatastoreAttributedPrimaryKeyValues && ownerSM == sm && api.getIdForObject(value) != null))
            {
                // The PC is either already inserted, or inserted down to the level we need, or not inserted at all,
                // or the field is a PK and identity not attributed by the datastore

                // Object either already exists, or is not yet being inserted.
                id = api.getIdForObject(value);

                // Check if the PersistenceCapable exists in this datastore
                boolean requiresPersisting = false;
                if (om.getApiAdapter().isDetached(value) && ownerSM != null)
                {
                    // Detached object so needs attaching
                    if (ownerSM.isInserting())
                    {
                        // Inserting other object, and this object is detached but if detached from this datastore
                        // we can just return the value now and attach later (in InsertRequest)
                        if (!om.getOMFContext().getPersistenceConfiguration().getBooleanProperty("datanucleus.attachSameDatastore"))
                        {
                            if (om.getObjectFromCache(api.getIdForObject(value)) != null)
                            {
                                // Object is in cache so exists for this datastore, so no point checking
                            }
                            else
                            {
                                try
                                {
                                    Object obj = om.findObject(api.getIdForObject(value), true, false,
                                        value.getClass().getName());
                                    if (obj != null)
                                    {
                                        // Make sure this object is not retained in cache etc
                                        StateManager objSM = om.findStateManager(obj);
                                        if (objSM != null)
                                        {
                                            om.evictFromTransaction(objSM);
                                        }
                                        om.removeObjectFromCache(value, api.getIdForObject(value));
                                    }
                                }
                                catch (NucleusObjectNotFoundException onfe)
                                {
                                    // Object doesnt yet exist
                                    requiresPersisting = true;
                                }
                            }
                        }
                    }
                    else
                    {
                        requiresPersisting = true;
                    }
                }
                else if (id == null)
                {
                    // Transient object, so we need to persist it
                    requiresPersisting = true;
                }
                else
                {
                    ObjectManager pcPM = om.getApiAdapter().getObjectManager(value);
                    if (pcPM != null && om != pcPM)
                    {
                        throw new NucleusUserException(LOCALISER.msg("041015"), id);
                    }
                }

                if (requiresPersisting)
                {
                    // PERSISTENCE-BY-REACHABILITY
                    // This PC object needs persisting (new or detached) to do the "set"
                    if (mmd != null && !mmd.isCascadePersist() && !om.getApiAdapter().isDetached(value))
                    {
                        // Related PC object not persistent, but cant do cascade-persist so throw exception
                        if (NucleusLogger.REACHABILITY.isDebugEnabled())
                        {
                            NucleusLogger.REACHABILITY.debug(LOCALISER.msg("007006", 
                                mmd.getFullFieldName()));
                        }
                        throw new ReachableObjectNotCascadedException(mmd.getFullFieldName(), value);
                    }

                    if (NucleusLogger.REACHABILITY.isDebugEnabled())
                    {
                        NucleusLogger.REACHABILITY.debug(LOCALISER.msg("007007", 
                            mmd != null ? mmd.getFullFieldName() : null));
                    }

                    try
                    {
                        Object pcNew = om.persistObjectInternal(value, null, null, -1, StateManager.PC);
                        if (hasDatastoreAttributedPrimaryKeyValues)
                        {
                            om.flushInternal(false);
                        }
                        id = api.getIdForObject(pcNew);
                        if (om.getApiAdapter().isDetached(value) && ownerSM != null)
                        {
                            // Update any detached reference to refer to the attached variant
                            ownerSM.replaceField(ownerFieldNumber, pcNew, true);
                            int relationType = mmd.getRelationType(clr);
                            if (relationType == Relation.MANY_TO_ONE_BI)
                            {
                                // TODO Update the container to refer to the attached object
                                if (NucleusLogger.PERSISTENCE.isInfoEnabled())
                                {
                                    NucleusLogger.PERSISTENCE.info("PCMapping.setObject : object " + ownerSM.getInternalObjectId() + 
                                        " has field " + ownerFieldNumber + " that is 1-N bidirectional." + 
                                        " Have just attached the N side so should really update the reference in the 1 side collection" +
                                        " to refer to this attached object. Not yet implemented");
                                }
                            }
                            else if (relationType == Relation.ONE_TO_ONE_BI)
                            {
                                AbstractMemberMetaData[] relatedMmds = mmd.getRelatedMemberMetaData(clr);
                                // TODO Cater for more than 1 related field
                                StateManager relatedSM = om.findStateManager(pcNew);
                                relatedSM.replaceField(relatedMmds[0].getAbsoluteFieldNumber(), ownerSM.getObject(), true);
                            }
                        }
                    }
                    catch (NotYetFlushedException e)
                    {
                        setObjectAsNull(om, ps, param);
                        throw new NotYetFlushedException(value);
                    }
                }

                if (sm != null)
                {
                    sm.setStoringPC();
                }

                // If the field doesn't map to any datastore fields, omit the set process
                if (getNumberOfDatastoreMappings() > 0)
                {
                    if (id instanceof OID)
                    {
                        super.setObject(om, ps, param, id);
                    }
                    else
                    {
                        // TODO Factor out this PersistenceCapable reference
                        ((PersistenceCapable)value).jdoCopyKeyFieldsFromObjectId(
                            new AppIDObjectIdFieldConsumer(param, om, ps, javaTypeMappings), id);
                    }
                }
            }
            else
            {
                if (sm != null)
                {
                    sm.setStoringPC();
                }

                if (getNumberOfDatastoreMappings() > 0)
                {
                    // Object is in the process of being inserted so we cant use its id currently and we need to store
                    // a foreign key to it (which we cant yet do). Just put "null" in and throw "NotYetFlushedException",
                    // to be caught by ParameterSetter and will signal to the PC object being inserted that it needs 
                    // to inform this object when it is inserted.
                    setObjectAsNull(om, ps, param);
                    throw new NotYetFlushedException(value);
                }
            }
        }
        finally
        {
            if (sm != null)
            {
                sm.unsetStoringPC();
            }
        }
    }

    /**
     * Returns a instance of a PersistenceCapable class.
     * Processes a FK field and converts the id stored firstly into an OID/AID
     * and then into the object that the FK id relates to.
     * @param om The Object Manager
     * @param rs The ResultSet
     * @param param Array of parameter ids in the ResultSet to retrieve
     * @return The Persistence Capable object
     */
    public Object getObject(ObjectManager om, final Object rs, int[] param)
    {
        // Check for null FK
        MappedStoreManager storeMgr = (MappedStoreManager)om.getStoreManager();
        if (storeMgr.getResultValueAtPosition(rs, this, param[0]) == null)
        {
            // if the first param is null, then the field is null
            return null;
        }

        if (cmd == null)
        {
            cmd = om.getMetaDataManager().getMetaDataForClass(getType(),om.getClassLoaderResolver());
        }

        // Return the object represented by this mapping
        if (cmd.getIdentityType() == IdentityType.DATASTORE)
        {
            return getObjectForDatastoreIdentity(om,rs,param,cmd);
        }
        else if (cmd.getIdentityType() == IdentityType.APPLICATION)
        {
            return getObjectForApplicationIdentity(om,rs,param,cmd);
        }
        else
        {
            return null;
        }
    }

    // ---------------------------- JDOQL Query Methods --------------------------------------

    public ScalarExpression newLiteral(QueryExpression qs, Object value)
    {
        ScalarExpression expr = new ObjectLiteral(qs, this, value,getType());
        return expr;        
    }

    public ScalarExpression newScalarExpression(QueryExpression qs, LogicSetExpression te)
    {
        if (getNumberOfDatastoreMappings() > 0)
        {
            return new ObjectExpression(qs, this, te);
        }
        else
        {
            ClassLoaderResolver clr = qs.getClassLoaderResolver();
            MappedStoreManager srm = qs.getStoreManager();
            int relationType = mmd.getRelationType(clr);
            if (relationType == Relation.ONE_TO_ONE_BI)
            {
                // Create an expression joining to the related field in the related table
                DatastoreClass targetTable = srm.getDatastoreClass(mmd.getTypeName(), clr);
                AbstractMemberMetaData[] relatedMmds = mmd.getRelatedMemberMetaData(clr);
                // TODO Cater for more than one related field
                JavaTypeMapping refMapping = targetTable.getMemberMapping(relatedMmds[0]);
                JavaTypeMapping selectMapping = targetTable.getIdMapping();
                DatastoreIdentifier targetTableIdentifier =
                    srm.getIdentifierFactory().newIdentifier(IdentifierType.TABLE, "RELATED" + mmd.getAbsoluteFieldNumber());
                LogicSetExpression targetTe = qs.newTableExpression(targetTable, targetTableIdentifier);
                return new ObjectExpression(qs, this, te, refMapping, targetTe, selectMapping);
            }
            else if (relationType == Relation.MANY_TO_ONE_BI)
            {
                AbstractMemberMetaData[] relatedMmds = mmd.getRelatedMemberMetaData(clr);
                // TODO Cater for more than one related field
                if (mmd.getJoinMetaData() != null || relatedMmds[0].getJoinMetaData() != null)
                {
                    // Join table relation - only allows for Collection/Array
                    // Create an expression this table to the join table on the element id, selecting the owner id
                    DatastoreContainerObject targetTable = srm.getDatastoreContainerObject(relatedMmds[0]);
                    JavaTypeMapping refMapping = null;
                    JavaTypeMapping selectMapping = null;
                    DatastoreElementContainer elementTable = (DatastoreElementContainer)targetTable;
                    refMapping = elementTable.getElementMapping();
                    selectMapping = elementTable.getOwnerMapping();
                    DatastoreIdentifier targetTableIdentifier =
                        srm.getIdentifierFactory().newIdentifier(IdentifierType.TABLE, "JOINTABLE" + mmd.getAbsoluteFieldNumber());
                    LogicSetExpression targetTe = qs.newTableExpression(targetTable, targetTableIdentifier);
                    return new ObjectExpression(qs, this, te, refMapping, targetTe, selectMapping);
                }
            }
        }

        // TODO Throw an exception since should be impossible
        return null;
    }

    // ------------------------------------- Utility Methods ------------------------------------------

    /**
     * Get the object instance for a class using datastore identity
     * @param om the ObjectManager
     * @param rs the ResultSet
     * @param param the parameters
     * @param cmd the AbstractClassMetaData
     * @return the id
     */
    private Object getObjectForDatastoreIdentity(ObjectManager om, final Object rs, int[] param, AbstractClassMetaData cmd)
    {
        // Datastore Identity - retrieve the OID for the class.
        // Note that this is a temporary OID that is simply formed from the type of base class in the relationship
        // and the id stored in the FK. The real OID for the object may be of a different class.
        // For that reason we get the object by checking the inheritance (final param in getObjectById())
        Object oid = super.getObject(om, rs, param);
        ApiAdapter api = om.getApiAdapter();
        if (api.isPersistable(oid)) //why check this?
        {
        	return oid;
        }
        return oid == null ? null : om.findObject(oid, false, true, null);
    }

    /**
     * Create a SingleFieldIdentity instance
     * @param om the ObjectManager
     * @param rs the ResultSet
     * @param param the parameters
     * @param cmd the AbstractClassMetaData
     * @param objectIdClass the object id class
     * @param pcClass the PersistenceCapable class
     * @return the id
     */
    private Object createSingleFieldIdentity(ObjectManager om, final Object rs, int[] param, AbstractClassMetaData cmd, 
            Class objectIdClass, Class pcClass)
    {
        // SingleFieldIdentity
        int paramNumber = param[0];
        try
        {
            MappedStoreManager storeMgr = (MappedStoreManager)om.getStoreManager();
            Object idObj = storeMgr.getResultValueAtPosition(rs, this, paramNumber);
            if (idObj == null)
            {
                throw new NucleusException(LOCALISER.msg("041039")).setFatal();
            }
            else
            {
                // Make sure the key type is correct for the type of SingleFieldIdentity
                Class keyType = om.getApiAdapter().getKeyTypeForSingleFieldIdentityType(objectIdClass);
                idObj = ClassUtils.convertValue(idObj, keyType);
            }
            return om.getApiAdapter().getNewSingleFieldIdentity(objectIdClass, pcClass, idObj);
        }
        catch (Exception e)
        {
            NucleusLogger.PERSISTENCE.error(LOCALISER.msg("041036", cmd.getObjectidClass(), 
                e));
            return null;
        }
    }

    /**
     * Create an object id instance and fill the fields using reflection
     * @param om the ObjectManager
     * @param rs the ResultSet
     * @param param the parameters
     * @param cmd the AbstractClassMetaData
     * @param objectIdClass the object id class
     * @return the id
     */
    private Object createObjectIdInstanceReflection(ObjectManager om, final Object rs, int[] param, 
            AbstractClassMetaData cmd, Class objectIdClass)
    {
        // Users own AID
        Object fieldValue = null;
        try
        {
            // Create an AID
            Object id = objectIdClass.newInstance();

            // Set the fields of the AID
            int paramIndex = 0;
            for (int i=0; i<cmd.getPKMemberPositions().length; ++i)
            {
                AbstractMemberMetaData fmd = cmd.getMetaDataForManagedMemberAtAbsolutePosition(cmd.getPKMemberPositions()[i]);
                Field field = objectIdClass.getField(fmd.getName());

                MappedStoreManager storeMgr = (MappedStoreManager)om.getStoreManager();
                JavaTypeMapping m = storeMgr.getDatastoreClass(cmd.getFullClassName(), om.getClassLoaderResolver()).getMemberMapping(fmd);
                // NOTE This assumes that each field has one datastore column.
                for (int j = 0; j < m.getNumberOfDatastoreMappings(); j++)
                {
                    Object obj = storeMgr.getResultValueAtPosition(rs, this, param[paramIndex++]);
                    if ((obj instanceof BigDecimal))
                    {
                        BigDecimal bigDecimal = (BigDecimal) obj;
                        // Oracle 10g returns BigDecimal for NUMBER columns,
                        // resulting in IllegalArgumentException when reflective 
                        // setter is invoked for incompatible field type
                        // (see http://www.jpox.org/servlet/jira/browse/CORE-2624)
                        Class keyType = om.getApiAdapter().getKeyTypeForSingleFieldIdentityType(field.getType());
                        obj = ClassUtils.convertValue(bigDecimal, keyType);
                        if (!bigDecimal.subtract(new BigDecimal("" + obj)).equals(new BigDecimal("0")))
                        {
                            throw new NucleusException("Cannot convert retrieved BigInteger value to field of object id class!").setFatal();
                        }
                    }
                    // field with multiple columns should have values returned from db merged here
                    fieldValue = obj;
                }
                field.set(id, fieldValue);
            }
            return id;
        }
        catch (Exception e)
        {
            NucleusLogger.PERSISTENCE.error(LOCALISER.msg("041037",
                cmd.getObjectidClass(), mmd == null ? null : mmd.getName(), fieldValue, e));
            return null;
        }
    }

    /**
     * Create an object id instance and fill the fields using reflection
     * @param om the ObjectManager
     * @param rs the ResultSet
     * @param param the parameters
     * @param cmd the AbstractClassMetaData
     * @return the id
     */
    private Object getObjectForAbstractClass(ObjectManager om, final Object rs, int[] param, AbstractClassMetaData cmd)
    {
        ClassLoaderResolver clr = om.getClassLoaderResolver();

        // Abstract class, so we need to generate an AID before proceeding
        Class objectIdClass = clr.classForName(cmd.getObjectidClass());
        Class pcClass = clr.classForName(cmd.getFullClassName());
        Object id;
        if (cmd.usesSingleFieldIdentityClass())
        {
            id = createSingleFieldIdentity(om, rs, param, cmd, objectIdClass, pcClass); 
        }
        else
        {
            id = createObjectIdInstanceReflection(om, rs, param, cmd, objectIdClass); 
        }
        return om.findObject(id, false, true, null);
    }

    /**
     * Get the object instance for a class using application identity
     * @param om the ObjectManager
     * @param rs the ResultSet
     * @param param the parameters
     * @param cmd the AbstractClassMetaData
     * @return the id
     */
    private Object getObjectForApplicationIdentity(ObjectManager om, final Object rs, int[] param, AbstractClassMetaData cmd)
    {
        ClassLoaderResolver clr = om.getClassLoaderResolver();

        // Abstract class
        if (((ClassMetaData)cmd).isAbstract() && cmd.getObjectidClass() != null)
        {
            return getObjectForAbstractClass(om,rs,param,cmd);
        }

        int totalFieldCount = cmd.getNoOfManagedMembers() + cmd.getNoOfInheritedManagedMembers();
        final StatementMappingIndex[] statementExpressionIndex = new StatementMappingIndex[totalFieldCount];
        int paramIndex = 0;

        final MappedStoreManager storeMgr = (MappedStoreManager)om.getStoreManager();
        DatastoreClass datastoreClass = storeMgr.getDatastoreClass(cmd.getFullClassName(), clr);
        final int[] pkFieldNumbers = cmd.getPKMemberPositions();

        for (int i=0; i<pkFieldNumbers.length; ++i)
        {
            AbstractMemberMetaData fmd = cmd.getMetaDataForManagedMemberAtAbsolutePosition(pkFieldNumbers[i]);
            JavaTypeMapping m = datastoreClass.getMemberMapping(fmd);
            statementExpressionIndex[fmd.getAbsoluteFieldNumber()] = new StatementMappingIndex(m);
            int expressionsIndex[] = new int[m.getNumberOfDatastoreMappings()];
            for (int j = 0; j < expressionsIndex.length; j++)
            {
                expressionsIndex[j] = param[paramIndex++];
            }
            statementExpressionIndex[fmd.getAbsoluteFieldNumber()].setColumnPositions(expressionsIndex);
        }

        final StatementClassMapping resultMappings = new StatementClassMapping();
        for (int i=0;i<pkFieldNumbers.length;i++)
        {
            resultMappings.addMappingForMember(pkFieldNumbers[i], statementExpressionIndex[pkFieldNumbers[i]]);
        }
        return om.findObjectUsingAID(clr.classForName(cmd.getFullClassName()),
            new FieldValues()
            {
                // StateManager calls the fetchFields method
                public void fetchFields(StateManager sm)
                {
                    sm.replaceFields(pkFieldNumbers, 
                        storeMgr.getFieldManagerForResultProcessing(sm, rs, resultMappings));
                }
                public void fetchNonLoadedFields(StateManager sm)
                {
                    sm.replaceNonLoadedFields(pkFieldNumbers, 
                        storeMgr.getFieldManagerForResultProcessing(sm, rs, resultMappings));
                }
                public FetchPlan getFetchPlanForLoading()
                {
                    return null;
                }
            }, false, true);
    }

    // ----------------------- Implementation of MappingCallbacks --------------------------

    /**
     * Method executed just after a fetch of the owning object, allowing any necessary action
     * to this field and the object stored in it.
     * @param sm StateManager for the owner.
     */
    public void postFetch(StateManager sm)
    {
    }

    /**
     * Method executed just after the insert of the owning object, allowing any necessary action
     * to this field and the object stored in it.
     * @param sm StateManager for the owner
     */
    public void postInsert(StateManager sm)
    {
        Object pc = sm.provideField(mmd.getAbsoluteFieldNumber());
        if (pc == null)
        {
            // Has been set to null so nothing to do
            return;
        }

        ClassLoaderResolver clr = sm.getObjectManager().getClassLoaderResolver();
        AbstractMemberMetaData[] relatedMmds = mmd.getRelatedMemberMetaData(clr);
        int relationType = mmd.getRelationType(clr);
        if (pc != null)
        {
            if (relationType == Relation.ONE_TO_ONE_BI)
            {
                StateManager otherSM = sm.getObjectManager().findStateManager(pc);
                AbstractMemberMetaData relatedMmd = mmd.getRelatedMemberMetaDataForObject(clr, sm.getObject(), pc);
                Object relatedValue = otherSM.provideField(relatedMmd.getAbsoluteFieldNumber());
                if (relatedValue == null)
                {
                    // Managed Relations : Other side not set so update it in memory
                    if (NucleusLogger.PERSISTENCE.isDebugEnabled())
                    {
                        NucleusLogger.PERSISTENCE.debug(LOCALISER.msg("041018",
                            StringUtils.toJVMIDString(sm.getObject()), mmd.getFullFieldName(),
                            StringUtils.toJVMIDString(pc), relatedMmd.getFullFieldName()));
                    }
                    otherSM.replaceField(relatedMmd.getAbsoluteFieldNumber(), sm.getObject(), false);
                }
                else if (relatedValue != sm.getObject())
                {
                    // Managed Relations : Other side is inconsistent so throw exception
                    throw new NucleusUserException(
                        LOCALISER.msg("041020",
                            StringUtils.toJVMIDString(sm.getObject()), mmd.getFullFieldName(),
                            StringUtils.toJVMIDString(pc),
                            StringUtils.toJVMIDString(relatedValue)));
                }
            }
            else if (relationType == Relation.MANY_TO_ONE_BI && relatedMmds[0].hasCollection())
            {
                // TODO Make sure we have this PC in the collection at the other side
                StateManager otherSM = sm.getObjectManager().findStateManager(pc);
                if (otherSM != null)
                {
                    // Managed Relations : add to the collection on the other side
                    Collection relatedColl = (Collection)otherSM.provideField(relatedMmds[0].getAbsoluteFieldNumber());
                    if (relatedColl != null && !(relatedColl instanceof SCOCollection))
                    {
                        // TODO Make sure the collection is a wrapper
                        boolean contained = relatedColl.contains(sm.getObject());
                        if (!contained)
                        {
                            NucleusLogger.PERSISTENCE.info(
                                LOCALISER.msg("041022",
                                StringUtils.toJVMIDString(sm.getObject()), mmd.getFullFieldName(),
                                StringUtils.toJVMIDString(pc), relatedMmds[0].getFullFieldName()));
                            // TODO Enable this. CUrrently causes issues with
                            // PMImplTest, InheritanceStrategyTest, TCK "inheritance1.conf"
                            /*relatedColl.add(sm.getObject());*/
                        }
                    }
                }
            }
        }
    }

    /**
     * Method executed just afer any update of the owning object, allowing any necessary action
     * to this field and the object stored in it.
     * @param sm StateManager for the owner
     */
    public void postUpdate(StateManager sm)
    {
        Object pc = sm.provideField(mmd.getAbsoluteFieldNumber());
        if (pc == null)
        {
            // Has been set to null so nothing to do
            return;
        }

        if (pc != null)
        {
            StateManager otherSM = sm.getObjectManager().findStateManager(pc);
            if (otherSM == null)
            {
                ClassLoaderResolver clr = sm.getObjectManager().getClassLoaderResolver();
                int relationType = mmd.getRelationType(clr);
                if (relationType == Relation.ONE_TO_ONE_BI || relationType == Relation.MANY_TO_ONE_BI)
                {
                    // Related object is not yet persisted (e.g 1-1 with FK at other side) so persist it
                    sm.getObjectManager().persistObjectInternal(pc, null, null, -1, StateManager.PC);
                }
            }
        }
    }

    /**
     * Method executed just before the owning object is deleted, allowing tidying up of any
     * relation information.
     * @param sm StateManager for the owner
     */
    public void preDelete(StateManager sm)
    {
        ObjectManager om = sm.getObjectManager();

        // makes sure field is loaded
        int fieldNumber = mmd.getAbsoluteFieldNumber();
        try
        {
            om.getApiAdapter().isLoaded(sm, fieldNumber);
        }
        catch (JDOObjectNotFoundException onfe)
        {
            // Already deleted so just return
            return;
        }

        Object pc = sm.provideField(fieldNumber);
        if (pc == null)
        {
            // Null value so nothing to do
            return;
        }

        // Check if the field has a FK defined
        ClassLoaderResolver clr = om.getClassLoaderResolver();
        MappedStoreManager storeMgr = (MappedStoreManager)om.getStoreManager();
        AbstractMemberMetaData[] relatedMmds = mmd.getRelatedMemberMetaData(clr);
        // TODO Cater for more than 1 related field
        boolean dependent = mmd.isDependent();
        boolean hasFK = false;
        if (!dependent)
        {
            // Not dependent, so check if the datastore has a FK and will take care of it for us
            if (mmd.getForeignKeyMetaData() != null)
            {
                hasFK = true;
            }
            if (relatedMmds != null && relatedMmds[0].getForeignKeyMetaData() != null)
            {
                hasFK = true;
            }
            if (om.getOMFContext().getPersistenceConfiguration().getStringProperty("datanucleus.deletionPolicy").equals("JDO2"))
            {
                // JDO2 doesnt currently (2.0 spec) take note of foreign-key
                hasFK = false;
            }
        }

        // Basic rules for the following :-
        // 1. If it is dependent then we delete it (maybe after nulling).
        // 2. If it is not dependent and they have defined no FK then null it, else delete it
        // 3. If it is not dependent and they have a FK, let the datastore handle the delete
        // There may be some corner cases that this code doesnt yet cater for
        int relationType = mmd.getRelationType(clr);
        if (pc != null)
        {
            if (relationType == Relation.ONE_TO_ONE_UNI ||
                (relationType == Relation.ONE_TO_ONE_BI && mmd.getMappedBy() == null))
            {
                // 1-1 with FK at this side (owner of the relation)
                if (dependent)
                {
                    boolean relatedObjectDeleted = om.getApiAdapter().isDeleted(pc);
                    if (isNullable() && !relatedObjectDeleted)
                    {
                        // Other object not yet deleted - just null out the FK
                        // TODO Not doing this would cause errors in 1-1 uni relations (e.g AttachDetachTest)
                        // TODO Log this since it affects the resultant objects
                        sm.replaceField(fieldNumber, null, true);
                        sm.getStoreManager().getPersistenceHandler().updateObject(sm, new int[]{fieldNumber});
                    }
                    if (!relatedObjectDeleted)
                    {
                        // Mark the other object for deletion since not yet tagged
                        om.deleteObjectInternal(pc);
                    }
                }
                else
                {
                    // We're deleting the FK at this side so shouldnt be an issue
                    AbstractMemberMetaData relatedMmd = mmd.getRelatedMemberMetaDataForObject(clr, sm.getObject(), pc);
                    if (relatedMmd != null)
                    {
                        StateManager otherSM = om.findStateManager(pc);
                        if (otherSM != null)
                        {
                            // Managed Relations : 1-1 bidir, so null out the object at the other
                            if (NucleusLogger.PERSISTENCE.isDebugEnabled())
                            {
                                NucleusLogger.PERSISTENCE.debug(LOCALISER.msg("041019",
                                    StringUtils.toJVMIDString(pc), relatedMmd.getFullFieldName(),
                                    StringUtils.toJVMIDString(sm.getObject())));
                            }
                            otherSM.replaceField(relatedMmd.getAbsoluteFieldNumber(), null, true);
                        }
                    }
                }
            }
            else if (relationType == Relation.ONE_TO_ONE_BI && mmd.getMappedBy() != null)
            {
                // 1-1 with FK at other side
                DatastoreClass relatedTable = storeMgr.getDatastoreClass(relatedMmds[0].getClassName(), clr);
                JavaTypeMapping relatedMapping = relatedTable.getMemberMapping(relatedMmds[0]);
                boolean isNullable = relatedMapping.isNullable();
                StateManager otherSM = om.findStateManager(pc);
                if (dependent)
                {
                    if (isNullable)
                    {
                        // Null out the FK in the datastore using a direct update (since we are deleting)
                        otherSM.replaceField(relatedMmds[0].getAbsoluteFieldNumber(), null, true);
                        otherSM.getStoreManager().getPersistenceHandler().updateObject(
                            otherSM, new int[]{relatedMmds[0].getAbsoluteFieldNumber()});
                    }
                    // Mark the other object for deletion
                    om.deleteObjectInternal(pc);
                }
                else if (!hasFK)
                {
                    if (isNullable())
                    {
                        // Null out the FK in the datastore using a direct update (since we are deleting)
                        otherSM.replaceField(relatedMmds[0].getAbsoluteFieldNumber(), null, true);
                        otherSM.getStoreManager().getPersistenceHandler().updateObject(
                            otherSM, new int[]{relatedMmds[0].getAbsoluteFieldNumber()});
                    }
                    else
                    {
                        // TODO Remove it
                    }
                }
                else
                {
                    // User has a FK defined (in MetaData) so let the datastore take care of it
                }
            }
            else if (relationType == Relation.MANY_TO_ONE_BI)
            {
                StateManager otherSM = om.findStateManager(pc);
                if (relatedMmds[0].getJoinMetaData() == null)
                {
                    // N-1 with FK at this side
                    if (otherSM.isDeleting())
                    {
                        // Other object is being deleted too but this side has the FK so just delete this object
                    }
                    else
                    {
                        // Other object is not being deleted so delete it if necessary
                        if (dependent)
                        {
                            if (isNullable())
                            {
                                // TODO Datastore nullability info can be unreliable so try to avoid this call
                                // Null out the FK in the datastore using a direct update (since we are deleting)
                                sm.replaceField(fieldNumber, null, true);
                                sm.getStoreManager().getPersistenceHandler().updateObject(sm, new int[]{fieldNumber});
                            }

                            if (om.getApiAdapter().isDeleted(pc))
                            {
                                // Object is already tagged for deletion but we're deleting the FK so leave til flush()
                            }
                            else
                            {
                                // Mark the other object for deletion
                                om.deleteObjectInternal(pc);
                            }
                        }
                        else
                        {
                            // Managed Relations : remove element from collection/map
                            if (relatedMmds[0].hasCollection())
                            {
                                // Only update the other side if not already being deleted
                                if (!om.getApiAdapter().isDeleted(otherSM.getObject()) && !otherSM.isDeleting())
                                {
                                    // Make sure the other object is updated in any caches
                                    om.markDirty(otherSM, false);
                                    Collection otherColl = (Collection)otherSM.provideField(relatedMmds[0].getAbsoluteFieldNumber());
                                    if (otherColl != null)
                                    {
                                        // TODO Localise this message
                                        NucleusLogger.PERSISTENCE.debug("ManagedRelationships : delete of object causes removal from collection at " + relatedMmds[0].getFullFieldName());
                                        otherColl.remove(sm.getObject());
                                    }
                                }
                            }
                            else if (relatedMmds[0].hasMap())
                            {
                                // TODO Cater for maps, but what is the key/value pair ?
                            }
                        }
                    }
                }
                else
                {
                    // N-1 with join table so no FK here so need to remove from Collection/Map first? (managed relations)
                    if (dependent)
                    {
                        // Mark the other object for deletion
                        om.deleteObjectInternal(pc);
                    }
                    else
                    {
                        // Managed Relations : remove element from collection/map
                        if (relatedMmds[0].hasCollection())
                        {
                            // Only update the other side if not already being deleted
                            if (!om.getApiAdapter().isDeleted(otherSM.getObject()) && !otherSM.isDeleting())
                            {
                                // Make sure the other object is updated in any caches
                                om.markDirty(otherSM, false);

                                // Make sure the other object has the collection loaded so does this change
                                otherSM.isLoaded((PersistenceCapable)otherSM.getObject(), relatedMmds[0].getAbsoluteFieldNumber());
                                Collection otherColl = (Collection)otherSM.provideField(relatedMmds[0].getAbsoluteFieldNumber());
                                if (otherColl != null)
                                {
                                    // TODO Localise this
                                    NucleusLogger.PERSISTENCE.debug("ManagedRelationships : delete of object causes removal from collection at " + relatedMmds[0].getFullFieldName());
                                    otherColl.remove(sm.getObject());
                                }
                            }
                        }
                        else if (relatedMmds[0].hasMap())
                        {
                            // TODO Cater for maps, but what is the key/value pair ?
                        }
                    }
                }
            }
            else
            {
                // No relation so what is this field ?
            }
        }
    }
}
